/*
    Hotspot Filter for VirtualDub -- Correct hotspot arising from a
	projector's bulb when shooting projected video. Uses a mask image
	obtained by projecting full white (no film in projector).
	Copyright (C) 2002 Donald A. Graft

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	The author can be contacted at:
	Donald Graft
	neuron2@attbi.com.
*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <crtdbg.h>
#include <math.h>
#include <stdlib.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
int StartProc(FilterActivation *fa, const FilterFunctions *ff);
int EndProc(FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);
BOOL LoadBitmapFromBMPFile(LPTSTR szFileName, HBITMAP *phBitmap, HPALETTE *phPalette);

///////////////////////////////////////////////////////////////////////////

typedef struct MyFilterData {
	IFilterPreview		*ifp;
	unsigned char		*bitmap, *bits;
	long				bwidth, bheight, bpitch;
	unsigned int		depth;
	char				filename[1024];
	int					mult, add;
} MyFilterData;

bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
    char save[255];
	int i, j;

	i = j = 0;
	while(mfd->filename[i])
	{
		if (mfd->filename[i] == '\\')
		{
			save[j++] = '\\';
			save[j++] = '\\';
			i++;
		}
		else
		{
			save[j++] = mfd->filename[i++];
		}
	}
	save[j] = 0;
	_snprintf(buf, buflen, "Config(%d, %d, \"%s\")",
		mfd->mult,
		mfd->add,
		save
);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->mult				= argv[0].asInt();
	mfd->add				= argv[1].asInt();
	strcpy(mfd->filename, *argv[2].asString());
}

ScriptFunctionDef func_defs[]={
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0iis" },
	{ NULL },
};

CScriptObject script_obj={
	NULL, func_defs
};

struct FilterDefinition filterDef_tutorial = {

	NULL, NULL, NULL,		// next, prev, module
	"hotspot (1.0)",	// name
	"Correct hotspot from projector bulb.",
							// desc
	"Donald Graft", 		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	InitProc,				// initProc
	NULL,					// deinitProc
	RunProc,				// runProc
	NULL,					// paramProc
	ConfigProc, 			// configProc
	StringProc, 			// stringProc
	StartProc,				// startProc
	EndProc,				// endProc

	&script_obj,			// script_obj
	FssProc,				// fssProc

};

int Load(MyFilterData *mfd)
{
	FILE *fp;
	char buf[6];
	DWORD size;
    char *p;
	char path[255];

	mfd->depth = 0;
	strcpy(path, mfd->filename);
	p = path + strlen(path);
	while (p >= path && *p != '.') p--;
	if (*p != '.')
	{
		MessageBox(NULL,
			"The mask file must be a BMP file (*.bmp)."
			,
			"Invalid mask file",
			MB_ICONEXCLAMATION);
		return 1;
	}
	if ((fp = fopen(path, "rb")) == NULL)
	{
		MessageBox(NULL,
			"Cannot open the specified mask file."
			,
			"Invalid mask file",
			MB_ICONEXCLAMATION);
		return 1;
	}
	fread(buf, 1, 2, fp);
	if (buf[0] != 'B' || buf[1] != 'M')
	{
		MessageBox(NULL,
			"Invalid BMP file (bad signature)."
			,
			"Invalid mask file",
			MB_ICONEXCLAMATION);
		return 1;
	}
	fread(&size, 1, 4, fp);
	fclose(fp);
	if (mfd->bitmap) free(mfd->bitmap);
	mfd->bitmap = (unsigned char *) malloc((size_t) size);
    if ((fp = fopen(path, "rb")) == NULL)
	{
		MessageBox(NULL,
			"Cannot open the specified mask file."
			,
			"Invalid mask file",
			MB_ICONEXCLAMATION);
		return 1;
	}
	fread(mfd->bitmap, 1, size, fp);
	fclose(fp);
	mfd->bits = mfd->bitmap + *((DWORD *)(mfd->bitmap + 10));
	mfd->bwidth = *((LONG *)(mfd->bitmap + 18));
	mfd->bheight = *((DWORD *)(mfd->bitmap + 22));
	mfd->bpitch = 3 * mfd->bwidth;
	if (mfd->bpitch & 0x3)
	{
		mfd->bpitch = (mfd->bpitch & 0xfffffffc) + 4;
	}
	mfd->depth = *((unsigned int *)(mfd->bitmap + 28));
	if (mfd->depth != 24)
	{
		MessageBox(NULL,
			"The bitmap must have a color depth of 24."
			,
			"Invalid mask file",
			MB_ICONEXCLAMATION);
		return 1;
	}
	return 0;
}

int StartProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
    int stat;

	/* Load bitmap and get pointer to bitmap bits. */
	stat = Load(mfd);
	if (mfd->bwidth != fa->src.w || mfd->bheight != fa->src.h) 
	{
		MessageBox(NULL,
			"The bitmap must have the same dimensions as the input frame."
			,
			"Invalid mask file",
			MB_ICONEXCLAMATION);
		stat = 1;
	}
	return stat;
}

int EndProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	free(mfd->bitmap);
	mfd->bitmap = NULL;
	return 0;
}

int RunProc(const FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	const PixDim	width = fa->src.w;
	const PixDim	height = fa->src.h;
	Pixel32 *src, *dst;
	int x, y;
	int r, g, b;
	unsigned char *bp;
	unsigned int m, mask;
	double mult, mscale, add, ascale;

	mscale = mfd->mult / 40000.0;
	ascale = mfd->add / 100000.0;
	src = fa->src.data;
	dst = fa->dst.data;
	for (y = 0; y < height; y++)
	{
		bp = mfd->bits + y * mfd->bpitch;
		for (x = 0; x < width; x++)
		{
			r = ((src[x] & 0xff0000) >> 16);
			g = ((src[x] & 0xff00) >> 8);
			b = (src[x] & 0xff);
			mask = *bp;
			bp += 3;
			mult = 1.0 + mask * mscale;
			add = mask * mask * ascale;
			r = (int) (r * mult + add);
			g = (int) (g * mult + add);
			b = (int) (b * mult + add);
			if ((m = max(r, max(g, b))) > 255)
			{
				r = (r * 255) / m; 
				g = (g * 255) / m; 
				b = (b * 255) / m; 
			}
			dst[x] = (r << 16) | (g << 8) | (b);		
		}
		src	= (Pixel *)((char *)src + fa->src.pitch);
		dst	= (Pixel *)((char *)dst + fa->dst.pitch);
	}
	return 0;
}

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_tutorial;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if (!(fd_tutorial = ff->addFilter(fm, &filterDef_tutorial, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_tutorial);
}

int InitProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
   	char prog[256];
	char path[256];
	LPTSTR ptr;

	mfd->mult = 0;
	mfd->add = 0;

	GetModuleFileName(NULL, prog, 255);
	GetFullPathName(prog, 255, path, &ptr);
	*ptr = 0;
	strcat(path, "plugins\\Mask.bmp");
	strcpy(mfd->filename, path);
	return 0;
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg) {
		case WM_INITDIALOG:
			SetWindowLong(hdlg, DWL_USER, lParam);
			mfd = (MyFilterData *)lParam;

			SetWindowText(GetDlgItem(hdlg, IDC_FILENAME), mfd->filename);

			SendMessage(GetDlgItem(hdlg, IDC_MSLIDER), TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 1000));
			SendMessage(GetDlgItem(hdlg, IDC_MSLIDER), TBM_SETPOS, (WPARAM)TRUE, mfd->mult);
			SetDlgItemInt(hdlg, IDC_MOFF, mfd->mult, FALSE);
			SendMessage(GetDlgItem(hdlg, IDC_ASLIDER), TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 1000));
			SendMessage(GetDlgItem(hdlg, IDC_ASLIDER), TBM_SETPOS, (WPARAM)TRUE, mfd->add);
			SetDlgItemInt(hdlg, IDC_AOFF, mfd->add, FALSE);

			mfd->ifp->InitButton(GetDlgItem(hdlg, IDPREVIEW));
			return TRUE;

		case WM_HSCROLL:
			if ((HWND) lParam == GetDlgItem(hdlg, IDC_MSLIDER))
			{
				int mult = SendMessage(GetDlgItem(hdlg, IDC_MSLIDER), TBM_GETPOS, 0, 0);
				if (mult != mfd->mult)
				{
					mfd->mult = mult;
					SetDlgItemInt(hdlg, IDC_MOFF, mfd->mult, FALSE);
					mfd->ifp->RedoFrame();
				}
			}
			else if ((HWND) lParam == GetDlgItem(hdlg, IDC_ASLIDER))
			{
				int add = SendMessage(GetDlgItem(hdlg, IDC_ASLIDER), TBM_GETPOS, 0, 0);
				if (add != mfd->add)
				{
					mfd->add = add;
					SetDlgItemInt(hdlg, IDC_AOFF, mfd->add, FALSE);
					mfd->ifp->RedoFrame();
				}
			}
			break;

		case WM_COMMAND:
			switch(LOWORD(wParam)) {

			case IDPREVIEW:
				mfd->ifp->Toggle(hdlg);
				break;

			case IDOK:
				EndDialog(hdlg, 0);
				return TRUE;

 			case IDHELP:
				{
				char prog[256];
				char path[256];
				LPTSTR ptr;

				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\Hotspot.html");
				ShellExecute(hdlg, "open", path, NULL, NULL, SW_SHOWNORMAL);
				return TRUE;
				}

			case IDC_BROWSE:
				{
				OPENFILENAME ofn;
				mfd->filename[0] = NULL;
				ofn.lStructSize = sizeof(OPENFILENAME);
				ofn.hwndOwner = hdlg;
				ofn.hInstance = NULL;
				ofn.lpTemplateName = NULL;
				ofn.lpstrFilter = "All Files\0*.*\0\0";
				ofn.lpstrCustomFilter = NULL;
				ofn.nMaxCustFilter = 0;
				ofn.nFilterIndex = 1;
				ofn.lpstrFile = mfd->filename;
				ofn.nMaxFile = 1024;
				ofn.lpstrFileTitle = NULL;
				ofn.nMaxFileTitle = 0;
				ofn.lpstrInitialDir = NULL;
				ofn.lpstrTitle = "Choose Logo Bitmap File";
				ofn.Flags = OFN_CREATEPROMPT;
				ofn.nFileOffset = 0;
				ofn.nFileExtension = 0;
				ofn.lpstrDefExt = NULL;
				ofn.lCustData = 0;
				ofn.lpfnHook = NULL;
				GetOpenFileName(&ofn);
				if (mfd->filename[0] != 0)
				{
					SetWindowText(GetDlgItem(hdlg, IDC_FILENAME), mfd->filename);
					Load(mfd);
					mfd->ifp->UndoSystem();
					mfd->ifp->RedoSystem();
				}
				break;
				}

			case IDC_MPLUS:
				if (mfd->mult < 1000)
				{
					mfd->mult += 1;
					SetDlgItemInt(hdlg, IDC_MOFF, mfd->mult, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_MSLIDER), TBM_SETPOS, (WPARAM)TRUE, mfd->mult);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_MMINUS:
				if (mfd->mult > 0)
				{
					mfd->mult -= 1;
					SetDlgItemInt(hdlg, IDC_MOFF, mfd->mult, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_MSLIDER), TBM_SETPOS, (WPARAM)TRUE, mfd->mult);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_APLUS:
				if (mfd->add < 1000)
				{
					mfd->add += 1;
					SetDlgItemInt(hdlg, IDC_AOFF, mfd->add, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_ASLIDER), TBM_SETPOS, (WPARAM)TRUE, mfd->add);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_AMINUS:
				if (mfd->add > 0)
				{
					mfd->add -= 1;
					SetDlgItemInt(hdlg, IDC_AOFF, mfd->add, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_ASLIDER), TBM_SETPOS, (WPARAM)TRUE, mfd->add);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDCANCEL:
				EndDialog(hdlg, 1);
				return TRUE;
			}
			break;
	}

	return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	MyFilterData mfd_old = *mfd;
	int ret;

	mfd->ifp = fa->ifp;

	if (DialogBoxParam(fa->filter->module->hInstModule, MAKEINTRESOURCE(IDD_FILTER),
		hwnd, ConfigDlgProc, (LPARAM)mfd))
	{
		*mfd = mfd_old;
		ret = TRUE;
	}
    else
	{
		ret = FALSE;
	}
	return(ret);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
}
