/*
    Telecide Filter for VirtualDub.
    Copyright (C) 1999-2000 Donald A. Graft

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	The author can be contacted at:
	Donald Graft
	neuron2@home.com.
*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <crtdbg.h>
#include <math.h>
#include <stdlib.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
int StartProc(FilterActivation *fa, const FilterFunctions *ff);
int EndProc(FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

typedef struct MyFilterData {
	IFilterPreview		*ifp;
	Pixel32				*pFrame;
	Pixel32 			*ppFrame;
	Pixel32				*pppFrame;
	Pixel32				*p, *pp, *ppp;
	int					swap;
	int					fast;
	int					debug;
	unsigned int		threshold;
	int					first, correct;
} MyFilterData;

bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	_snprintf(buf, buflen, "Config(%d, %d, %d, %d)",
		      mfd->swap, mfd->fast, mfd->debug, mfd->threshold);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->swap = argv[0].asInt();
	mfd->fast = argv[1].asInt();
	mfd->debug = argv[2].asInt();
	mfd->threshold = argv[3].asInt();
}

ScriptFunctionDef func_defs[]={
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0iiii" },
	{ NULL },
};

CScriptObject script_obj={
	NULL, func_defs
};

struct FilterDefinition filterDef_tutorial = {

	NULL, NULL, NULL,		// next, prev, module
	"telecide (1.3.0)",		// name
	"Kill telecining",
							// desc
	"Donald Graft", 		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	InitProc,				// initProc
	NULL,					// deinitProc
	RunProc,				// runProc
	NULL,					// paramProc
	ConfigProc, 			// configProc
	StringProc, 			// stringProc
	StartProc,				// startProc
	EndProc,				// endProc

	&script_obj,			// script_obj
	FssProc,				// fssProc

};

int StartProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->pFrame = (Pixel32 *) new int[fa->src.w*fa->src.h];
	mfd->ppFrame = (Pixel32 *) new int[fa->src.w*fa->src.h];
	mfd->pppFrame = (Pixel32 *) new int[fa->src.w*fa->src.h];
	mfd->p = mfd->pFrame;
	mfd->pp = mfd->ppFrame;
	mfd->ppp = mfd->pppFrame;
	mfd->first = 1;
	mfd->correct = 0;
	return 0;
}

int EndProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	delete[] mfd->pFrame;
	mfd->pFrame = NULL;
	delete[] mfd->ppFrame;
	mfd->ppFrame = NULL;
	delete[] mfd->pppFrame;
	mfd->pppFrame = NULL;
	return 0;
}

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	const int	width = fa->src.w;
	const int	height = fa->src.h;
	Pixel32 *src, *dst, *tFrame;
	unsigned long ddd, dd, d, DDD, DD, D, diff;
	Pixel32 *prev, *next;
	long tmp, luma, lumap, luman;
	unsigned long r, g, b, rp, gp, bp, rn, gn, bn;
	int x, y;
	int frame = fa->pfsi->lCurrentSourceFrame;
	int desperate;
    char chosen[4];
	int xinc, ydiv;

#if 0
{
	char buff[80];
	sprintf(buff, "frame %d\n", fa->pfsi->lCurrentSourceFrame);
	OutputDebugString(buff);
}
#endif

	if (mfd->fast)
	{
		xinc = 8;
		ydiv = 4;
	}
	else
	{
		xinc = 1;
		ydiv = 2;
	}

	/* Cycle the pointers. */
	tFrame = mfd->ppp;
	mfd->ppp = mfd->pp;
	mfd->pp = mfd->p;
	mfd->p = tFrame;

	/* Store the arriving frame. */
	src = fa->src.data;
	dst = mfd->p;
	for (y = 0; y < height; y++)
	{
		memcpy(dst, src, 4 * width);
		src	= (Pixel *)((char *)src + fa->src.pitch);
		dst	+= width;
	}

    /* This is a big kludge to try to make things work right
	   if the user loads a clip and then does a step forward
	   without doing a rewind first. */
	if (frame == 0) mfd->correct = 0;
	if (mfd->first)
	{
		mfd->first = 0;
		if (frame == 1)
		{
			mfd->correct = 1;
		}
	}
	frame -= mfd->correct;
	/* End big kludge. */
	
	/* Synthesize the output frame. */
	if (frame > 1)
	{
		/* Calculate up to six differences: between the top field
		   of the middle stored frame and the bottom fields of
		   all the stored frames (3) and (if needed) between the bottom
		   field of the middle stored frame and the top fields of
		   all the stored frames (3). We'll pick the best match from
		   these possibilities. */

		/* Use a combing detection algorithm on each of the six
		   composite frames as described above. */ 
		desperate = 0;
		src = mfd->pp + 2 * width;
		prev = mfd->p + width;
		next = mfd->p + 3 * width;
		d = 0;

		for (y = 0; y < height/ydiv - 2; y++)
		{
			for (x = 0; x < width; x+=xinc)
			{
				r = (src[x] & 0xff0000) >> 16;
				rp = (prev[x] & 0xff0000) >> 16;
				rn = (next[x] & 0xff0000) >> 16;
				b = (src[x] & 0xff);
				bp = (prev[x] & 0xff);
				bn = (next[x] & 0xff);
				g = (src[x] & 0xff00)   >> 8;
				gp = (prev[x] & 0xff00)   >> 8;
				gn = (next[x] & 0xff00)   >> 8;
				luma = (r + b + g) / 3;
				lumap = (rp + bp + gp) / 3;
				luman = (rn + bn + gn) / 3;
				tmp = (lumap - luma) * (luman - luma);
				if (tmp > 0) d += tmp;
			}
			src += ydiv * width;
			prev += ydiv * width;
			next += ydiv * width;
		}
		diff = d;
		strcpy(chosen, "d");

		src = mfd->pp + 2 * width;
		prev = mfd->pp + width;
		next = mfd->pp + 3 * width;
		dd = 0;
		for (y = 0; y < height/ydiv - 2; y++)
		{
			for (x = 0; x < width; x+=xinc)
			{
				r = (src[x] & 0xff0000) >> 16;
				rp = (prev[x] & 0xff0000) >> 16;
				rn = (next[x] & 0xff0000) >> 16;
				b = (src[x] & 0xff);
				bp = (prev[x] & 0xff);
				bn = (next[x] & 0xff);
				g = (src[x] & 0xff00)   >> 8;
				gp = (prev[x] & 0xff00)   >> 8;
				gn = (next[x] & 0xff00)   >> 8;
				luma = (r + b + g) / 3;
				lumap = (rp + bp + gp) / 3;
				luman = (rn + bn + gn) / 3;
				tmp = (lumap - luma) * (luman - luma);
				if (tmp > 0) dd += tmp;
			}
			src += ydiv * width;
			prev += ydiv * width;
			next += ydiv * width;
		}
		if (dd < diff)
		{
			diff = dd;
			strcpy(chosen, "dd");
		}

		src = mfd->pp + 2 * width;
		prev = mfd->ppp + width;
		next = mfd->ppp + 3 * width;
		ddd = 0;
		for (y = 0; y < height/ydiv - 2; y++)
		{
			for (x = 0; x < width; x+=xinc)
			{
				r = (src[x] & 0xff0000) >> 16;
				rp = (prev[x] & 0xff0000) >> 16;
				rn = (next[x] & 0xff0000) >> 16;
				b = (src[x] & 0xff);
				bp = (prev[x] & 0xff);
				bn = (next[x] & 0xff);
				g = (src[x] & 0xff00)   >> 8;
				gp = (prev[x] & 0xff00)   >> 8;
				gn = (next[x] & 0xff00)   >> 8;
				luma = (r + b + g) / 3;
				lumap = (rp + bp + gp) / 3;
				luman = (rn + bn + gn) / 3;
				tmp = (lumap - luma) * (luman - luma);
				if (tmp > 0) ddd += tmp;
			}
			src += ydiv * width;
			prev += ydiv * width;
			next += ydiv * width;
		}
		if (ddd < diff)
		{
			diff = ddd;
			strcpy(chosen, "ddd");
		}

		/* If we haven't yet found a good match, then keep trying
		   but center on the bottom field of the middle frame. */
		if (mfd->threshold && (diff / (width * height)) > mfd->threshold)
		{
			desperate = 1;
			src = mfd->pp + width;
			prev = mfd->p;
			next = mfd->p + 2 * width;
			D = 0;
			for (y = 0; y < height/ydiv - 2; y++)
			{
				for (x = 0; x < width; x+=xinc)
				{
					r = (src[x] & 0xff0000) >> 16;
					rp = (prev[x] & 0xff0000) >> 16;
					rn = (next[x] & 0xff0000) >> 16;
					b = (src[x] & 0xff);
					bp = (prev[x] & 0xff);
					bn = (next[x] & 0xff);
					g = (src[x] & 0xff00)   >> 8;
					gp = (prev[x] & 0xff00)   >> 8;
					gn = (next[x] & 0xff00)   >> 8;
					luma = (r + b + g) / 3;
					lumap = (rp + bp + gp) / 3;
					luman = (rn + bn + gn) / 3;
					tmp = (lumap - luma) * (luman - luma);
					if (tmp > 0) D += tmp;
				}
				src += ydiv * width;
				prev += ydiv * width;
				next += ydiv * width;
			}
			if (D < diff)
			{
				diff = D;
				strcpy(chosen, "D");
			}

			src = mfd->pp + width;
			prev = mfd->pp;
			next = mfd->pp + 2 * width;
			DD = 0;
			for (y = 0; y < height/ydiv - 2; y++)
			{
				for (x = 0; x < width; x+=xinc)
				{
					r = (src[x] & 0xff0000) >> 16;
					rp = (prev[x] & 0xff0000) >> 16;
					rn = (next[x] & 0xff0000) >> 16;
					b = (src[x] & 0xff);
					bp = (prev[x] & 0xff);
					bn = (next[x] & 0xff);
					g = (src[x] & 0xff00)   >> 8;
					gp = (prev[x] & 0xff00)   >> 8;
					gn = (next[x] & 0xff00)   >> 8;
					luma = (r + b + g) / 3;
					lumap = (rp + bp + gp) / 3;
					luman = (rn + bn + gn) / 3;
					tmp = (lumap - luma) * (luman - luma);
					if (tmp > 0) DD += tmp;
				}
				src += ydiv * width;
				prev += ydiv * width;
				next += ydiv * width;
			}
			if (DD < diff)
			{
				diff = DD;
				strcpy(chosen, "DD");
			}

			src = mfd->pp + width;
			prev = mfd->ppp;
			next = mfd->ppp + 2 * width;
			DDD = 0;
			for (y = 0; y < height/ydiv - 2; y++)
			{
				for (x = 0; x < width; x+=xinc)
				{
					r = (src[x] & 0xff0000) >> 16;
					rp = (prev[x] & 0xff0000) >> 16;
					rn = (next[x] & 0xff0000) >> 16;
					b = (src[x] & 0xff);
					bp = (prev[x] & 0xff);
					bn = (next[x] & 0xff);
					g = (src[x] & 0xff00)   >> 8;
					gp = (prev[x] & 0xff00)   >> 8;
					gn = (next[x] & 0xff00)   >> 8;
					luma = (r + b + g) / 3;
					lumap = (rp + bp + gp) / 3;
					luman = (rn + bn + gn) / 3;
					tmp = (lumap - luma) * (luman - luma);
					if (tmp > 0) DDD += tmp;
				}
				src += ydiv * width;
				prev += ydiv * width;
				next += ydiv * width;
			}
			if (DDD < diff)
			{
				diff = DDD;
				strcpy(chosen, "DDD");
			}
		}

		/* Set up the pointers in preparation to output final frame. */
		if (!strcmp(chosen, "d"))
		{
			src = mfd->p + width;
			dst = fa->dst.data;
			dst	= (Pixel *)((char *)dst + fa->dst.pitch);
		}
		else if (!strcmp(chosen, "dd"))
		{
			src = mfd->pp + width;
			dst = fa->dst.data;
			dst	= (Pixel *)((char *)dst + fa->dst.pitch);
		}
		else if (!strcmp(chosen, "ddd"))
		{
			src = mfd->ppp + width;
			dst = fa->dst.data;
			dst	= (Pixel *)((char *)dst + fa->dst.pitch);
		}
		else if (desperate)
		{
			if (!strcmp(chosen, "D"))
			{
				src = mfd->p;
				dst = fa->dst.data;
			}
			else if (!strcmp(chosen, "DD"))
			{
				src = mfd->pp;
				dst = fa->dst.data;
			}
			else if (!strcmp(chosen, "DDD"))
			{
				src = mfd->ppp;
				dst = fa->dst.data;
			}
		}

#if 1
		if (mfd->debug)
		{
			char buff[80];
			if (desperate)
				sprintf(buff, "%u %u %u %u %u %u: %u [%s]\n",
					    ddd / (width * height), dd / (width * height), d / (width * height),
						DDD / (width * height), DD / (width * height), D / (width * height),
						diff / (width * height), chosen);
			else
				sprintf(buff, "%u %u %u: %u [%s]\n",
					    ddd / (width * height), dd / (width * height), d / (width * height),
						diff / (width * height), chosen);
			OutputDebugString(buff);
		}
#endif

		/* First output the field selected from the set of three stored
		   frames. */
		if (mfd->swap)
		{
			if (dst == fa->dst.data)
				dst	= (Pixel *)((char *)dst + fa->dst.pitch);
			else
				dst = fa->dst.data;
		}
		for (y = 0; y < height/2; y++)
		{
			memcpy(dst, src, 4 * width);
			src += 2 * width;
			dst	= (Pixel *)((char *)dst + fa->dst.pitch);
			dst	= (Pixel *)((char *)dst + fa->dst.pitch);
		}

		/* Now output the appropriate field of the middle stored frame. */
		if (!strcmp(chosen, "D") || !strcmp(chosen, "DD") || !strcmp(chosen, "DDD"))
		{
			src = mfd->pp + width;
			dst = fa->dst.data;
			dst	= (Pixel *)((char *)dst + fa->dst.pitch);
		}
		else
		{
			src = mfd->pp;
			dst = fa->dst.data;
		}
		if (mfd->swap)
		{
			if (dst == fa->dst.data)
				dst	= (Pixel *)((char *)dst + fa->dst.pitch);
			else
				dst = fa->dst.data;
		}
		for (y = 0; y < height/2; y++)
		{
			memcpy(dst, src, 4 * width);
			src	+= 2 * width;
			dst	= (Pixel *)((char *)dst + fa->dst.pitch);
			dst	= (Pixel *)((char *)dst + fa->dst.pitch);
		}
	}
	else
	{
		/* We don't have enough frames stored to work yet, so
		   just throw away a field. */
		Pixel32 *pp, *p, *n, *nn;
		int rpp, gpp, bpp, rp, gp, bp, rn, gn, bn, rnn, gnn, bnn, R, G, B;

		/* Pick up a pointer to the first frame received. */
		if (frame == 0)
			src = mfd->p;
		else
			src = mfd->pp;
		/* Copy through the even lines and duplicate the odd lines for
		   which there won't be enough surrounding lines to do cubic interpolation. */
		dst = fa->dst.data;
		for (y = 0; y < height/2; y++)
		{
			memcpy(dst, src, 4 * width);
			if (y == 0 || y == height/2 - 1 || y == height/2 - 2)
			{
				dst	= (Pixel *)((char *)dst + fa->dst.pitch);
				memcpy(dst, src, 4 * width);
				dst	= (Pixel *)((char *)dst + fa->dst.pitch);
			}
			else
				dst	= (Pixel *)((char *)dst + 2 * fa->dst.pitch);
			src += 2 * width;
		}
		/* Generate the odd lines by cubic interpolation of the even lines. */
		if (frame == 0)
			src = mfd->p;
		else
			src = mfd->pp;
		pp = src;
		p = src + 2 * width;
		n = src + 4 * width;
		nn = src + 6 * width;
		dst = (Pixel *)((char *)fa->dst.data + 3 * fa->dst.pitch);
		for (y = 0; y < height/2 - 3; y++)
		{
			for (x = 0; x < width; x++)
			{
				rpp = (pp[x] >> 16) & 0xff;
				rp = (p[x] >>16) & 0xff;
				rn = (n[x] >> 16) & 0xff;
				rnn = (nn[x] >> 16) & 0xff;
				gpp = (pp[x] >> 8) & 0xff;
				gp = (p[x] >> 8) & 0xff;
				gn = (n[x] >> 8) & 0xff;
				gnn = (nn[x] >> 8) & 0xff;
				bpp = (pp[x]) & 0xff;
				bp = (p[x]) & 0xff;
				bn = (n[x]) & 0xff;
				bnn = (nn[x]) & 0xff;
				R = (5 * (rp + rn) - (rpp + rnn)) >> 3;
				if (R > 255) R = 255;
				else if (R < 0) R = 0;
				G = (5 * (gp + gn) - (gpp + gnn)) >> 3;
				if (G > 255) G = 255;
				else if (G < 0) G = 0;
				B = (5 * (bp + bn) - (bpp + bnn)) >> 3;
				if (B > 255) B = 255;
				else if (B < 0) B = 0;
				dst[x] = (R << 16) | (G << 8) | B;
			}
			dst	= (Pixel *)((char *)dst + 2 * fa->dst.pitch);
			pp += 2 * width;
			p += 2 * width;
			n += 2 * width;
			nn += 2 * width;
		}
		if (mfd->debug)
		{
			char buff[80];
			sprintf(buff, "interpolating\n");
			OutputDebugString(buff);
		}
	}

	return 0;
}

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_tutorial;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if (!(fd_tutorial = ff->addFilter(fm, &filterDef_tutorial, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_tutorial);
}

int InitProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->swap = 0;
	mfd->fast = 0;
	mfd->debug = 0;
	mfd->threshold = 0;
	return 0;
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg) {
		case WM_INITDIALOG:
			SetWindowLong(hdlg, DWL_USER, lParam);
			mfd = (MyFilterData *)lParam;
			CheckDlgButton(hdlg, IDC_FIELD_SWAP, mfd->swap ? BST_CHECKED : BST_UNCHECKED);
			CheckDlgButton(hdlg, IDC_FAST, mfd->fast ? BST_CHECKED : BST_UNCHECKED);
			CheckDlgButton(hdlg, IDC_DEBUG, mfd->debug ? BST_CHECKED : BST_UNCHECKED);
			SetDlgItemInt(hdlg, IDC_THRESHOLD, mfd->threshold, FALSE);
			return TRUE;

		case WM_COMMAND:
			switch(LOWORD(wParam)) {
			case IDOK:
				mfd->swap = !!IsDlgButtonChecked(hdlg, IDC_FIELD_SWAP);
				mfd->fast = !!IsDlgButtonChecked(hdlg, IDC_FAST);
				mfd->debug = !!IsDlgButtonChecked(hdlg, IDC_DEBUG);
				EndDialog(hdlg, 0);
				return TRUE;
 			case IDHELP:
				{
				char prog[256];
				char path[256];
				LPTSTR ptr;

				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\Telecide.html");
				ShellExecute(hdlg, "open", path, NULL, NULL, SW_SHOWNORMAL);
				return TRUE;
				}
			case IDCANCEL:
				EndDialog(hdlg, 1);
				return TRUE;
			case IDC_THRESHOLD:
				mfd->threshold = GetDlgItemInt(hdlg, IDC_THRESHOLD, 0, FALSE);
				break;
			}
			break;
	}

	return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	MyFilterData mfd_old = *mfd;
	int ret;

	mfd->ifp = fa->ifp;

	if (DialogBoxParam(fa->filter->module->hInstModule, MAKEINTRESOURCE(IDD_FILTER),
		hwnd, ConfigDlgProc, (LPARAM)mfd))
	{
		*mfd = mfd_old;
		ret = TRUE;
	}
    else
	{
		ret = FALSE;
	}
	return(ret);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
}

