/*
    Pixellate Filter for VirtualDub -- Create large single coloured
      blocks based on the average of the input pixels.
      Copyright (C) 2001 Tom Ford

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    The author can be contacted at:
      Tom Ford
      flend@compsoc.net

    This work includes code from:
      Colorize Filter for VirtualDub -- Change a video to a monochrome video
        with selectable base color. Copyright (C) 1999-2000 Donald A. Graft

      By 
        Donald Graft
        neuron2@home.com.
*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <crtdbg.h>
#include <math.h>
#include <stdlib.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

#define BLOCK_SIZE_MAX 100
#define HUE_EPSILON 0.05

int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);
void RGBtoHSL(Pixel32 src, double *H, double *S, double *L);
void HLStoRGB(double H, double L, double S, int *r, int *g, int *b);

typedef struct FilterDataStruct {
  IFilterPreview *ifp;
  int block_size;
} FilterData;


bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
  FilterData *filter_data = (FilterData *)fa->filter_data;

  _snprintf(buf, buflen, "Config(%d)", filter_data->block_size);
  return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {

  FilterActivation *fa = (FilterActivation *)lpVoid;
  FilterData *filter_data = (FilterData *)fa->filter_data;

  filter_data->block_size = argv[0].asInt();
}

ScriptFunctionDef func_defs[]={
  { (ScriptFunctionPtr)ScriptConfig, "Config", "0i" },
  { NULL },
};

CScriptObject script_obj={
  NULL, func_defs
};

struct FilterDefinition filterdef = {

  NULL, NULL, NULL,     // next, prev, module
  "pixellate",          // name
  "A simple pixellation effect.",  // desc
  "Tom Ford",           // maker
  NULL,                 // private_data
  sizeof(FilterData),   // inst_data_size

  InitProc,             // initProc
  NULL,                 // deinitProc
  RunProc,              // runProc
  NULL,                 // paramProc
  ConfigProc,           // configProc
  StringProc,           // stringProc
  NULL,                 // startProc
  NULL,                 // endProc

  &script_obj,          // script_obj
  FssProc,              // fssProc
};

int RunProc(const FilterActivation *fa, const FilterFunctions *ff) {

  FilterData *filter_data = (FilterData *)fa->filter_data;
  FilterStateInfo *info = fa->pfsi;
  const PixDim width = fa->src.w;
  const PixDim height = fa->src.h;
  Pixel32 *src, *dst;
  int x, y;
  int r, g, b;
  double H, S, L;
  int i,j,k;

  src = fa->src.data;
  dst = fa->dst.data;

  int start_y = -1 * (filter_data->block_size - ((height % filter_data->block_size) >> 1));
  int start_x = -1 * ((width % filter_data->block_size) >> 1);

  double *median_hue = (double *)malloc((filter_data->block_size * filter_data->block_size) * sizeof(double));
  double *median_sat = (double *)malloc((filter_data->block_size * filter_data->block_size) * sizeof(double));
  int *median_hue_count = (int *)malloc((filter_data->block_size * filter_data->block_size) * sizeof(int));

  for (y = start_y; y < height; y+=filter_data->block_size) {
    for (x = start_x; x < width; x+=filter_data->block_size) {

      //Calc mean lightness, saturation and median hue and set block to appropriate rgb 
        
      double mean_l, median_h, median_s;
      int next_hue, clip_modifier;
      bool found_hue;

      mean_l = median_h = median_s = 0;
      next_hue = clip_modifier = 0;

      int start_i = 0; int end_i = filter_data->block_size;
      int start_j = 0; int end_j = filter_data->block_size;

      if(x < 0)
	start_i = -1 * x;

      if(y < 0)
	start_j = -1 * y;

      if(x + filter_data->block_size > width)
	end_i = width - x;

      if(y + filter_data->block_size > height)
	end_j = height - y;

      //account for missed pixels
      clip_modifier = (end_i - start_i) * (end_j - start_j);

      for(i = start_i; i < end_i; i++) {
        for(j = start_j; j < end_j; j++) {

          Pixel32 *pixel_to_read;
          pixel_to_read = src + x + i;
          pixel_to_read = (Pixel32 *)((char *)pixel_to_read + (fa->src.pitch * (j+start_y)));

          RGBtoHSL(*pixel_to_read, &H, &S, &L);

          //mean

          mean_l += L;
  
          //store hues

          found_hue = false;

          for(k = 0; k < next_hue; k++) {
            if(fabs(median_hue[k] - H) < HUE_EPSILON && fabs(median_sat[k] - S) < HUE_EPSILON) {
              median_hue_count[k]++;
              found_hue = true;
              break;
            }
          }
              
          if(!found_hue) {
            median_hue[next_hue] = H;
            median_sat[next_hue] = S;
            median_hue_count[next_hue] = 1;
            next_hue++;
          }
        }
      }

      //mean lightness and saturation

      mean_l /= clip_modifier;

      //find median hue
          
      int highest_hue_count = 0;

      for(i = 0; i < next_hue; i++) {
        if(median_hue_count[i] > highest_hue_count) {
          median_h = median_hue[i];
          median_s = median_sat[i];
          highest_hue_count = median_hue_count[i];
        }
      }

      //compensate for rounding errors

      if(mean_l > 1.0)
        mean_l = 1.0;

      if(median_h > 1.0)
        median_h = 1.0;

      if(median_s > 1.0)
        median_s = 1.0;

      //write output block
          
      for(i = 0; i < filter_data->block_size; i++) {
        for(j = 0; j < filter_data->block_size; j++) {

          if(y+i >= height || y+i < 0 || x+j >= width || x+j < 0) {
            continue;
          }

          HLStoRGB(median_h, mean_l, median_s, &r, &g, &b);

          Pixel32 *pixel_to_write;
          pixel_to_write = dst + x + j;
          pixel_to_write = (Pixel32 *)((char *)pixel_to_write + (fa->dst.pitch * (i+start_y)));

          *(pixel_to_write) = (r << 16) | (g << 8) | (b);                   
        }
      }
    }

    src = (Pixel32 *)((char *)src + (filter_data->block_size * fa->src.pitch));
    dst = (Pixel32 *)((char *)dst + (filter_data->block_size * fa->dst.pitch));   
  }

  free(median_hue);
  free(median_sat);
  free(median_hue_count);

  return 0;
}

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_pixellate;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
  if (!(fd_pixellate = ff->addFilter(fm, &filterdef, sizeof(FilterDefinition))))
    return 1;

  vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
  vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

  return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
  ff->removeFilter(fd_pixellate);
}

int InitProc(FilterActivation *fa, const FilterFunctions *ff)
{
  FilterData *filter_data = (FilterData *)fa->filter_data;

  filter_data->block_size = 5;

  return 0;
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
  FilterData *filter_data = (FilterData *)GetWindowLong(hdlg, DWL_USER);

  switch(msg) {
    HWND hWnd;

  case WM_PAINT:
    return 0;

  case WM_INITDIALOG:
    SetWindowLong(hdlg, DWL_USER, lParam);
    filter_data = (FilterData *)lParam;

    hWnd = GetDlgItem(hdlg, IDC_BLOCKSIZE);
    SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(1, BLOCK_SIZE_MAX));
    SendMessage(hWnd, TBM_SETTICFREQ, 10 , 0);
    SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, filter_data->block_size);
    SetDlgItemInt(hdlg, IDC_EBLOCKSIZE, filter_data->block_size, FALSE);

    filter_data->ifp->InitButton(GetDlgItem(hdlg, IDPREVIEW));
    return TRUE;

  case WM_HSCROLL:
    {
      int block;
      HWND hwnd = GetDlgItem(hdlg, IDC_BLOCKSIZE);
      block = SendMessage(hwnd, TBM_GETPOS, 0, 0);
      SetDlgItemInt(hdlg, IDC_EBLOCKSIZE, block, FALSE);

      if (block != filter_data->block_size)
        filter_data->block_size = block;
      
      filter_data->ifp->RedoFrame();
      break;
    }

  case WM_COMMAND:
    switch(LOWORD(wParam)) {
    case IDPREVIEW:
      filter_data->ifp->Toggle(hdlg);
      break;
    case IDOK:
      EndDialog(hdlg, 0);
      return TRUE;
    case IDHELP:
      {
        MessageBox(NULL, "This filter should be pretty straightforward.", "Pixellate 'help'.", MB_OK | MB_ICONINFORMATION);
        /*char prog[256];
          char path[256];
          LPTSTR ptr;

          GetModuleFileName(NULL, prog, 255);
          GetFullPathName(prog, 255, path, &ptr);
          *ptr = 0;
          strcat(path, "plugins\\Colorize.html");
          ShellExecute(hdlg, "open", path, NULL, NULL, SW_SHOWNORMAL);*/
        return TRUE;
      }
    case IDCANCEL:
      EndDialog(hdlg, 1);
      return TRUE;

    case IDC_BLOCKSIZEPLUS:
   
      if (filter_data->block_size < BLOCK_SIZE_MAX) {

        filter_data->block_size++;
        SetDlgItemInt(hdlg, IDC_EBLOCKSIZE, filter_data->block_size, FALSE);
        SendMessage(GetDlgItem(hdlg, IDC_BLOCKSIZE), TBM_SETPOS, (WPARAM)TRUE, filter_data->block_size);

        filter_data->ifp->RedoFrame();
      }
      break;
    case IDC_BLOCKSIZEMINUS:

      if (filter_data->block_size > 1) {

        filter_data->block_size--;
        SetDlgItemInt(hdlg, IDC_EBLOCKSIZE, filter_data->block_size, FALSE);
        SendMessage(GetDlgItem(hdlg, IDC_BLOCKSIZE), TBM_SETPOS, (WPARAM)TRUE, filter_data->block_size);

        filter_data->ifp->RedoFrame();
      }
      break;
    }
    break;
  }

  return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
  FilterData *filter_data = (FilterData *)fa->filter_data;
  FilterData filter_data_old = *filter_data;
  int ret;

  filter_data->ifp = fa->ifp;

  ret = DialogBoxParam(fa->filter->module->hInstModule, MAKEINTRESOURCE(IDD_FILTER),
                       hwnd, ConfigDlgProc, (LPARAM) filter_data);
  if (ret) {
    *filter_data = filter_data_old;
    ret = TRUE;
  }
  else {
    ret = FALSE;
  }
  return(ret);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str)
{
  FilterData *filter_data = (FilterData *)fa->filter_data;

  sprintf(str, " (block size %d)", filter_data->block_size);
}

void RGBtoHSL(Pixel32 src, double *H, double *S, double *L)
{
  double delta;
  double r = (double)((src & 0xff0000) >> 16)/255;
  double g = (double)((src & 0xff00) >> 8)/255;
  double b = (double)(src & 0xff)/255;
  double cmax = max(r,max(g,b));
  double cmin = min(r,min(g,b));

  *L=(cmax+cmin)/2.0;
  if(cmax==cmin) {
    *S = 0;
    *H = 0; // it's really undefined
  } else {
    if(*L < 0.5) 
      *S = (cmax-cmin)/(cmax+cmin);
    else
      *S = (cmax-cmin)/(2.0-cmax-cmin);
    delta = cmax - cmin;
    if(r==cmax)
      *H = (g-b)/delta;
    else if(g==cmax)
      *H = 2.0 +(b-r)/delta;
    else
      *H=4.0+(r-g)/delta;
    *H /= 6.0;
    if(*H < 0.0)
      *H += 1;
  }
}

int HuetoRGB(double m1, double m2, double h)
{
  if( h < 0 ) h += 1.0;
  if( h > 1 ) h -= 1.0;
  if( 6.0*h < 1 )
    return ((int) (255 * (m1+(m2-m1)*h*6.0)));
  if( 2.0*h < 1 )
    return ((int) (255 * m2));
  if( 3.0*h < 2.0 )
    return ((int) (255 * (m1+(m2-m1)*((2.0/3.0)-h)*6.0)));
  return ((int) (255 * m1));
}

void HLStoRGB(double H, double L, double S, int *r, int *g, int *b)
{
  double m1, m2;

  if(S==0) {
    *r = *g = *b = (int) (255 * L);
  } else {
    if(L <=0.5)
      m2 = L*(1.0+S);
    else
      m2 = L+S-L*S;
    m1 = 2.0*L-m2;
    *r = HuetoRGB(m1,m2,H+1.0/3.0);
    *g = HuetoRGB(m1,m2,H);
    *b = HuetoRGB(m1,m2,H-1.0/3.0);
  }
}


