/*
    Histogram Equalization Filter for VirtualDub -- automatic contrast adjustment
	by means of histogram equalization. Copyright (C) 2001 Donald A. Graft

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	The author can be contacted at:
	Donald Graft
	neuron2@home.com.
*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <crtdbg.h>
#include <math.h>
#include <stdlib.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
int StartProc(FilterActivation *fa, const FilterFunctions *ff);
int EndProc(FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

#define NONE   0
#define WEAK   1
#define STRONG 2

typedef struct MyFilterData {
	IFilterPreview		*ifp;
	int					strength;
	int					intensity;
	int					antibanding;
	unsigned long       inHistogram[256];	// Input histogram
	unsigned long		outHistogram[256];	// Output histogram
	unsigned long		LUT[256];			// Lookup table derived from Histogram[]
	unsigned long		jran, ia, ic, im;
	RECT				rInHisto;
	RECT				rOutHisto;
	long				inHistoMax;
	long				outHistoMax;
	int					disable;
} MyFilterData;

bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	_snprintf(buf, buflen, "Config(%d, %d, %d)", mfd->strength, mfd->intensity, mfd->antibanding);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->strength		= argv[0].asInt();
	mfd->intensity		= argv[1].asInt();
	mfd->antibanding	= argv[2].asInt();
}

ScriptFunctionDef func_defs[]={
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0iii" },
	{ NULL },
};

CScriptObject script_obj={
	NULL, func_defs
};

struct FilterDefinition filterDef_tutorial = {

	NULL, NULL, NULL,		// next, prev, module
	"histogram equalize (1.1)",	// name
	"Perform global color histogram equalization.",
							// desc
	"Donald Graft", 		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	InitProc,				// initProc
	NULL,					// deinitProc
	RunProc,				// runProc
	ParamProc,					// paramProc
	ConfigProc, 			// configProc
	StringProc, 			// stringProc
	NULL,					// startProc
	NULL,					// endProc

	&script_obj,			// script_obj
	FssProc,				// fssProc

};

long ParamProc(FilterActivation *fa, const FilterFunctions *ff) {

	fa->dst.offset = fa->src.offset;
    return 0;
}

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	const PixDim	width = fa->src.w;
	const PixDim	height = fa->src.h;
	Pixel32 *src;
	int x, y;
	unsigned int r, g, b, luma, m;
	int luthi, lutlo, lut;
	int oluma;

	/* Seed random generator for antibanding. */
	mfd->jran = 739187UL;

	/* Calculate and store the luminance and calculate the global histogram
	   based on the luminance. */
	src = fa->src.data;
    for (x = 0; x < 256; x++) mfd->inHistogram[x] = 0;
	for (y = 0; y < height; y++)
	{
		for (x = 0; x < width; x++)
		{
			r = (src[x] >> 16) & 0xff;
			g = (src[x] >> 8) & 0xff;
			b = src[x] & 0xff;
			luma = (55 * r + 182 * g + 19 * b) >> 8;
			src[x] &= 0xffffff;
			src[x] |= luma << 24;
			mfd->inHistogram[luma]++;
		}
		src	= (Pixel *)((char *)src + fa->src.pitch);
	}

	/* Calculate the lookup table. */
	mfd->LUT[0] = mfd->inHistogram[0];
    /* Accumulate. */
    for (x = 1; x < 256; x++)
	{
		mfd->LUT[x] = mfd->LUT[x-1] + mfd->inHistogram[x];
	}
	/* Normalize. */
    for (x = 0; x < 256; x++)
	{
		mfd->LUT[x] = (mfd->LUT[x] * mfd->intensity) / (height * width);
	}

	/* Adjust the LUT based on the selected strength. This is an alpha
	   mix of the calculated LUT and a linear LUT with gain 1. */
	for (x = 0; x < 256; x++)
	{
		mfd->LUT[x] = (mfd->strength * mfd->LUT[x]) / 255 +
					  ((255 - mfd->strength) * x) / 255;
	}

	/* Output the equalized frame. */
	for (x = 0; x < 256; x++) mfd->outHistogram[x] = 0;
	src = fa->src.data;
	for (y = 0; y < height; y++)
	{
		for (x = 0; x < width; x++)
		{
			luma = src[x] >> 24;
			if (luma == 0)
			{
				src[x] = 0;
				mfd->outHistogram[0]++;
			}
			else
			{
				lut = mfd->LUT[luma];
				if (mfd->antibanding != NONE)
				{
					if (luma > 0)
					{
						if (mfd->antibanding == WEAK)
							lutlo = (mfd->LUT[luma] + mfd->LUT[luma-1]) / 2;
						else
							lutlo = mfd->LUT[luma-1];
					}
					else lutlo = lut;
					if (luma < 255)
					{
						if (mfd->antibanding == WEAK)
							luthi = (mfd->LUT[luma] + mfd->LUT[luma+1]) / 2;
						else
							luthi = mfd->LUT[luma+1];
					}
					else luthi = lut;
					if (lutlo != luthi)
					{
						mfd->jran = (mfd->jran * mfd->ia + mfd->ic) % mfd->im;
						lut = lutlo + ((luthi - lutlo + 1) * mfd->jran) / mfd->im;
					}
				}

				r = (src[x] >> 16) & 0xff;
				g = (src[x] >> 8) & 0xff;
				b = src[x] & 0xff;
				if (((m = max(r, max(g, b))) * lut) / luma > 255)
				{
					r = (r * 255) / m; 
					g = (g * 255) / m; 
					b = (b * 255) / m; 
				}
				else
				{
					r = (r * lut) / luma;
					g = (g * lut) / luma;
					b = (b * lut) / luma;
				}
				src[x] = (r << 16) | (g << 8) | (b);
				oluma = (55 * r + 182 * g + 19 * b) >> 8;
				mfd->outHistogram[oluma]++;
			}
		}
		src	= (Pixel *)((char *)src + fa->src.pitch);
	}
#ifdef DEBUG
	for (x = 0; x < 256; x++)
	{
		char buff[80];
		sprintf(buff, "%d: %u\n", x, mfd->outHistogram[x]);
		OutputDebugString(buff);
	}
#endif

	return 0;
}

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_tutorial;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if (!(fd_tutorial = ff->addFilter(fm, &filterDef_tutorial, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_tutorial);
}

int InitProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->strength = 200;
	mfd->intensity = 210;
	mfd->antibanding = NONE;
	mfd->ia = 4096UL;
	mfd->ic = 150889UL;
	mfd->im = 714025UL;
	mfd->disable = 1;

	return 0;
}

static void levelsButtonCallback(bool fNewState, void *pvData)
{
	HWND hdlg = (HWND)pvData;

	EnableWindow(GetDlgItem(hdlg, IDC_SAMPLE), fNewState);
	if (fNewState == FALSE)
	{
		ShowWindow(GetDlgItem(hdlg, IDC_INHISTO), SW_SHOW);
		ShowWindow(GetDlgItem(hdlg, IDC_OUTHISTO), SW_SHOW);
	}
}

static void updateHistoDisplay(HWND hdlg, MyFilterData *mfd)
{
	int i;

	mfd->inHistoMax = mfd->inHistogram[0];
	for(i=1; i<256; i++)
		if ((long) mfd->inHistogram[i] > mfd->inHistoMax)
			mfd->inHistoMax = mfd->inHistogram[i];
	mfd->outHistoMax = mfd->outHistogram[0];
	for(i=1; i<256; i++)
		if ((long) mfd->outHistogram[i] > mfd->outHistoMax)
			mfd->outHistoMax = mfd->outHistogram[i];
	InvalidateRect(hdlg, &mfd->rInHisto, FALSE);
	InvalidateRect(hdlg, &mfd->rOutHisto, FALSE);
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg) {
		case WM_INITDIALOG:
			SetWindowLong(hdlg, DWL_USER, lParam);
			mfd = (MyFilterData *)lParam;

			GetWindowRect(GetDlgItem(hdlg, IDC_INHISTO), &mfd->rInHisto);
			ScreenToClient(hdlg, (POINT *)&mfd->rInHisto + 0);
			ScreenToClient(hdlg, (POINT *)&mfd->rInHisto + 1);
			GetWindowRect(GetDlgItem(hdlg, IDC_OUTHISTO), &mfd->rOutHisto);
			ScreenToClient(hdlg, (POINT *)&mfd->rOutHisto + 0);
			ScreenToClient(hdlg, (POINT *)&mfd->rOutHisto + 1);
			mfd->ifp->SetButtonCallback(levelsButtonCallback, (void *)hdlg);

			CheckDlgButton(hdlg, IDC_WEAK, mfd->antibanding == WEAK ? BST_CHECKED : BST_UNCHECKED);
			CheckDlgButton(hdlg, IDC_STRONG, mfd->antibanding == STRONG ? BST_CHECKED : BST_UNCHECKED);
			SendMessage(GetDlgItem(hdlg, IDC_STRENGTH), TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 255));
			SendMessage(GetDlgItem(hdlg, IDC_STRENGTH), TBM_SETTICFREQ, 10 , 0);
			SendMessage(GetDlgItem(hdlg, IDC_STRENGTH), TBM_SETPOS, (WPARAM)TRUE, mfd->strength);
			SendMessage(GetDlgItem(hdlg, IDC_INTENSITY), TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 255));
			SendMessage(GetDlgItem(hdlg, IDC_INTENSITY), TBM_SETTICFREQ, 10 , 0);
			SendMessage(GetDlgItem(hdlg, IDC_INTENSITY), TBM_SETPOS, (WPARAM)TRUE, mfd->intensity);
			SetDlgItemInt(hdlg, IDC_STRENGTHVAL, mfd->strength, FALSE);
			SetDlgItemInt(hdlg, IDC_INTENSITYVAL, mfd->intensity, FALSE);
			mfd->ifp->InitButton(GetDlgItem(hdlg, IDPREVIEW));

			return TRUE;

		case WM_COMMAND:
			switch(LOWORD(wParam)) {
			case IDPREVIEW:
				mfd->disable = 1;
				mfd->ifp->Toggle(hdlg);
				break;
			case IDOK:
				EndDialog(hdlg, 0);
				return TRUE;
 			case IDHELP:
				{
				char prog[256];
				char path[256];
				LPTSTR ptr;

				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\Histo.html");
				ShellExecute(hdlg, "open", path, NULL, NULL, SW_SHOWNORMAL);
				return TRUE;
				}
			case IDCANCEL:
				EndDialog(hdlg, 1);
				return TRUE;
			case IDC_SAMPLE:
				mfd->disable = 0;
				ShowWindow(GetDlgItem(hdlg, IDC_INHISTO), SW_HIDE);
				ShowWindow(GetDlgItem(hdlg, IDC_OUTHISTO), SW_HIDE);
				updateHistoDisplay(hdlg, mfd);
				break;
			case IDC_WEAK:
				if (IsDlgButtonChecked(hdlg, IDC_WEAK))
				{
					mfd->antibanding = WEAK;
					CheckDlgButton(hdlg, IDC_STRONG, BST_UNCHECKED);
				}
				else
				{
					mfd->antibanding = NONE;
				}
				mfd->ifp->RedoFrame();
				updateHistoDisplay(hdlg, mfd);
				break;
			case IDC_STRONG:
				if (IsDlgButtonChecked(hdlg, IDC_STRONG))
				{
					mfd->antibanding = STRONG;
					CheckDlgButton(hdlg, IDC_WEAK, BST_UNCHECKED);
				}
				else
				{
					mfd->antibanding = NONE;
				}
				mfd->ifp->RedoFrame();
				updateHistoDisplay(hdlg, mfd);
				break;
		}
		case WM_HSCROLL:
			{
				int strength, intensity;
				strength = SendMessage(GetDlgItem(hdlg, IDC_STRENGTH), TBM_GETPOS, 0, 0);
				intensity = SendMessage(GetDlgItem(hdlg, IDC_INTENSITY), TBM_GETPOS, 0, 0);
				if (strength != mfd->strength || intensity != mfd->intensity)
				{
					mfd->strength = strength;
					mfd->intensity = intensity;
					SetDlgItemInt(hdlg, IDC_STRENGTHVAL, strength, FALSE);
					SetDlgItemInt(hdlg, IDC_INTENSITYVAL, intensity, FALSE);
					mfd->ifp->RedoFrame();
					updateHistoDisplay(hdlg, mfd);
				}
				break;
			}
		case WM_PAINT:
		{
			HDC hdc;
			PAINTSTRUCT ps;
			RECT rPaint, rClip;
			int i;
			int x, xlo, xhi, w;
			long lMax;
			int xp, yp, h;
			long y;

			if (mfd->disable == 1)
			{
				return FALSE;
			}
			hdc = BeginPaint(hdlg, &ps);
			i = GetClipBox(hdc, &rClip);
			/* Input histogram. */
			if (i==ERROR || i==NULLREGION || IntersectRect(&rPaint, &mfd->rInHisto, &rClip))
			{
				lMax = mfd->inHistoMax;
				w = mfd->rInHisto.right - mfd->rInHisto.left;
				if (i==NULLREGION) {
					xlo = 0;
					xhi = w;
				} else {
					xlo = rPaint.left - mfd->rInHisto.left;
					xhi = rPaint.right - mfd->rInHisto.left;
				}
				FillRect(hdc, &mfd->rInHisto, (HBRUSH)GetStockObject(WHITE_BRUSH));
				for(x=xlo; x<xhi; x++)
				{
					i = (x * 0xFF00) / (w-1);

					if (i >= 0xFF00)
						y = mfd->inHistogram[255];

					else
						y = mfd->inHistogram[i>>8];

					xp = x+mfd->rInHisto.left;
					yp = mfd->rInHisto.bottom-1;
					h = MulDiv(y, mfd->rInHisto.bottom-mfd->rInHisto.top, lMax);

					if (h>0) {
						MoveToEx(hdc, x+mfd->rInHisto.left, yp, NULL);
						LineTo(hdc, x+mfd->rInHisto.left, yp - h);
					}
				}
			}
			/* Output histogram. */
			if (i==ERROR || i==NULLREGION || IntersectRect(&rPaint, &mfd->rOutHisto, &rClip))
			{
				lMax = mfd->outHistoMax;
				w = mfd->rOutHisto.right - mfd->rOutHisto.left;
				if (i==NULLREGION) {
					xlo = 0;
					xhi = w;
				} else {
					xlo = rPaint.left - mfd->rOutHisto.left;
					xhi = rPaint.right - mfd->rOutHisto.left;
				}
				FillRect(hdc, &mfd->rOutHisto, (HBRUSH)GetStockObject(WHITE_BRUSH));
				for(x=xlo; x<xhi; x++)
				{
					i = (x * 0xFF00) / (w-1);

					if (i >= 0xFF00)
						y = mfd->outHistogram[255];

					else
						y = mfd->outHistogram[i>>8];

					xp = x+mfd->rOutHisto.left;
					yp = mfd->rOutHisto.bottom-1;
					h = MulDiv(y, mfd->rOutHisto.bottom-mfd->rOutHisto.top, lMax);
					if (h>0) {
						MoveToEx(hdc, x+mfd->rOutHisto.left, yp, NULL);
						LineTo(hdc, x+mfd->rOutHisto.left, yp - h);
					}
				}
			}
			EndPaint(hdlg, &ps);
			break;
		}
	}
	return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	MyFilterData mfd_old = *mfd;
	int ret;

	mfd->disable = 1;
	mfd->ifp = fa->ifp;

	if (DialogBoxParam(fa->filter->module->hInstModule, MAKEINTRESOURCE(IDD_FILTER),
		hwnd, ConfigDlgProc, (LPARAM)mfd))
	{
		*mfd = mfd_old;
		ret = TRUE;
	}
    else
	{
		ret = FALSE;
	}
	return(ret);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	char tmp[5];

	switch (mfd->antibanding)
	{
	case NONE:
		strcpy(tmp, "none");
		break;
	case WEAK:
		strcpy(tmp, "weak");
		break;
	case STRONG:
		strcpy(tmp, "strong");
		break;
	}
	sprintf(str, " (str %d, inten %d, %s)", mfd->strength, mfd->intensity, tmp);
}
