/*
    Smart Deinterlacing Filter for VirtualDub -- performs deinterlacing only
    in moving picture areas, allowing full resolution in static areas.
    Copyright (C) 1999-2001 Donald A. Graft
    Miscellaneous suggestions and optimizations by Avery Lee.
    Useful suggestions by Hans Zimmer, Jim Casaburi, Ondrej Kavka, 
	and Gunnar Thalin. Field-only differencing based on algorithm by
	Gunnar Thalin.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    The author can be contacted at:
    Donald Graft
    neuron2@home.com.
*/

#include <windows.h>
#include <stdio.h>
#include <crtdbg.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
int StartProc(FilterActivation *fa, const FilterFunctions *ff);
int EndProc(FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

#define FRAME_ONLY 0
#define FIELD_ONLY 1
#define FRAME_AND_FIELD 2

typedef struct MyFilterData {
	int					*prevFrame;
	int					*saveFrame;
	unsigned char		*moving;
	unsigned char		*fmoving;
	int					motionOnly;
	int 				Blend;
	int 				threshold;
	int					scenethreshold;
	int					fieldShift;
	int					inswap;
	int					outswap;
	int					highq;
	int					diffmode;
	int					colordiff;
	int					noMotion;
	int					cubic;
} MyFilterData;

bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	_snprintf(buf, buflen, "Config(%d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d)",
		mfd->motionOnly,
		mfd->Blend,
		mfd->threshold,
		mfd->scenethreshold,
		mfd->fieldShift,
		mfd->inswap,
		mfd->outswap,
		mfd->noMotion,
		mfd->highq,
		mfd->diffmode,
		mfd->colordiff,
		mfd->cubic);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->motionOnly		= !!argv[0].asInt();
	mfd->Blend			= !!argv[1].asInt();
	mfd->threshold		= argv[2].asInt();
	mfd->scenethreshold	= argv[3].asInt();
	mfd->fieldShift		= !!argv[4].asInt();
	mfd->inswap			= !!argv[5].asInt();
	mfd->outswap		= !!argv[6].asInt();
	mfd->noMotion		= !!argv[7].asInt();
	mfd->highq			= !!argv[8].asInt();
	mfd->diffmode		= argv[9].asInt();
	mfd->colordiff		= !!argv[10].asInt();
	mfd->cubic			= !!argv[11].asInt();
}

ScriptFunctionDef func_defs[] =
{
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0iiiiiiiiiiii" },
	{ NULL },
};

CScriptObject script_obj =
{
	NULL, func_defs
};

struct FilterDefinition filterDef_tutorial =
{

	NULL, NULL, NULL,		// next, prev, module
	"smart deinterlace (2.7 beta 2)",	// name
	"Performs deinterlacing only in moving picture areas.",
							// desc
	"Donald Graft", 		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	InitProc,				// initProc
	NULL,					// deinitProc
	RunProc,				// runProc
	NULL,					// paramProc
	ConfigProc, 			// configProc
	NULL, 					// stringProc
	StartProc,				// startProc
	EndProc,				// endProc

	&script_obj,			// script_obj
	FssProc,				// fssProc

};

int StartProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	if (mfd->diffmode == FRAME_ONLY || mfd->diffmode == FRAME_AND_FIELD)
	{
		mfd->prevFrame = new int[fa->src.w*fa->src.h];
		memset(mfd->prevFrame, 0, fa->src.w*fa->src.h*sizeof(int));
	}

	if (mfd->fieldShift ||
		(mfd->inswap && !mfd->outswap) || (!mfd->inswap && mfd->outswap))
	{
		mfd->saveFrame = new int[fa->src.w*fa->src.h];
	}

	if (!mfd->noMotion)
	{
		mfd->moving = new unsigned char[fa->src.w*fa->src.h];
		memset(mfd->moving, 0, fa->src.w*fa->src.h*sizeof(unsigned char));
	}

	if (mfd->highq)
	{
		mfd->fmoving = new unsigned char[fa->src.w*fa->src.h];
	}
	
	return 0;
}

int EndProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	if (mfd->diffmode == FRAME_ONLY || mfd->diffmode == FRAME_AND_FIELD)
	{
		delete[] mfd->prevFrame;
		mfd->prevFrame = NULL;
	}

	if (mfd->fieldShift ||
		(mfd->inswap && !mfd->outswap) || (!mfd->inswap && mfd->outswap))
	{
		delete[] mfd->saveFrame;
		mfd->saveFrame = NULL;
	}

	if (!mfd->noMotion)
	{
		delete[] mfd->moving;
		mfd->moving = NULL;
	}

	if (mfd->highq)
	{
		delete[] mfd->fmoving;
		mfd->fmoving = NULL;
	}

	return 0;
}

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData	*mfd = (MyFilterData *)fa->filter_data;
	const int		srcpitch = fa->src.pitch;
	const int		srcpitchtimes2 = 2 * srcpitch;
	const int		dstpitch = fa->dst.pitch;
	const int		dstpitchtimes2 = 2 * dstpitch;
	const PixDim	w = fa->src.w;
	const int		wminus1 = w - 1;
	const int		wtimes2 = w * 2;
	const int		wtimes4 = w * 4;
	const PixDim	h = fa->src.h;
	const int		hminus1 = h - 1;
	const int		hminus3 = h - 3;
	const int		hover2 = h / 2;
	Pixel32			*src, *dst, *srcminus, *srcplus, *srcminusminus, *srcplusplus;
	unsigned char	*moving, *movingminus, *movingplus;
	unsigned char	*fmoving;
	int				*saved, *sv;
	Pixel32 		*src1, *src2, *s1, *s2;
	Pixel32 		*dst1, *dst2, *d1, *d2;
	int				*prev;
	int				scenechange;
	long			count;
	int				x, y;
	long			prevValue, nextValue, luma, lumap, luman;
	Pixel32			p0, p1, p2;
	long			r, g, b, rp, gp, bp, rn, gn, bn, T;
	long			rpp, gpp, bpp, rnn, gnn, bnn, R, G, B;
	unsigned char	frMotion, fiMotion;
	int				copyback;
	int				cubic = mfd->cubic;

	/* If we are performing Advanced Processing... */
	if (mfd->inswap || mfd->outswap || mfd->fieldShift)
	{
		/* Advanced Processing is used typically to clean up PAL video
		   which has erroneously been digitized with the field phase off by
		   one field. The result is that the frames look interlaced,
		   but really if we can phase shift by one field, we'll get back
		   the original progressive frames. Also supported are field swaps
		   before and/or after the phase shift to accommodate different
		   capture cards and telecining methods, as explained in the
		   help file. Finally, the user can optionally disable full
		   motion processing after this processing. */
		copyback = 1;
		if (!mfd->fieldShift)
		{
			/* No phase shift enabled, but we have swap(s) enabled. */
			if (mfd->inswap && mfd->outswap)
			{
				if (mfd->noMotion)
				{
					/* Swapping twice is a null operation. */
					src1 = fa->src.data;
					dst1 = fa->dst.data;
					for (y = 0; y < h; y++)
					{
						memcpy(dst1, src1, wtimes4);
						src1 = (Pixel *)((char *)src1 + srcpitch);
						dst1 = (Pixel *)((char *)dst1 + dstpitch);
					}
					return 0;
				}
				else
				{
					copyback = 0;
				}
			}
			else
			{
				/* Swap fields. */
				src1 = (Pixel32 *)((char *)fa->src.data + fa->src.pitch);
				saved = mfd->saveFrame + w;
				for (y = 0; y < hover2; y++)
				{					
					memcpy(saved, src1, wtimes4);
					src1 = (Pixel *)((char *)src1 + srcpitchtimes2);
					saved += wtimes2;
				}
				src1 = fa->src.data;
				dst1 = (Pixel32 *)((char *)fa->dst.data + fa->dst.pitch);
				for (y = 0; y < hover2; y++)
				{					
					memcpy(dst1, src1, wtimes4);
					src1 = (Pixel *)((char *)src1 + srcpitchtimes2);
					dst1 = (Pixel *)((char *)dst1 + dstpitchtimes2);
				}
				dst1 = fa->dst.data;
				saved = mfd->saveFrame + w;
				for (y = 0; y < hover2; y++)
				{					
					memcpy(dst1, saved, wtimes4);
					dst1 = (Pixel *)((char *)dst1 + dstpitchtimes2);
					saved += wtimes2;
				}
			}
		}
		/* If we reach here, then phase shift has been enabled. */
		else
		{
			switch (mfd->inswap | (mfd->outswap << 1))
			{
			case 0:
				/* No inswap, no outswap. */
				src1 = fa->src.data;
				src2 = (Pixel32 *)((char *)fa->src.data + srcpitch);
				dst1 = (Pixel32 *)((char *)fa->dst.data + dstpitch);
				dst2 = fa->dst.data;
				saved = mfd->saveFrame + w;
				break;
			case 1:
				/* Inswap, no outswap. */
				src1 = (Pixel32 *)((char *)fa->src.data + srcpitch);
				src2 = fa->src.data;
				dst1 = (Pixel32 *)((char *)fa->dst.data + dstpitch);
				dst2 = fa->dst.data;
				saved = mfd->saveFrame;
				break;
			case 2:
				/* No inswap, outswap. */
				src1 = fa->src.data;
				src2 = (Pixel32 *)((char *)fa->src.data + srcpitch);
				dst1 = fa->dst.data;
				dst2 = (Pixel32 *)((char *)fa->dst.data + dstpitch);
				saved = mfd->saveFrame + w;
				break;
			case 3:
				/* Inswap, outswap. */
				src1 = (Pixel32 *)((char *)fa->src.data + srcpitch);
				src2 = fa->src.data;
				dst1 = fa->dst.data;
				dst2 = (Pixel32 *)((char *)fa->dst.data + dstpitch);
				saved = mfd->saveFrame;
				break;
			}

			s1 = src1;
			d1 = dst1;
			for (y = 0; y < hover2; y++)
			{
				memcpy(d1, s1, wtimes4);
				s1 = (Pixel *)((char *)s1 + srcpitchtimes2);
				d1 = (Pixel *)((char *)d1 + dstpitchtimes2);
			}

			/* If this is not the first frame, copy the buffered field
			   of the last frame to the output. This creates a correct progressive
			   output frame. If this is the first frame, a buffered field is not
			   available, so interpolate the field from the current field. */
			if (fa->pfsi->lCurrentSourceFrame == 0)
			{
				s1 = src1;
				d2 = dst2;
				for (y = 0; y < hover2; y++)
				{
					memcpy(d2, s1, wtimes4);
					s1 = (Pixel *)((char *)s1 + srcpitchtimes2);
					d2 = (Pixel *)((char *)d2 + dstpitchtimes2);
				}
			}
			else
			{
				d2 = dst2;
				sv = saved;
				for (y = 0; y < hover2; y++)
				{
					memcpy(d2, sv, wtimes4);
					sv += wtimes2;
					d2 = (Pixel *)((char *)d2 + dstpitchtimes2);
				}
			}
			/* Finally, save the unused field of the current frame in the buffer.
			   It will be used to create the next frame. */
			s2 = src2;
			sv = saved;
			for (y = 0; y < hover2; y++)
			{
				memcpy(sv, s2, wtimes4);
				sv += wtimes2;
				s2 = (Pixel *)((char *)s2 + srcpitchtimes2);
			}
		}
		if (mfd->noMotion) return 0;

		if (copyback)
		{
			/* We're going to do motion processing also, so copy
			   the result back into the src bitmap. */
			src1 = fa->dst.data;
			dst1 = fa->src.data;
			for (y = 0; y < h; y++)
			{
				memcpy(dst1, src1, wtimes4);
				src1 = (Pixel *)((char *)src1 + srcpitch);
				dst1 = (Pixel *)((char *)dst1 + dstpitch);
			}
		}
	}
	else if (mfd->noMotion)
	{
		/* Well, I suppose somebody might select no advanced processing options
		   but tick disable motion processing. This covers that. */
		src1 = fa->src.data;
		dst1 = fa->dst.data;
		for (y = 0; y < h; y++)
		{
			memcpy(dst1, src1, srcpitch);
			src1 = (Pixel *)((char *)src1 + srcpitch);
			dst1 = (Pixel *)((char *)dst1 + dstpitch);
		}
		return 0;
	}

	/* End advanced processing mode code. Now do full motion-adaptive deinterlacing. */

	/* Not much deinterlacing to do if there aren't at least 2 lines. */
	if (h < 2) return 0;

	count = 0;
	if (mfd->diffmode == FRAME_ONLY || mfd->diffmode == FRAME_AND_FIELD)
	{
		/* Skip first and last lines, they'll get a free ride. */
		src = (Pixel *)((char *)fa->src.data + srcpitch);
		srcminus = (Pixel *)((char *)src - srcpitch);
		prev = mfd->prevFrame + w;
		moving = mfd->moving + w;
		for (y = 1; y < hminus1; y++)
		{
			x = 0;
			do
			{
				// First check frame motion.
				// Set the moving flag if the diff exceeds the configured
				// threshold.
				moving[x] = 0;
				frMotion = 0;
				prevValue = *prev;
				if (!mfd->colordiff)
				{
					r = (src[x] >> 16) & 0xff;
					g = (src[x] >> 8) & 0xff;
					b = src[x] & 0xff;
					luma = (76 * r + 30 * b + 150 * g) >> 8;
					if (abs(luma - prevValue) > mfd->threshold) frMotion = 1;
				}
				else
				{
					b = src[x] & 0xff;
					bp = prevValue & 0xff;
					if (abs(b - bp) > mfd->threshold) frMotion = 1;
					else
					{
						r = (src[x] >>16) & 0xff;
						rp = (prevValue >> 16) & 0xff;
						if (abs(r - rp) > mfd->threshold) frMotion = 1;
						else
						{
							g = (src[x] >> 8) & 0xff;
							gp = (prevValue >> 8) & 0xff;
							if (abs(g - gp) > mfd->threshold) frMotion = 1;
						}
					}
				}

				// Now check field motion if applicable.
				if (mfd->diffmode == FRAME_ONLY) moving[x] = frMotion;
				else
				{
					fiMotion = 0;
					if (y & 1)
						prevValue = srcminus[x];
					else
						prevValue = *(prev + w);
					if (!mfd->colordiff)
					{
						r = (src[x] >> 16) & 0xff;
						g = (src[x] >> 8) & 0xff;
						b = src[x] & 0xff;
						luma = (76 * r + 30 * b + 150 * g) >> 8;
						if (abs(luma - prevValue) > mfd->threshold) fiMotion = 1;
					}
					else
					{
						b = src[x] & 0xff;
						bp = prevValue & 0xff;
						if (abs(b - bp) > mfd->threshold) fiMotion = 1;
						else
						{
							r = (src[x] >> 16) & 0xff;
							rp = (prevValue >> 16) & 0xff;
							if (abs(r - rp) > mfd->threshold) fiMotion = 1;
							else
							{
								g = (src[x] >> 8) & 0xff;
								gp = (prevValue >> 8) & 0xff;
								if (abs(g - gp) > mfd->threshold) fiMotion = 1;
							}
						}
					}
					moving[x] = (fiMotion && frMotion);
				}
				if (!mfd->colordiff)
					*prev++ = luma;
				else
					*prev++ = src[x];
				/* Keep a count of the number of moving pixels for the
				   scene change detection. */
				if (moving[x]) count++;
			} while(++x < w);
			src = (Pixel *)((char *)src + srcpitch);
			srcminus = (Pixel *)((char *)srcminus + srcpitch);
			moving += w;
		}

		/* Determine whether a scene change has occurred. */
		if ((100L * count) / (h * w) >= mfd->scenethreshold) scenechange = 1;
		else scenechange = 0;

		/* Perform a denoising of the motion map if enabled. */
		if (!scenechange && mfd->highq)
		{
			int xlo, xhi, ylo, yhi;
			int u, v;
			int N = 5;
			int Nover2 = N/2;
			int sum;
			unsigned char *m;

			// Erode.
			fmoving = mfd->fmoving;
			for (y = 0; y < h; y++)
			{
				for (x = 0; x < w; x++)
				{
					if (!((mfd->moving + y * w)[x]))
					{
						fmoving[x] = 0;	
						continue;
					}
					xlo = x - Nover2; if (xlo < 0) xlo = 0;
					xhi = x + Nover2; if (xhi >= w) xhi = wminus1;
					ylo = y - Nover2; if (ylo < 0) ylo = 0;
					yhi = y + Nover2; if (yhi >= h) yhi = hminus1;
					m = mfd->moving + ylo * w;
					sum = 0;
					for (u = ylo; u <= yhi; u++)
					{
						for (v = xlo; v <= xhi; v++)
						{
							sum += m[v];
						}
						m += w;
					}
					if (sum > 9)
						fmoving[x] = 1;
					else
						fmoving[x] = 0;
				}
				fmoving += w;
			}
			// Dilate.
			N = 5;
			Nover2 = N/2;
			moving = mfd->moving;
			for (y = 0; y < h; y++)
			{
				for (x = 0; x < w; x++)
				{
					if (!((mfd->fmoving + y * w)[x]))
					{
						moving[x] = 0;	
						continue;
					}
					xlo = x - Nover2; if (xlo < 0) xlo = 0;
					xhi = x + Nover2; if (xhi >= w) xhi = wminus1;
					ylo = y - Nover2; if (ylo < 0) ylo = 0;
					yhi = y + Nover2; if (yhi >= h) yhi = hminus1;
					m = mfd->moving + ylo * w;
					for (u = ylo; u <= yhi; u++)
					{
						for (v = xlo; v <= xhi; v++)
						{
							m[v] = 1;
						}
						m += w;
					}
				}
				moving += w;
			}
		}
	}
	else
	{
		/* Field differencing only mode. */
		T = mfd->threshold * mfd->threshold;
		src = (Pixel *)((char *)fa->src.data + srcpitch);
		srcminus = (Pixel *)((char *)src - srcpitch);
		srcplus = (Pixel *)((char *)src + srcpitch);
		moving = mfd->moving + w;
		for (y = 1; y < hminus1; y++)
		{
			x = 0;
			do
			{
				// Set the moving flag if the diff exceeds the configured
				// threshold.
				moving[x] = 0;
				if (y & 1)
				{
					// Now check field motion.
					fiMotion = 0;
					nextValue = srcplus[x];
					prevValue = srcminus[x];
					if (!mfd->colordiff)
					{
						r = (src[x] >> 16) & 0xff;
						rp = (prevValue >> 16) & 0xff;
						rn = (nextValue >> 16) & 0xff;
						g = (src[x] >> 8) & 0xff;
						gp = (prevValue >> 8) & 0xff;
						gn = (nextValue >> 8) & 0xff;
						b = src[x] & 0xff;
						bp = prevValue & 0xff;
						bn = nextValue & 0xff;
						luma = (76 * r + 30 * b + 150 * g) >> 8;
						lumap = (76 * rp + 30 * bp + 150 * gp) >> 8;
						luman = (76 * rn + 30 * bn + 150 * gn) >> 8;
						if ((lumap - luma) * (luman - luma) > T)
							moving[x] = 1;
					}
					else
					{
						b = src[x] & 0xff;
						bp = prevValue & 0xff;
						bn = nextValue & 0xff;
						if ((bp - b) * (bn - b) > T) moving[x] = 1;
						else
						{
							r = (src[x] >> 16) & 0xff;
							rp = (prevValue >> 16) & 0xff;
							rn = (nextValue >> 16) & 0xff;
							if ((rp - r) * (rn - r) > T) moving[x] = 1;
							else
							{
								g = (src[x] >> 8) & 0xff;
								gp = (prevValue >> 8) & 0xff;
								gn = (nextValue >> 8) & 0xff;
								if ((gp - g) * (gn - g) > T) moving[x] = 1;
							}
						}
					}
				}
				/* Keep a count of the number of moving pixels for the
				   scene change detection. */
				if (moving[x]) count++;
			} while(++x < w);
			src = (Pixel *)((char *)src + srcpitch);
			srcminus = (Pixel *)((char *)srcminus + srcpitch);
			srcplus = (Pixel *)((char *)srcplus + srcpitch);
			moving += w;
		}

		/* Determine whether a scene change has occurred. */
		if ((100L * count) / (h * w) >= mfd->scenethreshold) scenechange = 1;
		else scenechange = 0;

		/* Perform a denoising of the motion map if enabled. */
		if (!scenechange && mfd->highq)
		{
			int xlo, xhi, ylo, yhi;
			int u, v;
			int N = 5;
			int Nover2 = N/2;
			int sum;
			unsigned char *m;

			// Erode.
			fmoving = mfd->fmoving;
			for (y = 0; y < h; y++)
			{
				for (x = 0; x < w; x++)
				{
					if (!((mfd->moving + y * w)[x]))
					{
						fmoving[x] = 0;	
						continue;
					}
					xlo = x - Nover2; if (xlo < 0) xlo = 0;
					xhi = x + Nover2; if (xhi >= w) xhi = wminus1;
					ylo = y - Nover2; if (ylo < 0) ylo = 0;
					yhi = y + Nover2; if (yhi >= h) yhi = hminus1;
					m = mfd->moving + ylo * w;
					sum = 0;
					for (u = ylo; u <= yhi; u++)
					{
						for (v = xlo; v <= xhi; v++)
						{
							sum += m[v];
						}
						m += w;
					}
					if (sum > 9)
						fmoving[x] = 1;
					else
						fmoving[x] = 0;
				}
				fmoving += w;
			}

			// Dilate.
			N = 5;
			Nover2 = N/2;
			moving = mfd->moving;
			for (y = 0; y < h; y++)
			{
				for (x = 0; x < w; x++)
				{
					if (!((mfd->fmoving + y * w)[x]))
					{
						moving[x] = 0;	
						continue;
					}
					xlo = x - Nover2; if (xlo < 0) xlo = 0;
					xhi = x + Nover2; if (xhi >= w) xhi = wminus1;
					ylo = y - Nover2; if (ylo < 0) ylo = 0;
					yhi = y + Nover2; if (yhi >= h) yhi = hminus1;
					m = mfd->moving + ylo * w;
					for (u = ylo; u <= yhi; u++)
					{
						for (v = xlo; v <= xhi; v++)
						{
							m[v] = 1;
						}
						m += w;
					}
				}
				moving += w;
			}		
		}
	}

	// Render.
    // The first line gets a free ride.
	src = fa->src.data;
	dst = fa->dst.data;
	memcpy(dst, src, wtimes4);
	src = (Pixel *)((char *)fa->src.data + fa->src.pitch);
	srcminus = (Pixel *)((char *)src - srcpitch);
	srcplus = (Pixel *)((char *)src + srcpitch);
	if (cubic)
	{
		srcminusminus = (Pixel *)((char *)src - 3 * srcpitch);
		srcplusplus = (Pixel *)((char *)src + 3 * srcpitch);
	}
	dst = (Pixel *)((char *)fa->dst.data + fa->dst.pitch);
	moving = mfd->moving + w;
	movingminus = moving - w;
	movingplus = moving + w;
	for (y = 1; y < hminus1; y++)
	{
		if (mfd->motionOnly)
		{
			if (mfd->Blend)
			{
				x = 0;
				do {
					if (!(movingminus[x] | moving[x] | movingplus[x]) && !scenechange)
						dst[x] = 0x7f7f7f;
					else
					{	
						/* Blend fields. */
						p0 = src[x];
						p0 &= 0x00fefefe;

						p1 = srcminus[x];
						p1 &= 0x00fcfcfc;

						p2 = srcplus[x];
						p2 &= 0x00fcfcfc;

						dst[x] = (p0>>1) + (p1>>2) + (p2>>2);
					}
				} while(++x < w);
			}
			else
			{
				x = 0;
				do {
					if (!(movingminus[x] | moving[x] | movingplus[x]) && !scenechange)
						dst[x] = 0x7f7f7f;
					else if (y & 1)
					{
						if (cubic && (y > 2) && (y < hminus3))
						{
							rpp = (srcminusminus[x] >> 16) & 0xff;
							rp =  (srcminus[x] >> 16) & 0xff;
							rn =  (srcplus[x] >> 16) & 0xff;
							rnn = (srcplusplus[x] >>16) & 0xff;
							gpp = (srcminusminus[x] >> 8) & 0xff;
							gp =  (srcminus[x] >> 8) & 0xff;
							gn =  (srcplus[x] >>8) & 0xff;
							gnn = (srcplusplus[x] >> 8) & 0xff;
							bpp = (srcminusminus[x]) & 0xff;
							bp =  (srcminus[x]) & 0xff;
							bn =  (srcplus[x]) & 0xff;
							bnn = (srcplusplus[x]) & 0xff;
							R = (5 * (rp + rn) - (rpp + rnn)) >> 3;
							if (R > 255) R = 255;
							else if (R < 0) R = 0;
							G = (5 * (gp + gn) - (gpp + gnn)) >> 3;
							if (G > 255) G = 255;
							else if (G < 0) G = 0;
							B = (5 * (bp + bn) - (bpp + bnn)) >> 3;
							if (B > 255) B = 255;
							else if (B < 0) B = 0;
							dst[x] = (R << 16) | (G << 8) | B;  
						}
						else
						{
							p1 = srcminus[x];
							p1 &= 0x00fefefe;

							p2 = srcplus[x];
							p2 &= 0x00fefefe;
							dst[x] = (p1>>1) + (p2>>1);
						}
					}
					else
						dst[x] = src[x];
				} while(++x < w);
			}
		}
		else  /* Not motion only */
		{
			if (mfd->Blend)
			{
				x = 0;
				do {
					if (!(movingminus[x] | moving[x] | movingplus[x]) && !scenechange)
						dst[x] = src[x];
					else
					{
						/* Blend fields. */
						p0 = src[x];
						p0 &= 0x00fefefe;

						p1 = srcminus[x];
						p1 &= 0x00fcfcfc;

						p2 = srcplus[x];
						p2 &= 0x00fcfcfc;

						dst[x] = (p0>>1) + (p1>>2) + (p2>>2);
					}
				} while(++x < w);
			}
			else
			{
				// Doing line interpolate. Thus, even lines are going through
				// for moving and non-moving mode. Odd line pixels will be subject
				// to the motion test.
				if (y&1)
				{
					x = 0;
					do {
						if (!(movingminus[x] | moving[x] | movingplus[x]) && !scenechange)
							dst[x] = src[x];
						else
						{
							if (cubic && (y > 2) && (y < hminus3))
							{
								rpp = (srcminusminus[x] >> 16) & 0xff;
								rp =  (srcminus[x] >> 16) & 0xff;
								rn =  (srcplus[x] >> 16) & 0xff;
								rnn = (srcplusplus[x] >>16) & 0xff;
								gpp = (srcminusminus[x] >> 8) & 0xff;
								gp =  (srcminus[x] >> 8) & 0xff;
								gn =  (srcplus[x] >>8) & 0xff;
								gnn = (srcplusplus[x] >> 8) & 0xff;
								bpp = (srcminusminus[x]) & 0xff;
								bp =  (srcminus[x]) & 0xff;
								bn =  (srcplus[x]) & 0xff;
								bnn = (srcplusplus[x]) & 0xff;
								R = (5 * (rp + rn) - (rpp + rnn)) >> 3;
								if (R > 255) R = 255;
								else if (R < 0) R = 0;
								G = (5 * (gp + gn) - (gpp + gnn)) >> 3;
								if (G > 255) G = 255;
								else if (G < 0) G = 0;
								B = (5 * (bp + bn) - (bpp + bnn)) >> 3;
								if (B > 255) B = 255;
								else if (B < 0) B = 0;
								dst[x] = (R << 16) | (G << 8) | B;  
							}
							else
							{
								p1 = srcminus[x];
								p1 &= 0x00fefefe;

								p2 = srcplus[x];
								p2 &= 0x00fefefe;

								dst[x] = (p1>>1) + (p2>>1);
							}
						}
					} while(++x < w);
				}
				else
				{
					// Even line; pass it through.
					memcpy(dst, src, wtimes4);
				}
			}
		}
		src = (Pixel *)((char *)src + fa->src.pitch);
		srcminus = (Pixel *)((char *)srcminus + fa->src.pitch);
		srcplus = (Pixel *)((char *)srcplus + fa->src.pitch);
		if (cubic)
		{
			srcminusminus = (Pixel *)((char *)srcminusminus + fa->src.pitch);
			srcplusplus = (Pixel *)((char *)srcplusplus + fa->src.pitch);
		}
		dst = (Pixel *)((char *)dst + fa->dst.pitch);
		moving += w;
		movingminus += w;
		movingplus += w;
	}
	
	// The last line gets a free ride.
	memcpy(dst, src, wtimes4);

	return 0;
}

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_tutorial;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if (!(fd_tutorial = ff->addFilter(fm, &filterDef_tutorial, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_tutorial);
}

int InitProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->threshold = 15;
	mfd->scenethreshold = 100;
	mfd->highq = 0;
	mfd->diffmode = FRAME_ONLY;
	mfd->colordiff = 1;
	mfd->noMotion = 0;
	mfd->cubic = 0;

	return 0;
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg) {
		case WM_INITDIALOG:
			SetWindowLong(hdlg, DWL_USER, lParam);
			mfd = (MyFilterData *)lParam;

			CheckDlgButton(hdlg, IDC_MOTION_ONLY, mfd->motionOnly ? BST_CHECKED : BST_UNCHECKED);
			CheckDlgButton(hdlg, IDC_BLEND, mfd->Blend ? BST_CHECKED : BST_UNCHECKED);
			SetDlgItemInt(hdlg, IDC_THRESHOLD, mfd->threshold, FALSE);
			SetDlgItemInt(hdlg, IDC_SCENETHRESHOLD, mfd->scenethreshold, FALSE);
			CheckDlgButton(hdlg, IDC_SHIFT, mfd->fieldShift ? BST_CHECKED : BST_UNCHECKED);
			CheckDlgButton(hdlg, IDC_INSWAP, mfd->inswap ? BST_CHECKED : BST_UNCHECKED);
			CheckDlgButton(hdlg, IDC_OUTSWAP, mfd->outswap ? BST_CHECKED : BST_UNCHECKED);
			CheckDlgButton(hdlg, IDC_NOMOTION, mfd->noMotion ? BST_CHECKED : BST_UNCHECKED);
			CheckDlgButton(hdlg, IDC_HIGHQ, mfd->highq ? BST_CHECKED : BST_UNCHECKED);
			CheckDlgButton(hdlg, IDC_COLOR_DIFF, mfd->colordiff ? BST_CHECKED : BST_UNCHECKED);
			CheckDlgButton(hdlg, IDC_CUBIC, mfd->cubic ? BST_CHECKED : BST_UNCHECKED);
			CheckDlgButton(hdlg, IDC_FRAME_ONLY, mfd->diffmode == FRAME_ONLY ? BST_CHECKED : BST_UNCHECKED);
			CheckDlgButton(hdlg, IDC_FIELD_ONLY, mfd->diffmode == FIELD_ONLY ? BST_CHECKED : BST_UNCHECKED);
			CheckDlgButton(hdlg, IDC_FRAME_AND_FIELD, mfd->diffmode == FRAME_AND_FIELD ? BST_CHECKED : BST_UNCHECKED);

			return TRUE;

		case WM_COMMAND:
			switch(LOWORD(wParam)) {
			case IDOK:
				mfd->motionOnly = !!IsDlgButtonChecked(hdlg, IDC_MOTION_ONLY);
				mfd->Blend = !!IsDlgButtonChecked(hdlg, IDC_BLEND);
				mfd->threshold = GetDlgItemInt(hdlg, IDC_THRESHOLD, &mfd->threshold, FALSE);
				mfd->scenethreshold = GetDlgItemInt(hdlg, IDC_SCENETHRESHOLD, &mfd->scenethreshold, FALSE);
				mfd->fieldShift = !!IsDlgButtonChecked(hdlg, IDC_SHIFT);
				mfd->inswap = !!IsDlgButtonChecked(hdlg, IDC_INSWAP);
				mfd->outswap = !!IsDlgButtonChecked(hdlg, IDC_OUTSWAP);
				mfd->noMotion = !!IsDlgButtonChecked(hdlg, IDC_NOMOTION);
				mfd->highq = !!IsDlgButtonChecked(hdlg, IDC_HIGHQ);
				mfd->colordiff = !!IsDlgButtonChecked(hdlg, IDC_COLOR_DIFF);
				mfd->cubic = !!IsDlgButtonChecked(hdlg, IDC_CUBIC);
				if (!!IsDlgButtonChecked(hdlg, IDC_FRAME_ONLY)) mfd->diffmode = FRAME_ONLY;
				else if (!!IsDlgButtonChecked(hdlg, IDC_FIELD_ONLY)) mfd->diffmode = FIELD_ONLY;
				else if (!!IsDlgButtonChecked(hdlg, IDC_FRAME_AND_FIELD)) mfd->diffmode = FRAME_AND_FIELD;
				EndDialog(hdlg, 0);
				return TRUE;
			case IDHELP:
				{
				char prog[256];
				char path[256];
				LPTSTR ptr;

				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\Smart.html");
				ShellExecute(hdlg, "open", path, NULL, NULL, SW_SHOWNORMAL);
				return TRUE;
				}
			case IDCANCEL:
				EndDialog(hdlg, 1);
				return FALSE;
			}
			break;
	}

	return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd) {
	return DialogBoxParam(fa->filter->module->hInstModule,
			MAKEINTRESOURCE(IDD_FILTER), hwnd,
			ConfigDlgProc, (LPARAM)fa->filter_data);
}
