#include <windows.h>
#include <stdio.h>
#include <commctrl.h>

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int dnrRunProc(const FilterActivation *fa, const FilterFunctions *ff);
int dnrStartProc(FilterActivation *fa, const FilterFunctions *ff);
int dnrEndProc(FilterActivation *fa, const FilterFunctions *ff);
int dnrInitProc(FilterActivation *fa, const FilterFunctions *ff);
int dnrConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void dnrStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *buf); 

///////////////////////////////////////////////////////////////////////////


typedef struct MyFilterData {
    unsigned char   *old_data;
    long            size, level;
} MyFilterData;

struct FilterDefinition filterDef_dnr = {

    NULL, NULL, NULL,					// next, prev, module
    "dynamic noise reduction (MMX)",	// name
    "Reduces changes across frames by eliminating dynamic noise without blurring.",
										// desc
    "Steven Don and Avery Lee",         // maker
    NULL,								// private_data
    sizeof(MyFilterData),				// inst_data_size
										
    dnrInitProc,						// initProc
    NULL,								// deinitProc
    dnrRunProc,							// runProc
    NULL,								// paramProc
    dnrConfigProc,						// configProc
    dnrStringProc,						// stringProc
    dnrStartProc,						// startProc
    dnrEndProc,							// endProc
										
    NULL,								// script_obj
    NULL,								// fssProc

};

static unsigned char divTable [766];

///////////////////////////////////////////////////////////////////////////

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_dnr;

///////////////////////////////////////////////////////////////////////////

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
    if (!(fd_dnr = ff->addFilter(fm, &filterDef_dnr, sizeof(FilterDefinition))))
        return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

    return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
    ff->removeFilter(fd_dnr);
}

///////////////////////////////////////////////////////////////////////////

int dnrStartProc(FilterActivation *fa, const FilterFunctions *ff) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

    mfd->size = fa->src.Size ();

	//MMX must be enabled
    if (!ff->isMMXEnabled())
		return 1;
    
    //Make buffer and clear it
    if (!(mfd->old_data = new unsigned char [mfd->size]))
        return 1;
    
    memset (mfd->old_data, 0, mfd->size);
    return 0;
}

//MMX optimised version
void __declspec(naked) dodnrMMX(Pixel32 *dst, Pixel32 *src, PixDim w, PixDim h, PixOffset modulo, __int64 thresh1, __int64 thresh2) {
static const __int64 bythree = 0x5555555555555555i64;
static const __int64 round2 = 0x0002000200020002i64;
static const __int64 three = 0x0003000300030003i64;

	__asm {
		push	ebp
		push	edi
		push	esi
		push	ebx

		mov		edi,[esp+4+16]
		mov		esi,[esp+8+16]
		mov		edx,[esp+12+16]
		mov		ecx,[esp+16+16]
		xor		eax,eax
		mov		ebx,[esp+20+16]
		pxor	mm7,mm7
		movq	mm6,[esp+32+16]
		movq	mm5,[esp+24+16]

yloop:
		mov		ebp,edx
xloop:
		movd	mm0,[esi]		;previous
		pxor	mm7,mm7

		movd	mm1,[edi]		;current
		punpcklbw	mm0,mm7

		punpcklbw	mm1,mm7
		movq	mm2,mm0

		movq	mm4,mm1
		movq	mm3,mm1

		movq	mm7,mm0
		paddw	mm4,mm4

		pmullw	mm0,three
		psubusb	mm2,mm1

		paddw	mm4,mm7
		psubusb	mm3,mm7

		pmulhw	mm4,bythree
		por		mm2,mm3

		movq	mm3,mm2
		paddw	mm0,mm1

		paddw	mm0,round2
		pcmpgtw	mm2,mm5			;set if diff > thresh1

		pcmpgtw	mm3,mm6			;set if diff > thresh2
		psrlw	mm0,2


		;	mm2		mm3		meaning						mm1		mm0		mm4
		;	FALSE	FALSE	diff <= thresh1				off		on		off
		;	FALSE	TRUE	impossible
		;	TRUE	FALSE	thresh1 < diff <= thresh2	off		off		on
		;	TRUE	TRUE	diff > thresh2				on		off		off

		pand	mm1,mm3			;keep pixels exceeding threshold2
		pand	mm4,mm2			;	average pixels <= threshold2...
		pandn	mm2,mm0			;replace pixels below threshold1
		pandn	mm3,mm4			;	but >= threshold1...
		por		mm1,mm2
		add		esi,4
		por		mm1,mm3
		add		edi,4
		packuswb	mm1,mm1
		dec		ebp

		movd	[esi-4],mm1		;store to both
		movd	[edi-4],mm1
		jne		xloop

		add		esi,eax
		add		edi,ebx
		dec		ecx
		jne		yloop

		pop		ebx
		pop		esi
		pop		edi
		pop		ebp
		emms
		ret
	}
}

int dnrRunProc(const FilterActivation *fa, const FilterFunctions *ff) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

    if (mfd->level) {
    	__int64 thresh1 = 0x0001000100010001i64 * ((mfd->level >> 1) + 1);
		__int64 thresh2 = 0x0001000100010001i64 * (mfd->level);
        memcpy (fa->dst.data, fa->src.data, mfd->size);
		dodnrMMX ((Pixel32 *)fa->dst.data, (Pixel32 *)mfd->old_data, fa->src.w, fa->src.h, fa->src.modulo, thresh1, thresh2);
    }

    return 0;
}

int dnrEndProc(FilterActivation *fa, const FilterFunctions *ff) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;
    delete[] mfd->old_data;
    mfd->old_data = NULL;
    return 0;
}

int dnrInitProc(FilterActivation *fa, const FilterFunctions *ff) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;
    mfd->level = 12;

    for (int index = 0; index < 766; index++) divTable [index] = index / 3;

    return 0;
}

BOOL CALLBACK dnrConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
    MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

    switch(msg) {
        case WM_INITDIALOG:
            {
                SetWindowLong(hdlg, DWL_USER, lParam);
                mfd = (MyFilterData *)lParam;
                SendDlgItemMessage (hdlg, IDC_SLIDER_LEVEL, TBM_SETTICFREQ, 4, 0);
                SendDlgItemMessage (hdlg, IDC_SLIDER_LEVEL, TBM_SETRANGE, false, MAKELONG (0, 31));
                SendDlgItemMessage (hdlg, IDC_SLIDER_LEVEL, TBM_SETPOS, true, mfd->level);
                char leveltext [4];
                sprintf (leveltext, "%d", mfd->level);
                SetDlgItemText (hdlg, IDC_LEVEL, leveltext);
            }

            return TRUE;

        case WM_NOTIFY:
            {
                long level = SendDlgItemMessage (hdlg, IDC_SLIDER_LEVEL, TBM_GETPOS, 0, 0);
                char leveltext [4];
                sprintf (leveltext, "%d", level);
                SetDlgItemText (hdlg, IDC_LEVEL, leveltext);
            }

        case WM_COMMAND:
            switch(LOWORD(wParam)) {
                case IDOK:
                    mfd->level = SendDlgItemMessage (hdlg, IDC_SLIDER_LEVEL, TBM_GETPOS, 0, 0);
                    EndDialog(hdlg, 0);
                    return TRUE;
                case IDCANCEL:
                    EndDialog(hdlg, 1);
                    return FALSE;
             }
        break;
    }

    return FALSE;
}

int dnrConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd) {
    return DialogBoxParam(fa->filter->module->hInstModule,
        MAKEINTRESOURCE(IDD_DIALOG_CONFIG), hwnd,
        dnrConfigDlgProc, (LPARAM)fa->filter_data);
}

void dnrStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;
    sprintf(str, " (%d)", mfd->level);
}