#include <windows.h>
#include <stdio.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int  InitProc(FilterActivation *fa, const FilterFunctions *ff);
int  StartProc(FilterActivation *fa, const FilterFunctions *ff);
int  EndProc(FilterActivation *fa, const FilterFunctions *ff);
int  RunProc(const FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int  ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

typedef struct MyFilterData
{
    int 			xorigin;
	int				yorigin;
    int 			xspeed;
	int				xneg;
	int				yspeed;
	int             yneg;
    int 			xsize;
	int				ysize;
	int				startingframe;
	int				endingframe;
} MyFilterData;

bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	_snprintf(buf, buflen, "Config(%i, %i, %i, %i, %i, %i, %i, %i, %i, %i)",
		      mfd->xorigin, mfd->yorigin, mfd->xspeed, mfd->xneg, mfd->yspeed, mfd->yneg,
			  mfd->xsize, mfd->ysize, mfd->startingframe, mfd->endingframe);
	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc)
{
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	mfd->xorigin = argv[0].asInt();
	mfd->yorigin = argv[1].asInt();
	mfd->xspeed = argv[2].asInt();
	mfd->xneg = argv[3].asInt();
	mfd->yspeed = argv[4].asInt();
	mfd->yneg = argv[5].asInt();
	mfd->xsize = argv[6].asInt();
	mfd->ysize = argv[7].asInt();
	mfd->startingframe = argv[8].asInt();
	mfd->endingframe = argv[9].asInt();
}

ScriptFunctionDef func_defs[] =
{
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0iiiiiiiiii" },
	{ NULL },
};

CScriptObject script_obj=
{
	NULL, func_defs
};

struct FilterDefinition filterDef =
{

	NULL, NULL, NULL,	// next, prev, module
	"pan (1.0 beta 2)",	// name
	"Pan across a video.",	// desc
	"Donald Graft",		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	InitProc,		// initProc
	NULL,			// deinitProc
	RunProc,		// runProc
	ParamProc,		// paramProc
	ConfigProc,		// configProc
	StringProc,		// stringProc
	StartProc,		// startProc
	EndProc,		// endProc

	&script_obj,	// script_obj
	FssProc,		// fssProc

};

///////////////////////////////////////////////////////////////////////////

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd;

///////////////////////////////////////////////////////////////////////////

int __declspec(dllexport) __cdecl
VirtualdubFilterModuleInit2(FilterModule *fm,
						  const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat)
{
    if (!(fd = ff->addFilter(fm, &filterDef, sizeof(FilterDefinition))))
        return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

    return 0;
}

void __declspec(dllexport) __cdecl
VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff)
{
    ff->removeFilter(fd);
}

///////////////////////////////////////////////////////////////////////////

int InitProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->xorigin = 0;
	mfd->yorigin = 0;
	mfd->xspeed = 32;
	mfd->xneg = 0;
	mfd->yspeed = 32;
	mfd->yneg = 0;
	mfd->xsize = 160;
	mfd->ysize = 120;
	mfd->startingframe = 0;
	mfd->endingframe = 500;
	return 0;
}

int StartProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *) fa->filter_data;

	return 0;
}

int EndProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	return 0;
}

int RunProc(const FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	Pixel32 *src, *dst;
	int x, y, xspeed, yspeed;
	long X, Y;
	int frame = fa->pfsi->lCurrentSourceFrame;

	if (mfd->xneg) xspeed = - mfd->xspeed;
	else xspeed = mfd->xspeed;
	if (mfd->yneg) yspeed = - mfd->yspeed;
	else yspeed = mfd->yspeed;

	if (frame <= mfd->startingframe)
	{
		X = mfd->xorigin;
		Y = mfd->yorigin;
	}
	else if (frame > mfd->startingframe && frame <= mfd->endingframe)
	{
		X = mfd->xorigin + xspeed * (frame - mfd->startingframe) / 16;
		Y = mfd->yorigin + yspeed * (frame - mfd->startingframe) / 16;
	}
	else
	{
		X = mfd->xorigin + xspeed * (mfd->endingframe - mfd->startingframe) / 16;
		Y = mfd->yorigin + yspeed * (mfd->endingframe - mfd->startingframe) / 16;
	}
	if (X < 0) X = 0;
	if (Y < 0) Y = 0;
	if (X > fa->src.w - mfd->xsize) X = fa->src.w - mfd->xsize;
	if (Y > fa->src.h - mfd->ysize) Y = fa->src.h - mfd->ysize;
	src = (Pixel32 *)((char *) fa->src.data +
		              (fa->src.h - Y - mfd->ysize) * fa->src.pitch);
	dst = (Pixel32 *)fa->dst.data;
	for (y = mfd->ysize - 1; y >= 0; y--)
	{
		for (x = 0; x < mfd->xsize; x++)
		{
			dst[x] = src[x + X];
		}
		src = (Pixel32 *)((char *)src + fa->src.pitch);
		dst = (Pixel32 *)((char *)dst + fa->dst.pitch);
	}
    return 0;
}

long ParamProc(FilterActivation *fa, const FilterFunctions *ff)
{
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	if (mfd->xsize > fa->src.w) mfd->xsize = fa->src.w;
	if (mfd->ysize > fa->src.h) mfd->ysize = fa->src.h;
    fa->dst.w = mfd->xsize;
    fa->dst.h = mfd->ysize;
    return FILTERPARAM_SWAP_BUFFERS;
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
    MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

    switch(msg) {
        case WM_INITDIALOG:
            SetWindowLong(hdlg, DWL_USER, lParam);
            mfd = (MyFilterData *)lParam;
  			SetDlgItemInt(hdlg, IDC_X_ORIGIN, mfd->xorigin, FALSE);
  			SetDlgItemInt(hdlg, IDC_Y_ORIGIN, mfd->yorigin, FALSE);
  			SetDlgItemInt(hdlg, IDC_X_SPEED, mfd->xspeed, FALSE);
  			SetDlgItemInt(hdlg, IDC_Y_SPEED, mfd->yspeed, FALSE);
  			SetDlgItemInt(hdlg, IDC_X_SIZE, mfd->xsize, FALSE);
  			SetDlgItemInt(hdlg, IDC_Y_SIZE, mfd->ysize, FALSE);
  			SetDlgItemInt(hdlg, IDC_STARTING_FRAME, mfd->startingframe, FALSE);
  			SetDlgItemInt(hdlg, IDC_ENDING_FRAME, mfd->endingframe, FALSE);
			CheckDlgButton(hdlg, IDC_X_NEG, mfd->xneg ? BST_CHECKED : BST_UNCHECKED);
			CheckDlgButton(hdlg, IDC_Y_NEG, mfd->yneg ? BST_CHECKED : BST_UNCHECKED);
			return TRUE;

        case WM_COMMAND:
            switch(LOWORD(wParam)) {
            case IDOK:
 				mfd->xorigin = GetDlgItemInt(hdlg, IDC_X_ORIGIN, &mfd->xorigin, FALSE);
 				mfd->yorigin = GetDlgItemInt(hdlg, IDC_Y_ORIGIN, &mfd->yorigin, FALSE);
 				mfd->xspeed = GetDlgItemInt(hdlg, IDC_X_SPEED, &mfd->xspeed, FALSE);
				mfd->xneg = !!IsDlgButtonChecked(hdlg, IDC_X_NEG);
 				mfd->yspeed = GetDlgItemInt(hdlg, IDC_Y_SPEED, &mfd->yspeed, FALSE);
				mfd->yneg = !!IsDlgButtonChecked(hdlg, IDC_Y_NEG);
 				mfd->xsize = GetDlgItemInt(hdlg, IDC_X_SIZE, &mfd->xsize, FALSE);
 				mfd->ysize = GetDlgItemInt(hdlg, IDC_Y_SIZE, &mfd->ysize, FALSE);
 				mfd->startingframe = GetDlgItemInt(hdlg, IDC_STARTING_FRAME, &mfd->startingframe, FALSE);
 				mfd->endingframe = GetDlgItemInt(hdlg, IDC_ENDING_FRAME, &mfd->endingframe, FALSE);
                EndDialog(hdlg, 0);
                return TRUE;
 			case IDHELP:
				{
				char prog[256];
				char path[256];
				LPTSTR ptr;
				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\Pan.txt");
				OutputDebugString(path);
				OutputDebugString("\n");
				strcpy(prog, "Notepad ");
				strcat(prog, path);
				WinExec(prog, SW_SHOW);
				return TRUE;
				}
           case IDCANCEL:
                EndDialog(hdlg, 1);
                return FALSE;
            }
            break;
    }

    return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
    return DialogBoxParam(fa->filter->module->hInstModule,
            MAKEINTRESOURCE(IDD_FILTER_FIELD_SHIFT), hwnd,
            (DLGPROC)ConfigDlgProc, (LPARAM)fa->filter_data);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str)
{

}
