/*
    Windowed Histogram Equalization Filter for VirtualDub -- automatic
    contrast adjustment by means of histogram equalization.
    Copyright (C) 2001 Donald A. Graft

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	The author can be contacted at:
	Donald Graft
	neuron2@home.com.
*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <crtdbg.h>
#include <math.h>
#include <stdlib.h>
#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"
#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int  RunProc(const FilterActivation *fa, const FilterFunctions *ff);
int  StartProc(FilterActivation *fa, const FilterFunctions *ff);
int  EndProc(FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int  InitProc(FilterActivation *fa, const FilterFunctions *ff);
int  ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

#define SIZEOF_HISTOGRAM 1024	// 256 * sizeof(unsigned long)

///////////////////////////////////////////////////////////////////////////

typedef struct MyFilterData
{
	IFilterPreview		*ifp;
	int					strength, strengthInv;
	int					intensity;
	int					xWindowSize, yWindowSize, WindowArea;
	int					xWindowPitch, yWindowPitch;
	char				*Histograms;		// Input histograms
	char				*LUTs;			// Lookup tables derived from Histograms[]
	int					numXHistos, numYHistos;
} MyFilterData;

bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	_snprintf(buf, buflen, "Config(%d, %d, %d, %d)",
			  mfd->strength, mfd->intensity, mfd->xWindowSize, mfd->yWindowSize);
	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc)
{
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->strength		= argv[0].asInt();
	mfd->intensity		= argv[1].asInt();
	mfd->xWindowSize	= argv[2].asInt();
	mfd->yWindowSize	= argv[3].asInt();
}

ScriptFunctionDef func_defs[] =
{
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0iiii" },
	{ NULL },
};

CScriptObject script_obj =
{
	NULL, func_defs
};

struct FilterDefinition filterDef_whe =
{
	NULL, NULL, NULL,		// next, prev, module
	"windowed histogram equalize (1.0)",	// name
	"Perform windowed color histogram equalization.", // desc
	"Donald Graft", 		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size
	InitProc,				// initProc
	NULL,					// deinitProc
	RunProc,				// runProc
	ParamProc,				// paramProc
	ConfigProc, 			// configProc
	StringProc, 			// stringProc
	StartProc,				// startProc
	EndProc,				// endProc
	&script_obj,			// script_obj
	FssProc,				// fssProc
};

long ParamProc(FilterActivation *fa, const FilterFunctions *ff)
{
	/* Implement as in-place filter. */
	fa->dst.offset = fa->src.offset;
    return 0;
}

int StartProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	/* Set window sizes and allocate data space. */
	mfd->numXHistos = fa->src.w / mfd->xWindowSize;
	mfd->numYHistos = fa->src.h / mfd->yWindowSize;
	mfd->Histograms = (char *) malloc(mfd->numXHistos * mfd->numYHistos *
						                         SIZEOF_HISTOGRAM);
	mfd->LUTs = (char *) malloc(mfd->numXHistos * mfd->numYHistos *
						                         SIZEOF_HISTOGRAM);
	/* Tricks to avoid multiplies in the loops. */
	mfd->strengthInv = 255 - mfd->strength;
	mfd->xWindowPitch = mfd->xWindowSize * sizeof(unsigned long);
	mfd->yWindowPitch = mfd->yWindowSize * fa->src.pitch;
	mfd->WindowArea = mfd->xWindowSize * mfd->yWindowSize;
	return 0;
}

int EndProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	/* Free the data space. */
	if (mfd->Histograms) free(mfd->Histograms);
	if (mfd->LUTs) free(mfd->LUTs);
	return 0;
}

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	const PixDim width = fa->src.w;
	const PixDim height = fa->src.h;
	Pixel32 *src, *src0;
	Pixel32 *p, *p0;
	int x, y, z, yIndex, m, n;
	unsigned int r, g, b, luma;
	unsigned long *pHisto;
	unsigned long *pLUT;
	int SubX, SubY, YU, YB, XL, XR;
	unsigned long *pLU, *pRU, *pLB, *pRB;

	/* Calculate and store the pixel luminances and calculate the per-window
	   histograms based on the luminances. */
	src = src0 = fa->src.data;
	memset(mfd->Histograms, 0, mfd->numXHistos * mfd->numYHistos * SIZEOF_HISTOGRAM);
	for (y = 0; y < mfd->numYHistos; y++)
	{
		for (x = 0; x < mfd->numXHistos; x++)
		{
			pHisto = (unsigned long *) (mfd->Histograms + SIZEOF_HISTOGRAM *
					 (y * mfd->numXHistos + x));
			p = src;
			for (n = 0; n < mfd->yWindowSize; n++)
			{
				for (m = 0; m < mfd->xWindowSize; m++)
				{
					/* Calculate and store the pixel luminance in the
					   alpha channel. */
					r = (p[m] >> 16) & 0xff;
					g = (p[m] >> 8) & 0xff;
					b = p[m] & 0xff;
					luma = (55 * r + 182 * g + 19 * b) >> 8;
					p[m] &= 0xffffff;
					p[m] |= luma << 24;
					/* Accumulate the pixel into the appropriate window's
					   histogram. */
					pHisto[luma]++;
				}
				p = (Pixel32 *)((char *) p + fa->src.pitch);
			}
			src = (Pixel32 *)((char *) src + mfd->xWindowPitch);
		}
		src = src0 = (Pixel32 *)((char *) src0 + mfd->yWindowPitch);
	}

	/* Make the look-up table for each window. */
    for (y = 0; y < mfd->numYHistos; y++)
	{
		for (x = 0; x < mfd->numXHistos; x++)
		{
			pHisto = (unsigned long *) (mfd->Histograms + SIZEOF_HISTOGRAM *
					 (y * mfd->numXHistos + x));
			pLUT =   (unsigned long *) (mfd->LUTs + SIZEOF_HISTOGRAM *
					 (y * mfd->numXHistos + x));
			/* Calculate the lookup table. */
			pLUT[0] = pHisto[0];
			/* Accumulate. */
			for (z = 1; z < 256; z++)
			{
				pLUT[z] = pLUT[z-1] + pHisto[z];
			}
			/* Normalize. */
			for (z = 0; z < 256; z++)
			{
				pLUT[z] = (pLUT[z] * mfd->intensity) / mfd->WindowArea;
			}
			/* Adjust the LUT based on the selected strength. This is an alpha
			   mix of the calculated LUT and a linear LUT with gain 1. */
			for (z = 0; z < 256; z++)
			{
				pLUT[z] = (mfd->strength * pLUT[z] + mfd->strengthInv * z) / 255;
			}
		}
	}

	/* Output the equalized frame. LUT interpolation is modeled on
	   public-domain code by Karel Zuiderveld. */
	src = fa->src.data;
	yIndex = 0;
	for (y = 0; y <= mfd->numYHistos; y++)
	{
		if (y == 0)
		{
			SubY = mfd->yWindowSize >> 1;
			YU = 0;
			YB = 0;
		}
		else
		{
			if (y == mfd->numYHistos)
			{
				SubY = mfd->yWindowSize >> 1;
				YU = y - 1;
				YB = YU;
			}
			else
			{
				SubY = mfd->yWindowSize;
				YU = y - 1;
				YB = y;
			}
		}
		for (x = 0; x <= mfd->numXHistos; x++)
		{
			if (x == 0)
			{
				SubX = mfd->xWindowSize >> 1;
				XL = 0;
				XR = 0;
			}
			else
			{
				if (x == mfd->numXHistos)
				{
					SubX = mfd->xWindowSize >> 1;
					XL = x - 1;
					XR = XL;
				}
				else
				{
					SubX = mfd->xWindowSize;
					XL = x - 1;
					XR = x;
				}
			}
			pLU = (unsigned long *)
				  (mfd->LUTs + SIZEOF_HISTOGRAM * (YU * mfd->numXHistos + XL));
			pRU = (unsigned long *)
				  (mfd->LUTs + SIZEOF_HISTOGRAM * (YU * mfd->numXHistos + XR));
			pLB = (unsigned long *)
				  (mfd->LUTs + SIZEOF_HISTOGRAM * (YB * mfd->numXHistos + XL));
			pRB = (unsigned long *)
				  (mfd->LUTs + SIZEOF_HISTOGRAM * (YB * mfd->numXHistos + XR));
			
			/* Interpolate the LUTs and calculate the output pixel. */
			{
				int x, y, xInv, yInv;
				int luma, lut;
				int r, g, b, m;

				p = p0 = src;
				for (y = 0, yInv = SubY; y < SubY; y++, yInv--)
				{
					for (x = 0, xInv = SubX; x < SubX; x++, xInv--)
					{
						luma = *p >> 24;
						if (luma == 0) *p = 0;
						else
						{
							lut = (yInv * (xInv * pLU[luma] + x * pRU[luma]) +
								   y * (xInv * pLB[luma] + x * pRB[luma])) / (SubX * SubY);
							r = (*p >> 16) & 0xff;
							g = (*p >> 8) & 0xff;
							b = *p & 0xff;
							/* Clip the output RGB value if necessary. */
							if (((m = max(r, max(g, b))) * lut) / luma > 255)
							{
								r = (r * 255) / m; 
								g = (g * 255) / m; 
								b = (b * 255) / m; 
							}
							else
							{
								r = (r * lut) / luma;
								g = (g * lut) / luma;
								b = (b * lut) / luma;
							}
							*p = (r << 16) | (g << 8) | (b);
						}
						p++;
					}
					p = p0 = (Pixel32 *)((char *) p0 + fa->src.pitch);
				}
			}
			src = (Pixel32 *)((char *) src + SubX * sizeof(Pixel32)); 
		}
		yIndex += SubY;
		src	= (Pixel *)((char *) fa->src.data + yIndex * fa->src.pitch);
	}
	return 0;
}

extern "C" int __declspec(dllexport) __cdecl
VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);

extern "C" void __declspec(dllexport) __cdecl
VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_whe;

int __declspec(dllexport) __cdecl
VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat)
{
	if (!(fd_whe = ff->addFilter(fm, &filterDef_whe, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}

void __declspec(dllexport) __cdecl
VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff)
{
	ff->removeFilter(fd_whe);
}

int InitProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->strength = 200;
	mfd->intensity = 210;
	mfd->xWindowSize = 64;
	mfd->yWindowSize = 64;

	return 0;
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg) {
		case WM_INITDIALOG:
			SetWindowLong(hdlg, DWL_USER, lParam);
			mfd = (MyFilterData *)lParam;

			SendMessage(GetDlgItem(hdlg, IDC_STRENGTH), TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 255));
			SendMessage(GetDlgItem(hdlg, IDC_STRENGTH), TBM_SETTICFREQ, 10 , 0);
			SendMessage(GetDlgItem(hdlg, IDC_STRENGTH), TBM_SETPOS, (WPARAM)TRUE, mfd->strength);
			SendMessage(GetDlgItem(hdlg, IDC_INTENSITY), TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 255));
			SendMessage(GetDlgItem(hdlg, IDC_INTENSITY), TBM_SETTICFREQ, 10 , 0);
			SendMessage(GetDlgItem(hdlg, IDC_INTENSITY), TBM_SETPOS, (WPARAM)TRUE, mfd->intensity);
			SetDlgItemInt(hdlg, IDC_STRENGTHVAL, mfd->strength, FALSE);
			SetDlgItemInt(hdlg, IDC_INTENSITYVAL, mfd->intensity, FALSE);
			SetDlgItemInt(hdlg, IDC_XSIZE, mfd->xWindowSize, FALSE);
			SetDlgItemInt(hdlg, IDC_YSIZE, mfd->yWindowSize, FALSE);
			mfd->ifp->InitButton(GetDlgItem(hdlg, IDPREVIEW));

			return TRUE;

		case WM_COMMAND:
			switch(LOWORD(wParam)) {
			case IDPREVIEW:
				mfd->ifp->Toggle(hdlg);
				break;
			case IDOK:
				EndDialog(hdlg, 0);
				return TRUE;
 			case IDHELP:
				{
				char prog[256];
				char path[256];
				LPTSTR ptr;

				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\Winhisto.html");
				ShellExecute(hdlg, "open", path, NULL, NULL, SW_SHOWNORMAL);
				return TRUE;
				}
			case IDCANCEL:
				EndDialog(hdlg, 1);
				return TRUE;
			case IDC_XSIZE:
				if (HIWORD(wParam) == EN_UPDATE)
				{
					long num;
					BOOL success;

					num = GetDlgItemInt(hdlg, IDC_XSIZE, &success, FALSE);
					if (!success || num < 10)
					{
						return TRUE;
					}
					mfd->xWindowSize = num;
					mfd->xWindowPitch = mfd->xWindowSize * sizeof(unsigned long);
					mfd->ifp->UndoSystem();
					mfd->ifp->RedoSystem();
				}
				return TRUE;
			case IDC_YSIZE:
				if (HIWORD(wParam) == EN_UPDATE)
				{
					long num;
					BOOL success;

					num = GetDlgItemInt(hdlg, IDC_YSIZE, &success, FALSE);
					if (!success || num < 10)
					{
						return TRUE;
					}
					mfd->yWindowSize = num;
					mfd->yWindowPitch = mfd->yWindowSize * sizeof(unsigned long);
					mfd->ifp->UndoSystem();
					mfd->ifp->RedoSystem();
				}
				return TRUE;
		}
		case WM_HSCROLL:
			{
				int strength, intensity;
				strength = SendMessage(GetDlgItem(hdlg, IDC_STRENGTH), TBM_GETPOS, 0, 0);
				intensity = SendMessage(GetDlgItem(hdlg, IDC_INTENSITY), TBM_GETPOS, 0, 0);
				if (strength != mfd->strength || intensity != mfd->intensity)
				{
					mfd->strength = strength;
					mfd->strengthInv = 255 - mfd->strength;
					mfd->intensity = intensity;
					SetDlgItemInt(hdlg, IDC_STRENGTHVAL, strength, FALSE);
					SetDlgItemInt(hdlg, IDC_INTENSITYVAL, intensity, FALSE);
					mfd->ifp->RedoFrame();
				}
				break;
			}
	}
	return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	MyFilterData mfd_old = *mfd;
	int ret;

	mfd->ifp = fa->ifp;

	if (DialogBoxParam(fa->filter->module->hInstModule, MAKEINTRESOURCE(IDD_FILTER),
		hwnd, ConfigDlgProc, (LPARAM)mfd))
	{
		*mfd = mfd_old;
		ret = TRUE;
	}
    else
	{
		ret = FALSE;
	}
	return(ret);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	sprintf(str, " (str %d, inten %d, x %d y %d)", mfd->strength, mfd->intensity,
		mfd->xWindowSize, mfd->yWindowSize);
}
