/*//////////////////////////////////////////////////////////////////////////

	flaXen's Boundary Tool for VirtualDub
	(c) Copyright 2000 Dan Flower (flaXen)

	Clamps the intensity of a pixel to within a specific range.

*//////////////////////////////////////////////////////////////////////////

#include <windows.h>
#include <stdio.h>
#include <math.h>

#include "resource.h"
#include "filter.h"
#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

typedef struct MyFilterData {
	int		boLow, boHigh;
} MyFilterData;

///////////////////////////////////////////////////////////////////////////

int		fxBoundRunProc(const FilterActivation *fa, const FilterFunctions *ff);
int		fxBoundStartProc(FilterActivation *fa, const FilterFunctions *ff);
int		fxBoundEndProc(FilterActivation *fa, const FilterFunctions *ff); 
int		fxBoundInitProc(FilterActivation *fa, const FilterFunctions *ff);
int		fxBoundConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void	fxBoundStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *buf);
void	fxBoundScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool	fxBoundFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

ScriptFunctionDef fxBound_func_defs[]={
    { (ScriptFunctionPtr)fxBoundScriptConfig, "Config", "0ii" },
    { NULL },
};

CScriptObject fxBoundobj={
    NULL, fxBound_func_defs
};

struct FilterDefinition filterDef_fxBound = {
	NULL, NULL, NULL,		// next, prev, module

	"Boundary",				// name
	"flaXen's Boundary Tool",	// desc
	"flaXen",				// maker

	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size
	fxBoundInitProc,		// initProc
	NULL,					// deinitProc
	fxBoundRunProc,			// runProc
	NULL,					// paramProc
	fxBoundConfigProc,		// configProc
	fxBoundStringProc,		// stringProc
	fxBoundStartProc,		// startProc
	fxBoundEndProc,			// endProc
	&fxBoundobj,			// script_obj
	fxBoundFssProc,			// fssProc
};

///////////////////////////////////////////////////////////////////////////

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_fxBound;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if (!(fd_fxBound = ff->addFilter(fm, &filterDef_fxBound, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver    = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;
	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_fxBound);
}

///////////////////////////////////////////////////////////////////////////

int fxBoundRunProc(const FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData	*mfd = (MyFilterData *)fa->filter_data;
	Pixel32		*isrc, *idst, *src, *dst;
	long		x, y, tr, tg, tb, c, MMX;
	_int64		qlow, qhigh;
	const long	swid = fa->src.w, shei = fa->src.h;
	const long	spit = fa->src.pitch >> 2;
	const long	ilow = mfd->boLow;
	const long	ihigh = mfd->boHigh;

	isrc = (Pixel32 *)fa->src.data;
	idst = (Pixel32 *)fa->dst.data;

	MMX = ff->isMMXEnabled();
	if (swid & 1) MMX = 0;
	
	if (MMX) {
		__asm {
			mov		eax, [ilow]
			mov		ah, al
			mov		ebx, eax
			shl		eax, 16
			or		eax, ebx
			mov		[dword ptr qlow], eax
			mov		[dword ptr qlow + 4], eax
			movq	mm1, [qlow]

			mov		eax, 255
			sub		eax, [ihigh]
			mov		ah, al
			mov		ebx, eax
			shl		eax, 16
			or		eax, ebx
			mov		[dword ptr qhigh], eax
			mov		[dword ptr qhigh + 4], eax
			movq	mm2, [qhigh]

			movq	mm3, mm1
			paddusb	mm3, mm2
		}
		for (y = 0; y < shei; y++) {
			src = isrc + y * spit;
			dst = idst + y * spit;

			__asm {
				mov		ecx, [swid]
				shr		ecx, 1
				mov		eax, [src]
				mov		ebx, [dst]
bo10:			movq	mm0, [eax]
				psubusb	mm0, mm1
				paddusb	mm0, mm3
				psubusb	mm0, mm2
				movq	[ebx], mm0
				add		eax, 8
				add		ebx, 8
				dec		ecx
				jnz		bo10
			}
		}
		__asm emms;
	} else {
		for (y = 0; y < shei; y++) {
			src = isrc + y * spit;
			dst = idst + y * spit;
			for (x = 0; x < swid; x++) {
				c = *src++;
				tr = (c >> 16) & 255; tg = (c >> 8) & 255; tb = c & 255;

				if (tr < ilow) tr = ilow;
				else if (tr > ihigh) tr = ihigh;
				if (tg < ilow) tg = ilow;
				else if (tg > ihigh) tg = ihigh;
				if (tb < ilow) tb = ilow;
				else if (tb > ihigh) tb = ihigh;

				*dst++ = (tr << 16) | (tg << 8) | tb;
			}
		}
	}
	return 0;
}

///////////////////////////////////////////////////////////////////////////

int fxBoundStartProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

int fxBoundEndProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

int fxBoundInitProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->boLow = 0;
	mfd->boHigh = 240;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

BOOL CALLBACK fxBoundConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg) {
	case WM_INITDIALOG:
		SetWindowLong(hdlg, DWL_USER, lParam);
		mfd = (MyFilterData *)lParam;
		SetDlgItemInt(hdlg, IDC_BO_LOW, mfd->boLow, FALSE);
		SetDlgItemInt(hdlg, IDC_BO_HIGH, mfd->boHigh, FALSE);
		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam)) {
		case IDOK:
			mfd->boLow = GetDlgItemInt(hdlg, IDC_BO_LOW, &mfd->boLow, FALSE);
			mfd->boHigh = GetDlgItemInt(hdlg, IDC_BO_HIGH, &mfd->boHigh, FALSE);
			EndDialog(hdlg, 0);

			if (mfd->boLow > 255) mfd->boLow = 255;
			if (mfd->boHigh > 255) mfd->boHigh = 255;
			return TRUE;
		case IDCANCEL:
			EndDialog(hdlg, 1);
			return FALSE;
		}
		break;
	}

	return FALSE;
}

///////////////////////////////////////////////////////////////////////////

int fxBoundConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd) {
	return DialogBoxParam(fa->filter->module->hInstModule,
	       MAKEINTRESOURCE(IDD_FXBOUND_CONFIG), hwnd,
		   fxBoundConfigDlgProc, (LPARAM)fa->filter_data);
}

///////////////////////////////////////////////////////////////////////////

void fxBoundStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	sprintf(str, " Low: %d  High: %d", mfd->boLow, mfd->boHigh);
}

///////////////////////////////////////////////////////////////////////////

void fxBoundScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
    FilterActivation *fa = (FilterActivation *)lpVoid;
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->boLow = argv[0].asInt();
	mfd->boHigh = argv[1].asInt();
}

///////////////////////////////////////////////////////////////////////////

bool fxBoundFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

    _snprintf(buf, buflen, "Config(%d, %d)", mfd->boLow, mfd->boHigh);

    return true;
}