//	VirtualDub resize engine
//	Copyright (C) 1998-2000 Avery Lee
//
//   Size calculator enhancement
//   Copyright (C) 2000-2001 Donald Graft
//
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#define VDEXT_VIDEO_FILTER
#define VDEXT_MAIN

#include <windows.h>
#include <commctrl.h>
#include <crtdbg.h>
#include <assert.h>
#include <stdio.h>
#include <math.h>

#include "ScriptInterpreter.h"
#include "ScriptValue.h"
#include "ScriptError.h"

#include "misc.h"
#include "cpuaccel.h"
#include "resource.h"
#include "gui.h"
#include "filter.h"
#include "resample.h"
#include "vbitmap.h"

///////////////////////

enum {
	FILTER_NONE				= 0,
	FILTER_BILINEAR			= 1,
	FILTER_BICUBIC			= 2,
	FILTER_TABLEBILINEAR	= 3,
	FILTER_TABLEBICUBIC		= 4
};

static char *filter_names[]={
	"Nearest neighbor",
	"Bilinear",
	"Bicubic",
	"Precise bilinear",
	"Precise bicubic",
};

#define FIT_WIDTH               0
#define FIT_HEIGHT              1
#define FIT_WIDTH_HEIGHT        2
#define FIT_WIDTH_HEIGHT_LETTER 3
#define FIT_PERCENT             4

#define UP          0
#define DOWN		1
#define NEAREST     2

#define PAD_WIDTH   0
#define PAD_HEIGHT  1
#define PAD_ADJUST  2

typedef struct MyFilterData {
	long new_x, new_y, new_xf, new_yf;
	int filter_mode;
	COLORREF	rgbColor;

	HBRUSH		hbrColor;
	IFilterPreview *ifp;

	Resampler *resampler;

	bool	fLetterbox;
	bool	fInterlaced;

	bool enable;
	long in_width, in_height;
	long num, denom;
	int  fit_mode;
	long fitwidth, fitheight, fitwidthheight, fitwidthheightletter, percent;
	bool no_enlarge, no_reduce;
	bool fixw, fixh;
	long fixwval, fixhval;
	int  round_mode;
	int  pad_mode;
} MyFilterData;

////////////////////

int revcolor(int c) {
	return ((c>>16)&0xff) | (c&0xff00) | ((c&0xff)<<16);
}

////////////////////

static int resize_run(const FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	long dstw = mfd->new_x;
	long dsth = mfd->new_y;
	long h_margin=0, w_margin=0;
	Pixel *dst, *src;

	dst = fa->dst.data;
	src = fa->src.data;

	// Draw letterbox bound

	if (mfd->fLetterbox) {
		Pixel *dst2 = dst, *dst3;
		Pixel fill = revcolor(mfd->rgbColor);

		long w1, w2, w, h;

		dsth = (dsth+1)&~1;

		w_margin = fa->dst.w - dstw;
		h_margin = fa->dst.h - dsth;

		if (w_margin < 0)
			w_margin = 0;

		if (h_margin < 0)
			h_margin = 0;

		h = h_margin/2;
		if (h>0) do {
			dst3  = dst2;
			w = fa->dst.w;
			do {
				*dst3++ = fill;
			} while(--w);

			dst2 = (Pixel32 *)((char *)dst2 + fa->dst.pitch);
		} while(--h);

		w1 = w_margin/2;
		w2 = w_margin - w_margin/2;

		h = mfd->new_y;
		do {
			dst3 = dst2;

			// fill left

			w = w1;
			if (w) do {
				*dst3++ = fill;
			} while(--w);

			// skip center

			dst3 += mfd->new_x;

			// fill right

			w = w2;
			if (w) do {
				*dst3++ = fill;
			} while(--w);


			dst2 = (Pixel32 *)((char *)dst2 + fa->dst.pitch);
		} while(--h);

		h = h_margin - h_margin/2;
		if (h>0) do {
			dst3  = dst2;
			w = fa->dst.w;
			do {
				*dst3++ = fill;
			} while(--w);

			dst2 = (Pixel32 *)((char *)dst2 + fa->dst.pitch);
		} while(--h);

		// offset resampled rectangle

		dst = (Pixel *)((char *)dst + fa->dst.pitch * (h_margin/2)) + (w_margin/2);
	}

#define USE_INTERLACED
#ifdef USE_INTERLACED
	if (mfd->fInterlaced) {
		VBitmap vbHalfSrc, vbHalfDst;

		vbHalfSrc = fa->src;
		vbHalfSrc.modulo += vbHalfSrc.pitch;
		vbHalfSrc.pitch *= 2;
		vbHalfSrc.data = fa->src.Address32i(0, fa->src.h&1);
		vbHalfSrc.h >>= 1;

		vbHalfDst = fa->dst;
		vbHalfDst.modulo += vbHalfDst.pitch;
		vbHalfDst.pitch *= 2;
		vbHalfDst.h >>= 1;

		double dy = 0.25 * (1.0 - (double)vbHalfSrc.h / (double)vbHalfDst.h);

		mfd->resampler->Process(&vbHalfDst, w_margin/2, h_margin/4, &vbHalfSrc, 0, -dy, false);

		vbHalfSrc.data = fa->src.Address32i(0, 1+(fa->src.h&1));
		vbHalfDst.data = fa->dst.Address32i(0, 1);

		mfd->resampler->Process(&vbHalfDst, w_margin/2, h_margin/4, &vbHalfSrc, 0, +dy, false);
	} else
#endif
		mfd->resampler->Process(&fa->dst, w_margin/2, h_margin/2, &fa->src, 0, 0, false);

	return 0;
}

static long resize_param(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	void update(MyFilterData *mfd);

	if (!mfd) return 0;
	mfd->in_width = fa->src.w;
	mfd->in_height = fa->src.h;
	update(mfd);

	if (mfd->fLetterbox) {
		fa->dst.w		= max(mfd->new_x, mfd->new_xf);
		fa->dst.h		= max(mfd->new_y, mfd->new_yf);
	} else {
		fa->dst.w		= mfd->new_x;
		fa->dst.h		= mfd->new_y;
	}

#ifdef USE_INTERLACED
	if (mfd->fInterlaced)
		fa->dst.h = (fa->dst.h+1)&~1;
#endif

	fa->dst.AlignTo8();

	return FILTERPARAM_SWAP_BUFFERS;
}

static COLORREF g_crCustomColors[16];

bool guiChooseColor(HWND hwnd, COLORREF& rgbOld) {
	CHOOSECOLOR cc;                 // common dialog box structure 

	// Initialize CHOOSECOLOR
	memset(&cc, 0, sizeof(CHOOSECOLOR));
	cc.lStructSize	= sizeof(CHOOSECOLOR);
	cc.hwndOwner	= hwnd;
	cc.lpCustColors	= (LPDWORD)g_crCustomColors;
	cc.rgbResult	= rgbOld;
	cc.Flags		= CC_FULLOPEN | CC_RGBINIT;

	if (ChooseColor(&cc)==TRUE) {;
		rgbOld = cc.rgbResult;
		return true;
	}

	return false;
}

void update(MyFilterData *mfd)
{
	long w, fw, cw;
	long h, fh, ch;
	long frameh, framew;

	if (mfd->enable == FALSE) return;
	
	/* First adjust the aspect ratio. */
	w = (mfd->in_width * mfd->num) / mfd->denom;
	if (w == 0) w = 1;
	h = mfd->in_height;

	/* Now do the fitting. */
	if (mfd->fit_mode == FIT_WIDTH)
	{
		fw = mfd->fitwidth;
		fh = (fw * h) / w;
		if ((mfd->no_enlarge && (fw > w || fh > h)) ||
		    (mfd->no_reduce  && (fw < w || fh < h)))
		{
			fw = w;
			fh = h;
		}
	}
	else if (mfd->fit_mode == FIT_HEIGHT)
	{
		fh = mfd->fitheight;
		fw = (fh * w) / h;
		if ((mfd->no_enlarge && (fw > w || fh > h)) ||
		    (mfd->no_reduce  && (fw < w || fh < h)))
		{
			fw = w;
			fh = h;
		}
	}
	else if (mfd->fit_mode == FIT_WIDTH_HEIGHT)
	{
		fh = mfd->fitheight;
		fw = (fh * w) / h;
		if (fw > mfd->fitwidth)
		{
			fw = mfd->fitwidth;
			fh = (fw * h) / w;
		}
		if ((mfd->no_enlarge && (fw > w || fh > h)) ||
		    (mfd->no_reduce  && (fw < w || fh < h)))
		{
			fw = w;
			fh = h;
		}
	}
	else if (mfd->fit_mode == FIT_WIDTH_HEIGHT_LETTER)
	{
		fh = mfd->fitheight;
		fw = (fh * w) / h;
		if (fw > mfd->fitwidth)
		{
			fw = mfd->fitwidth;
			fh = (fw * h) / w;
		}
		if ((mfd->no_enlarge && (fw > w || fh > h)) ||
		    (mfd->no_reduce  && (fw < w || fh < h)))
		{
			fw = w;
			fh = h;
		}
		mfd->new_x = fw;
		mfd->new_y = fh;
		mfd->new_xf = mfd->fitwidth;
		mfd->new_yf = mfd->fitheight;
		if (mfd->new_x == 0) mfd->new_x = 1;
		if (mfd->new_y == 0) mfd->new_y = 1;
		if (mfd->new_xf != mfd->new_x || mfd->new_yf != mfd->new_y)
		{
			mfd->fLetterbox = TRUE;
		}
		else
		{
			mfd->fLetterbox = FALSE;
		}
		return;
	}
	else
	{
		fh = (h * mfd->percent) / 100;
		fw = (w * mfd->percent) / 100;
	}
	if (fh == 0) fh = 1;
	if (fw == 0) fw = 1;

	/* Finally, do the constraining to multiples. */
	frameh = ch = fh;
	framew = cw = fw;
	if ((mfd->fixw == true) && (mfd->fixh == false) && (fw % mfd->fixwval))
	{
		if (mfd->round_mode == UP)
		{
			cw = fw + mfd->fixwval - (fw % mfd->fixwval);
			ch = (cw * fh) / fw;
		}
		else if (mfd->round_mode == DOWN)
		{
			cw = fw - (fw % mfd->fixwval);
			ch = (cw * fh) / fw;
		}
		else
		{
			if ((fw % mfd->fixwval) > (mfd->fixwval / 2))
			{
				cw = fw + mfd->fixwval - (fw % mfd->fixwval);
				ch = (cw * fh) / fw;
			}
			else
			{
				cw = fw - (fw % mfd->fixwval);
				ch = (cw * fh) / fw;
			}
		}
		framew = cw;
		frameh = ch;
	}
	else if ((mfd->fixh == true) && (mfd->fixw == false) && (fh % mfd->fixhval))
	{
		if (mfd->round_mode == UP)
		{
			ch = fh + mfd->fixhval - (fh % mfd->fixhval);
			cw = (ch * fw) / fh;
		}
		else if (mfd->round_mode == DOWN)
		{
			ch = fh - (fh % mfd->fixhval);
			cw = (ch * fw) / fh;
		}
		else
		{
			if ((fh % mfd->fixhval) > (mfd->fixhval / 2))
			{
				ch = fh + mfd->fixhval - (fh % mfd->fixhval);
				cw = (ch * fw) / fh;
			}
			else
			{
				ch = fh - (fh % mfd->fixhval);
				cw = (ch * fw) / fh;
			}
		}
		framew = cw;
		frameh = ch;
	}
	else if ((mfd->fixh == true) && (mfd->fixw == true))
	{
		if (mfd->pad_mode == PAD_ADJUST)
		{
			int mode = mfd->round_mode;

			if (mode == NEAREST)
			{
				if (fh % mfd->fixhval)
				{
					if ((fh % mfd->fixhval) > (mfd->fixhval / 2))
						ch = fh + mfd->fixhval - (fh % mfd->fixhval);
					else
						ch = fh - (fh % mfd->fixhval);
				}
				if (fw % mfd->fixwval)
				{
					if ((fw % mfd->fixwval) > (mfd->fixwval / 2))
						cw = fw + mfd->fixwval - (fw % mfd->fixwval);
					else
						cw = fw - (fw % mfd->fixwval);
				}
			}	
			else if (mode == UP)
			{
				if (fh % mfd->fixhval)
					ch = fh + mfd->fixhval - (fh % mfd->fixhval);
				if (fw % mfd->fixwval)
					cw = fw + mfd->fixwval - (fw % mfd->fixwval);
			}
			else if (mode == DOWN)
			{
				if (fh % mfd->fixhval)
					ch = fh - (fh % mfd->fixhval);
				if (fw % mfd->fixwval)
					cw = fw - (fw % mfd->fixwval);
			}
			framew = cw;
			frameh = ch;
		}
		else if (mfd->pad_mode == PAD_WIDTH)
		{
			if (fh % mfd->fixhval)
			{
				ch = fh + mfd->fixhval - (fh % mfd->fixhval);
			}
			cw = (ch * fw) / fh;
			if (cw % mfd->fixwval)
				framew = cw + mfd->fixwval - (cw % mfd->fixwval);
			else
				framew = cw;
			frameh = ch;
		}
		else
		{
			if (fw % mfd->fixwval)
			{
				cw = fw + mfd->fixwval - (fw % mfd->fixwval);
			}
			ch = (cw * fh) / fw;
			if (ch % mfd->fixhval)
				frameh = ch + mfd->fixhval - (ch % mfd->fixhval);
			else
				frameh = ch;
			framew = cw;
		}
	}
	mfd->new_x = cw;
	mfd->new_y = ch;
	mfd->new_xf = framew;
	mfd->new_yf = frameh;
	if (mfd->new_x == 0) mfd->new_x = 1;
	if (mfd->new_y == 0) mfd->new_y = 1;
	if (mfd->new_xf != mfd->new_x || mfd->new_yf != mfd->new_y)
	{
		mfd->fLetterbox = TRUE;
	}
	else
	{
		mfd->fLetterbox = FALSE;
	}
}

void redisplay(HWND hDlg, MyFilterData *mfd)
{
	if (mfd->enable == FALSE) return;
	mfd->ifp->UndoSystem();
	SetDlgItemInt(hDlg, IDC_WIDTH, mfd->new_x, FALSE);
	SetDlgItemInt(hDlg, IDC_HEIGHT, mfd->new_y, FALSE);
	SetDlgItemInt(hDlg, IDC_FRAMEWIDTH, mfd->new_xf, FALSE);
	SetDlgItemInt(hDlg, IDC_FRAMEHEIGHT, mfd->new_yf, FALSE);
	if (mfd->new_xf != mfd->new_x || mfd->new_yf != mfd->new_y)
	{
		CheckDlgButton(hDlg, IDC_LETTERBOX, BST_CHECKED);
		EnableWindow(GetDlgItem(hDlg, IDC_STATIC_FILLCOLOR), TRUE);
		EnableWindow(GetDlgItem(hDlg, IDC_COLOR), TRUE);
		EnableWindow(GetDlgItem(hDlg, IDC_PICKCOLOR), TRUE);
		if (mfd->enable == FALSE)
		{
			EnableWindow(GetDlgItem(hDlg, IDC_FRAMEWIDTH), TRUE);
			EnableWindow(GetDlgItem(hDlg, IDC_FRAMEHEIGHT), TRUE);
		}
	}
	else
	{
		CheckDlgButton(hDlg, IDC_LETTERBOX, BST_UNCHECKED);
		EnableWindow(GetDlgItem(hDlg, IDC_STATIC_FILLCOLOR), FALSE);
		EnableWindow(GetDlgItem(hDlg, IDC_COLOR), FALSE);
		EnableWindow(GetDlgItem(hDlg, IDC_PICKCOLOR), FALSE);
		EnableWindow(GetDlgItem(hDlg, IDC_FRAMEWIDTH), FALSE);
		EnableWindow(GetDlgItem(hDlg, IDC_FRAMEHEIGHT), FALSE);
	}

	mfd->ifp->RedoSystem();
}

static BOOL APIENTRY resizeDlgProc( HWND hDlg, UINT message, UINT wParam, LONG lParam) {
	MyFilterData *mfd = (struct MyFilterData *)GetWindowLong(hDlg, DWL_USER);

    switch (message)
    {
        case WM_INITDIALOG:
			{
				HWND hwndItem;
				int i;

				mfd = (MyFilterData *)lParam;

				SetDlgItemInt(hDlg, IDC_WIDTH, mfd->new_x, FALSE);
				SetDlgItemInt(hDlg, IDC_HEIGHT, mfd->new_y, FALSE);
				SetDlgItemInt(hDlg, IDC_FRAMEWIDTH, mfd->new_xf, FALSE);
				SetDlgItemInt(hDlg, IDC_FRAMEHEIGHT, mfd->new_yf, FALSE);

				SetDlgItemInt(hDlg, IDC_INWIDTH, mfd->in_width, FALSE);
				SetDlgItemInt(hDlg, IDC_INHEIGHT, mfd->in_height, FALSE);
				SetDlgItemInt(hDlg, IDC_NUM, mfd->num, FALSE);
				SetDlgItemInt(hDlg, IDC_DENOM, mfd->denom, FALSE);
				SetDlgItemInt(hDlg, IDC_FITWIDTH, mfd->fitwidth, FALSE);
				SetDlgItemInt(hDlg, IDC_FITHEIGHT, mfd->fitheight, FALSE);
				SetDlgItemInt(hDlg, IDC_PERCENT, mfd->percent, FALSE);
				CheckDlgButton(hDlg, IDC_FITWIDTHON, mfd->fit_mode == FIT_WIDTH ? BST_CHECKED : BST_UNCHECKED);
				CheckDlgButton(hDlg, IDC_FITHEIGHTON, mfd->fit_mode == FIT_HEIGHT ? BST_CHECKED : BST_UNCHECKED);
				CheckDlgButton(hDlg, IDC_FITWH, mfd->fit_mode == FIT_WIDTH_HEIGHT ? BST_CHECKED : BST_UNCHECKED);
				CheckDlgButton(hDlg, IDC_FITWH_LETTER, mfd->fit_mode == FIT_WIDTH_HEIGHT_LETTER ? BST_CHECKED : BST_UNCHECKED);
				CheckDlgButton(hDlg, IDC_PERCENTON, mfd->fit_mode == FIT_PERCENT ? BST_CHECKED : BST_UNCHECKED);
				CheckDlgButton(hDlg, IDC_NOENLARGE, mfd->no_enlarge == true ? BST_CHECKED : BST_UNCHECKED);
				CheckDlgButton(hDlg, IDC_NOREDUCE, mfd->no_reduce == true ? BST_CHECKED : BST_UNCHECKED);
				CheckDlgButton(hDlg, IDC_FIXWIDTH, mfd->fixw == true ? BST_CHECKED : BST_UNCHECKED);
				CheckDlgButton(hDlg, IDC_FIXHEIGHT, mfd->fixh == true ? BST_CHECKED : BST_UNCHECKED);
				SetDlgItemInt(hDlg, IDC_FIXWIDTHVAL, mfd->fixwval, FALSE);
				SetDlgItemInt(hDlg, IDC_FIXHEIGHTVAL, mfd->fixhval, FALSE);
				CheckDlgButton(hDlg, IDC_UP, mfd->round_mode == UP ? BST_CHECKED : BST_UNCHECKED);
				CheckDlgButton(hDlg, IDC_DOWN, mfd->round_mode == DOWN ? BST_CHECKED : BST_UNCHECKED);
				CheckDlgButton(hDlg, IDC_NEAREST, mfd->round_mode == NEAREST ? BST_CHECKED : BST_UNCHECKED);
				CheckDlgButton(hDlg, IDC_PADWIDTH, mfd->pad_mode == PAD_WIDTH ? BST_CHECKED : BST_UNCHECKED);
				CheckDlgButton(hDlg, IDC_PADHEIGHT, mfd->pad_mode == PAD_HEIGHT ? BST_CHECKED : BST_UNCHECKED);
				CheckDlgButton(hDlg, IDC_ADJUSTAR, mfd->pad_mode == PAD_ADJUST ? BST_CHECKED : BST_UNCHECKED);
				if (mfd->fit_mode == FIT_PERCENT)
				{
					EnableWindow(GetDlgItem(hDlg, IDC_NOREDUCE), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_NOENLARGE), FALSE);
				}
				else
				{
					EnableWindow(GetDlgItem(hDlg, IDC_NOREDUCE), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_NOENLARGE), TRUE);
				}
			    if (mfd->fit_mode == FIT_WIDTH_HEIGHT_LETTER)
				{
					EnableWindow(GetDlgItem(hDlg, IDC_NOENLARGE), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_NOREDUCE), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_PADWIDTH), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_FIXWIDTH), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_FIXHEIGHT), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_FIXWIDTHVAL), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_FIXHEIGHTVAL), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_PADHEIGHT), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_ADJUSTAR), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_UP), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_DOWN), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_NEAREST), FALSE);
				}
				if (mfd->fixh == true && mfd->fixw == true)
				{
					EnableWindow(GetDlgItem(hDlg, IDC_PADWIDTH), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_PADHEIGHT), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_ADJUSTAR), TRUE);
				}
				else
				{
					EnableWindow(GetDlgItem(hDlg, IDC_PADWIDTH), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_PADHEIGHT), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_ADJUSTAR), FALSE);
				}
				if (mfd->fixh == false && mfd->fixw == false)
				{
					EnableWindow(GetDlgItem(hDlg, IDC_UP), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_DOWN), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_NEAREST), FALSE);
				}
				else if (mfd->fixh == true && mfd->fixw == true && mfd->pad_mode != PAD_ADJUST)
				{
					mfd->round_mode = UP;
					CheckDlgButton(hDlg, IDC_UP, BST_CHECKED);
					CheckDlgButton(hDlg, IDC_DOWN, BST_UNCHECKED);
					CheckDlgButton(hDlg, IDC_NEAREST, BST_UNCHECKED);
					EnableWindow(GetDlgItem(hDlg, IDC_DOWN), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_NEAREST), FALSE);
				}
				else
				{
					EnableWindow(GetDlgItem(hDlg, IDC_DOWN), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_NEAREST), TRUE);
				}

				hwndItem = GetDlgItem(hDlg, IDC_FILTER_MODE);

				for(i=0; i<(sizeof filter_names/sizeof filter_names[0]); i++)
					SendMessage(hwndItem, CB_ADDSTRING, 0, (LPARAM)filter_names[i]);

				SendMessage(hwndItem, CB_SETCURSEL, mfd->filter_mode, 0);

				CheckDlgButton(hDlg, IDC_INTERLACED, mfd->fInterlaced ? BST_CHECKED : BST_UNCHECKED);

				if (mfd->enable == true)
				{
					mfd->fLetterbox = TRUE;
					CheckDlgButton(hDlg, IDC_ENABLE, BST_CHECKED);
					EnableWindow(GetDlgItem(hDlg, IDC_WIDTH), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_HEIGHT), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_FRAMEWIDTH), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_FRAMEHEIGHT), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_LETTERBOX), FALSE);
				}
				else
				{
					CheckDlgButton(hDlg, IDC_ENABLE, BST_UNCHECKED);
					EnableWindow(GetDlgItem(hDlg, IDC_WIDTH), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_HEIGHT), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_LETTERBOX), TRUE);
					if (mfd->fLetterbox) {
						CheckDlgButton(hDlg, IDC_LETTERBOX, BST_CHECKED);
					} else {
						CheckDlgButton(hDlg, IDC_LETTERBOX, BST_UNCHECKED);
						EnableWindow(GetDlgItem(hDlg, IDC_FRAMEWIDTH), FALSE);
						EnableWindow(GetDlgItem(hDlg, IDC_FRAMEHEIGHT), FALSE);
						EnableWindow(GetDlgItem(hDlg, IDC_STATIC_FILLCOLOR), FALSE);
						EnableWindow(GetDlgItem(hDlg, IDC_COLOR), FALSE);
						EnableWindow(GetDlgItem(hDlg, IDC_PICKCOLOR), FALSE);
					}
				}

				mfd->hbrColor = CreateSolidBrush(mfd->rgbColor);

				SetWindowLong(hDlg, DWL_USER, (LONG)mfd);

				mfd->ifp->InitButton(GetDlgItem(hDlg, IDC_PREVIEW));

				update(mfd);
				redisplay(hDlg, mfd);
			}
            return (TRUE);

        case WM_COMMAND:                      
			switch(LOWORD(wParam)) {
			case IDOK:
				mfd->ifp->Close();
				EndDialog(hDlg, 0);
				return TRUE;

			case IDCANCEL:
				mfd->ifp->Close();
                EndDialog(hDlg, 1);
                return TRUE;

			case IDC_WIDTH:
				if (HIWORD(wParam) == EN_KILLFOCUS) {
					long new_x;
					BOOL success;

					new_x = GetDlgItemInt(hDlg, IDC_WIDTH, &success, FALSE);
					if (!success || new_x < 16) {
						SetFocus((HWND)lParam);
						MessageBeep(MB_ICONQUESTION);
						return TRUE;
					}

					mfd->ifp->UndoSystem();
					mfd->new_x = new_x;
					mfd->ifp->RedoSystem();
				}
				return TRUE;

			case IDC_HEIGHT:
				if (HIWORD(wParam) == EN_KILLFOCUS) {
					long new_y;
					BOOL success;

					new_y = GetDlgItemInt(hDlg, IDC_HEIGHT, &success, FALSE);
					if (!success || new_y < 16) {
						SetFocus((HWND)lParam);
						MessageBeep(MB_ICONQUESTION);
						return TRUE;
					}

					mfd->ifp->UndoSystem();
					mfd->new_y = new_y;
					mfd->ifp->RedoSystem();
				}
				return TRUE;

			case IDC_FRAMEWIDTH:
				if (HIWORD(wParam) == EN_KILLFOCUS) {
					long new_xf;
					BOOL success;

					new_xf = GetDlgItemInt(hDlg, IDC_FRAMEWIDTH, &success, FALSE);
					if (!success || new_xf < mfd->new_x) {
						SetFocus((HWND)lParam);
						MessageBeep(MB_ICONQUESTION);
						return TRUE;
					}

					mfd->ifp->UndoSystem();
					mfd->new_xf = new_xf;
					mfd->ifp->RedoSystem();
				}
				return TRUE;

			case IDC_FRAMEHEIGHT:
				if (HIWORD(wParam) == EN_KILLFOCUS) {
					long new_yf;
					BOOL success;

					new_yf = GetDlgItemInt(hDlg, IDC_FRAMEHEIGHT, &success, FALSE);
					if (!success || new_yf < mfd->new_y) {
						SetFocus((HWND)lParam);
						MessageBeep(MB_ICONQUESTION);
						return TRUE;
					}

					mfd->ifp->UndoSystem();
					mfd->new_yf = new_yf;
					mfd->ifp->RedoSystem();
				}
				return TRUE;

			case IDC_PREVIEW:
				mfd->ifp->Toggle(hDlg);
				return TRUE;

			case IDC_FILTER_MODE:
				if (HIWORD(wParam) == CBN_SELCHANGE) {
					mfd->ifp->UndoSystem();
					mfd->filter_mode = SendDlgItemMessage(hDlg, IDC_FILTER_MODE, CB_GETCURSEL, 0, 0);
					mfd->ifp->RedoSystem();
				}
				return TRUE;

			case IDC_INTERLACED:
				if (HIWORD(wParam) == BN_CLICKED) {
					BOOL f = IsDlgButtonChecked(hDlg, IDC_INTERLACED);

					mfd->ifp->UndoSystem();
					mfd->fInterlaced = !!f;

					mfd->ifp->RedoSystem();
				}
				return TRUE;

			case IDC_LETTERBOX:
				if (HIWORD(wParam) == BN_CLICKED) {
					BOOL f = IsDlgButtonChecked(hDlg, IDC_LETTERBOX);

					mfd->ifp->UndoSystem();
					mfd->fLetterbox = !!f;

					EnableWindow(GetDlgItem(hDlg, IDC_STATIC_FILLCOLOR), f);
					EnableWindow(GetDlgItem(hDlg, IDC_COLOR), f);
					EnableWindow(GetDlgItem(hDlg, IDC_PICKCOLOR), f);
					EnableWindow(GetDlgItem(hDlg, IDC_FRAMEWIDTH), f);
					EnableWindow(GetDlgItem(hDlg, IDC_FRAMEHEIGHT), f);

					if (mfd->fLetterbox) {
						if (mfd->new_xf < mfd->new_x) {
							mfd->new_xf = mfd->new_x;
							SetDlgItemInt(hDlg, IDC_FRAMEWIDTH, mfd->new_xf, FALSE);
						}

						if (mfd->new_yf < mfd->new_y) {
							mfd->new_yf = mfd->new_y;
							SetDlgItemInt(hDlg, IDC_FRAMEHEIGHT, mfd->new_yf, FALSE);
						}
					}
					mfd->ifp->RedoSystem();
				}
				return TRUE;

			case IDC_PICKCOLOR:
				if (guiChooseColor(hDlg, mfd->rgbColor)) {
					DeleteObject(mfd->hbrColor);
					mfd->hbrColor = CreateSolidBrush(mfd->rgbColor);
					RedrawWindow(GetDlgItem(hDlg, IDC_COLOR), NULL, NULL, RDW_ERASE|RDW_INVALIDATE|RDW_UPDATENOW);
					mfd->ifp->RedoSystem();
				}
				break;

			case IDC_INWIDTH:
				if (HIWORD(wParam) == EN_UPDATE) {
					long inwidth;
					BOOL success;

					inwidth = GetDlgItemInt(hDlg, IDC_INWIDTH, &success, FALSE);
					if (!success || inwidth < 1) {
						SetDlgItemInt(hDlg, IDC_INWIDTH, mfd->in_width, FALSE);
						return TRUE;
					}
					if (mfd != NULL)
					{
						mfd->in_width = inwidth;
						update(mfd);
						redisplay(hDlg, mfd);
					}
				}
				return TRUE;

			case IDC_INHEIGHT:
				if (HIWORD(wParam) == EN_UPDATE) {
					long inheight;
					BOOL success;

					inheight = GetDlgItemInt(hDlg, IDC_INHEIGHT, &success, FALSE);
					if (!success || inheight < 1) {
						SetDlgItemInt(hDlg, IDC_INHEIGHT, mfd->in_height, FALSE);
						return TRUE;
					}					
					if (mfd != NULL)
					{
						mfd->in_height = inheight;
						update(mfd);
						redisplay(hDlg, mfd);
					}
				}
				return TRUE;

			case IDC_NUM:
				if (HIWORD(wParam) == EN_UPDATE) {
					long num;
					BOOL success;
					if (mfd == NULL) return TRUE;

					num = GetDlgItemInt(hDlg, IDC_NUM, &success, FALSE);
					if (!success || num < 1 ||
						(mfd->enable && (num > 10 * mfd->denom || mfd->denom > 10 * num)))
					{
						SetDlgItemInt(hDlg, IDC_NUM, mfd->num, FALSE);
						return TRUE;
					}
					if (mfd != NULL)
					{
						mfd->num = num;
						update(mfd);
					    redisplay(hDlg, mfd);
					}
				}
				return TRUE;

			case IDC_DENOM:
				if (HIWORD(wParam) == EN_UPDATE) {
					long denom;
					BOOL success;
					if (mfd == NULL) return TRUE;

					denom = GetDlgItemInt(hDlg, IDC_DENOM, &success, FALSE);
					if (!success || denom < 1 ||
						(mfd->enable && (denom > 10 * mfd->num || mfd->num > 10 * denom)))
					{
						SetDlgItemInt(hDlg, IDC_DENOM, mfd->denom, FALSE);
						return TRUE;
					}
					if (mfd != NULL)
					{
						mfd->denom = denom;
						update(mfd);
						redisplay(hDlg, mfd);
                    }
				}
				return TRUE;

			case IDC_FITWIDTHON:
 				mfd->fit_mode = FIT_WIDTH;
				EnableWindow(GetDlgItem(hDlg, IDC_NOENLARGE), TRUE);
				EnableWindow(GetDlgItem(hDlg, IDC_NOREDUCE), TRUE);
				EnableWindow(GetDlgItem(hDlg, IDC_FIXWIDTH), TRUE);
				EnableWindow(GetDlgItem(hDlg, IDC_FIXHEIGHT), TRUE);
				EnableWindow(GetDlgItem(hDlg, IDC_FIXWIDTHVAL), TRUE);
				EnableWindow(GetDlgItem(hDlg, IDC_FIXHEIGHTVAL), TRUE);
				update(mfd);
				redisplay(hDlg, mfd);
				return TRUE;

			case IDC_FITHEIGHTON:
 				mfd->fit_mode = FIT_HEIGHT;
				EnableWindow(GetDlgItem(hDlg, IDC_NOENLARGE), TRUE);
				EnableWindow(GetDlgItem(hDlg, IDC_NOREDUCE), TRUE);
				EnableWindow(GetDlgItem(hDlg, IDC_FIXWIDTH), TRUE);
				EnableWindow(GetDlgItem(hDlg, IDC_FIXHEIGHT), TRUE);
				EnableWindow(GetDlgItem(hDlg, IDC_FIXWIDTHVAL), TRUE);
				EnableWindow(GetDlgItem(hDlg, IDC_FIXHEIGHTVAL), TRUE);
				update(mfd);
				redisplay(hDlg, mfd);
				return TRUE;

			case IDC_FITWH:
 				mfd->fit_mode = FIT_WIDTH_HEIGHT;
				EnableWindow(GetDlgItem(hDlg, IDC_NOENLARGE), TRUE);
				EnableWindow(GetDlgItem(hDlg, IDC_NOREDUCE), TRUE);
				EnableWindow(GetDlgItem(hDlg, IDC_FIXWIDTH), TRUE);
				EnableWindow(GetDlgItem(hDlg, IDC_FIXHEIGHT), TRUE);
				EnableWindow(GetDlgItem(hDlg, IDC_FIXWIDTHVAL), TRUE);
				EnableWindow(GetDlgItem(hDlg, IDC_FIXHEIGHTVAL), TRUE);
				update(mfd);
				redisplay(hDlg, mfd);
				return TRUE;

			case IDC_FITWH_LETTER:
 				mfd->fit_mode = FIT_WIDTH_HEIGHT_LETTER;
				EnableWindow(GetDlgItem(hDlg, IDC_NOENLARGE), TRUE);
				EnableWindow(GetDlgItem(hDlg, IDC_NOREDUCE), TRUE);
				EnableWindow(GetDlgItem(hDlg, IDC_PADWIDTH), FALSE);
				EnableWindow(GetDlgItem(hDlg, IDC_FIXWIDTH), FALSE);
				EnableWindow(GetDlgItem(hDlg, IDC_FIXHEIGHT), FALSE);
				EnableWindow(GetDlgItem(hDlg, IDC_FIXWIDTHVAL), FALSE);
				EnableWindow(GetDlgItem(hDlg, IDC_FIXHEIGHTVAL), FALSE);
				EnableWindow(GetDlgItem(hDlg, IDC_PADHEIGHT), FALSE);
				EnableWindow(GetDlgItem(hDlg, IDC_ADJUSTAR), FALSE);
				EnableWindow(GetDlgItem(hDlg, IDC_UP), FALSE);
				EnableWindow(GetDlgItem(hDlg, IDC_DOWN), FALSE);
				EnableWindow(GetDlgItem(hDlg, IDC_NEAREST), FALSE);
				update(mfd);
				redisplay(hDlg, mfd);
				return TRUE;

			case IDC_PERCENTON:
 				mfd->fit_mode = FIT_PERCENT;
				EnableWindow(GetDlgItem(hDlg, IDC_NOENLARGE), FALSE);
				EnableWindow(GetDlgItem(hDlg, IDC_NOREDUCE), FALSE);
				EnableWindow(GetDlgItem(hDlg, IDC_FIXWIDTH), TRUE);
				EnableWindow(GetDlgItem(hDlg, IDC_FIXHEIGHT), TRUE);
				EnableWindow(GetDlgItem(hDlg, IDC_FIXWIDTHVAL), TRUE);
				EnableWindow(GetDlgItem(hDlg, IDC_FIXHEIGHTVAL), TRUE);
				update(mfd);
				redisplay(hDlg, mfd);
				return TRUE;

			case IDC_FITHEIGHT:
				if (HIWORD(wParam) == EN_UPDATE) {
					long fitheight;
					BOOL success;

					fitheight = GetDlgItemInt(hDlg, IDC_FITHEIGHT, &success, FALSE);
					if (!success || fitheight < 1) {
						SetDlgItemInt(hDlg, IDC_FITHEIGHT, mfd->fitheight, FALSE);
						return TRUE;
					}
					if (mfd != NULL)
					{
						mfd->fitheight = fitheight;
						update(mfd);
						redisplay(hDlg, mfd);
					}
				}
				return TRUE;

			case IDC_FITWIDTH:
				if (HIWORD(wParam) == EN_UPDATE) {
					long fitwidth;
					BOOL success;

					fitwidth = GetDlgItemInt(hDlg, IDC_FITWIDTH, &success, FALSE);
					if (!success || fitwidth < 1) {
						SetDlgItemInt(hDlg, IDC_FITWIDTH, mfd->fitwidth, FALSE);
						return TRUE;
					}
					if (mfd != NULL)
					{
						mfd->fitwidth = fitwidth;
						update(mfd);
						redisplay(hDlg, mfd);
					}
				}
				return TRUE;

			case IDC_PERCENT:
				if (HIWORD(wParam) == EN_UPDATE) {
					long percent;
					BOOL success;

					percent = GetDlgItemInt(hDlg, IDC_PERCENT, &success, FALSE);
					if (!success || percent < 1) {
						SetDlgItemInt(hDlg, IDC_PERCENT, mfd->percent, FALSE);
						return TRUE;
					}
					if (mfd != NULL)
					{
						mfd->percent = percent;
						update(mfd);
						redisplay(hDlg, mfd);
					}
				}
				return TRUE;

			case IDC_NOENLARGE:
				if (IsDlgButtonChecked(hDlg, IDC_NOENLARGE))
				{
					mfd->no_enlarge = true;
					mfd->no_reduce = false;
					CheckDlgButton(hDlg, IDC_NOREDUCE, BST_UNCHECKED);
				}
				else
				{
					mfd->no_enlarge = false;
				}
				update(mfd);
				redisplay(hDlg, mfd);
				return TRUE;

			case IDC_NOREDUCE:
				if (IsDlgButtonChecked(hDlg, IDC_NOREDUCE))
				{
					mfd->no_reduce = true;
					mfd->no_enlarge = false;
					CheckDlgButton(hDlg, IDC_NOENLARGE, BST_UNCHECKED);
				}
				else
				{
					mfd->no_reduce = false;
				}
				update(mfd);
				redisplay(hDlg, mfd);
				return TRUE;

			case IDC_FIXWIDTH:
				if (IsDlgButtonChecked(hDlg, IDC_FIXWIDTH))
					mfd->fixw = true;
				else
					mfd->fixw = false;
				if (mfd->fixh == false && mfd->fixw == false)
				{
					EnableWindow(GetDlgItem(hDlg, IDC_UP), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_DOWN), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_NEAREST), FALSE);
				}
				else if (mfd->fixh == true && mfd->fixw == true)
				{
					EnableWindow(GetDlgItem(hDlg, IDC_PADWIDTH), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_PADHEIGHT), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_ADJUSTAR), TRUE);
					if (mfd->pad_mode != PAD_ADJUST)
					{
						EnableWindow(GetDlgItem(hDlg, IDC_DOWN), FALSE);
						EnableWindow(GetDlgItem(hDlg, IDC_NEAREST), FALSE);
						mfd->round_mode = UP;
						CheckDlgButton(hDlg, IDC_UP, BST_CHECKED);
						CheckDlgButton(hDlg, IDC_DOWN, BST_UNCHECKED);
						CheckDlgButton(hDlg, IDC_NEAREST, BST_UNCHECKED);
					}
					else
					{
						EnableWindow(GetDlgItem(hDlg, IDC_DOWN), TRUE);
						EnableWindow(GetDlgItem(hDlg, IDC_NEAREST), TRUE);
					}
				}
				else
				{
					EnableWindow(GetDlgItem(hDlg, IDC_PADWIDTH), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_PADHEIGHT), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_ADJUSTAR), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_UP), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_DOWN), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_NEAREST), TRUE);
				}
				update(mfd);
				redisplay(hDlg, mfd);
				return TRUE;
			
			case IDC_FIXHEIGHT:
				if (IsDlgButtonChecked(hDlg, IDC_FIXHEIGHT))
					mfd->fixh = true;
				else
					mfd->fixh = false;
				if (mfd->fixh == false && mfd->fixw == false)
				{
					EnableWindow(GetDlgItem(hDlg, IDC_UP), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_DOWN), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_NEAREST), FALSE);
				}
				else if (mfd->fixh == true && mfd->fixw == true)
				{
					EnableWindow(GetDlgItem(hDlg, IDC_PADWIDTH), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_PADHEIGHT), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_ADJUSTAR), TRUE);
					if (mfd->pad_mode != PAD_ADJUST)
					{
						EnableWindow(GetDlgItem(hDlg, IDC_DOWN), FALSE);
						EnableWindow(GetDlgItem(hDlg, IDC_NEAREST), FALSE);
						mfd->round_mode = UP;
						CheckDlgButton(hDlg, IDC_UP, BST_CHECKED);
						CheckDlgButton(hDlg, IDC_DOWN, BST_UNCHECKED);
						CheckDlgButton(hDlg, IDC_NEAREST, BST_UNCHECKED);
					}
					else
					{
						EnableWindow(GetDlgItem(hDlg, IDC_DOWN), TRUE);
						EnableWindow(GetDlgItem(hDlg, IDC_NEAREST), TRUE);
					}
				}
				else
				{
					EnableWindow(GetDlgItem(hDlg, IDC_PADWIDTH), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_PADHEIGHT), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_ADJUSTAR), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_UP), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_DOWN), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_NEAREST), TRUE);
				}
				update(mfd);
				redisplay(hDlg, mfd);
				return TRUE;

			case IDC_FIXWIDTHVAL:
				if (HIWORD(wParam) == EN_UPDATE) {
					long fixwval;
					BOOL success;

					fixwval = GetDlgItemInt(hDlg, IDC_FIXWIDTHVAL, &success, FALSE);
					if (!success || fixwval < 1) {
						SetDlgItemInt(hDlg, IDC_FIXWIDTHVAL, mfd->fixwval, FALSE);
						return TRUE;
					}
					if (mfd != NULL)
					{
						mfd->fixwval = fixwval;
						update(mfd);
						redisplay(hDlg, mfd);
					}
				}
				return TRUE;

			case IDC_FIXHEIGHTVAL:
				if (HIWORD(wParam) == EN_UPDATE) {
					long fixhval;
					BOOL success;

					fixhval = GetDlgItemInt(hDlg, IDC_FIXHEIGHTVAL, &success, FALSE);
					if (!success || fixhval < 1) {
						SetDlgItemInt(hDlg, IDC_FIXHEIGHTVAL, mfd->fixhval, FALSE);
						return TRUE;
					}
					if (mfd != NULL)
					{
						mfd->fixhval = fixhval;
						update(mfd);
						redisplay(hDlg, mfd);
					}
				}
				return TRUE;

			case IDC_UP:
 				mfd->round_mode = UP;
				update(mfd);
				redisplay(hDlg, mfd);
				return TRUE;

			case IDC_DOWN:
				mfd->round_mode = DOWN;
				update(mfd);
				redisplay(hDlg, mfd);
				return TRUE;

			case IDC_NEAREST:
				mfd->round_mode = NEAREST;
				update(mfd);
				redisplay(hDlg, mfd);
				return TRUE;

			case IDC_PADWIDTH:
 				mfd->pad_mode = PAD_WIDTH;
				mfd->round_mode = UP;
				if (mfd->fixh == true && mfd->fixw == true)
				{
					mfd->round_mode = UP;
					CheckDlgButton(hDlg, IDC_UP, BST_CHECKED);
					CheckDlgButton(hDlg, IDC_DOWN, BST_UNCHECKED);
					CheckDlgButton(hDlg, IDC_NEAREST, BST_UNCHECKED);
					EnableWindow(GetDlgItem(hDlg, IDC_DOWN), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_NEAREST), FALSE);
				}
				else
				{
					EnableWindow(GetDlgItem(hDlg, IDC_DOWN), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_NEAREST), TRUE);
				}
				update(mfd);
				redisplay(hDlg, mfd);
				return TRUE;

			case IDC_PADHEIGHT:
 				mfd->pad_mode = PAD_HEIGHT;
				mfd->round_mode = UP;
				if (mfd->fixh == true && mfd->fixw == true)
				{
					mfd->round_mode = UP;
					CheckDlgButton(hDlg, IDC_UP, BST_CHECKED);
					CheckDlgButton(hDlg, IDC_DOWN, BST_UNCHECKED);
					CheckDlgButton(hDlg, IDC_NEAREST, BST_UNCHECKED);
					EnableWindow(GetDlgItem(hDlg, IDC_DOWN), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_NEAREST), FALSE);
				}
				else
				{
					EnableWindow(GetDlgItem(hDlg, IDC_DOWN), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_NEAREST), TRUE);
				}
				update(mfd);
				redisplay(hDlg, mfd);
				return TRUE;

			case IDC_ADJUSTAR:
 				mfd->pad_mode = PAD_ADJUST;
				EnableWindow(GetDlgItem(hDlg, IDC_DOWN), TRUE);
				EnableWindow(GetDlgItem(hDlg, IDC_NEAREST), TRUE);
				update(mfd);
				redisplay(hDlg, mfd);
				return TRUE;

			case IDC_ENABLE:
				mfd->enable = (mfd->enable == true ? false : true);
				if (mfd->enable && (mfd->num > 10 * mfd->denom || mfd->denom > 10 * mfd->num))
				{
					mfd->num = 1;
					mfd->denom = 1;
					SetDlgItemInt(hDlg, IDC_NUM, mfd->num, FALSE);
					SetDlgItemInt(hDlg, IDC_DENOM, mfd->denom, FALSE);
				}
				if (mfd->enable == true)
				{
					EnableWindow(GetDlgItem(hDlg, IDC_WIDTH), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_HEIGHT), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_FRAMEWIDTH), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_FRAMEHEIGHT), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_LETTERBOX), FALSE);
					update(mfd);
					redisplay(hDlg, mfd);
				}
				else
				{
					EnableWindow(GetDlgItem(hDlg, IDC_WIDTH), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_HEIGHT), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_FRAMEWIDTH), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_FRAMEHEIGHT), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_LETTERBOX), TRUE);
				}
				return TRUE;

 			case IDHELP:
				{
				char prog[256];
				char path[256];
				LPTSTR ptr;

				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\Resize.html");
				ShellExecute(hDlg, "open", path, NULL, NULL, SW_SHOWNORMAL);
				return TRUE;
				}
            }
            break;

		case WM_CTLCOLORSTATIC:
			if (GetWindowLong((HWND)lParam, GWL_ID) == IDC_COLOR)
				return (BOOL)mfd->hbrColor;
			break;
    }
    return FALSE;
}

static int resize_config(FilterActivation *fa, const FilterFunctions *ff, HWND hWnd) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	MyFilterData mfd2 = *mfd;
	int ret;

	mfd->hbrColor = NULL;
	mfd->ifp = fa->ifp;

	mfd->in_width = fa->src.w;
	mfd->in_height = fa->src.h;

	if (mfd->new_x < 16)
		mfd->new_x = 320;
	if (mfd->new_y < 16)
		mfd->new_y = 240;

	ret = DialogBoxParam(fa->filter->module->hInstModule, MAKEINTRESOURCE(IDD_FILTER_RESIZE), hWnd, resizeDlgProc, (LONG)mfd);

	if (mfd->hbrColor) {
		DeleteObject(mfd->hbrColor);
		mfd->hbrColor = NULL;
	}

	if (ret)
		*mfd = mfd2;

	return ret;
}

static void resize_string(const FilterActivation *fa, const FilterFunctions *ff, char *buf) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	if (mfd->fLetterbox)
		wsprintf(buf, " (%s, lbox %dx%d #%06x)", filter_names[mfd->filter_mode],
				mfd->new_xf, mfd->new_yf, revcolor(mfd->rgbColor));
	else
		wsprintf(buf, " (%s)", filter_names[mfd->filter_mode]);
}

static int resize_start(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	long dstw = mfd->new_x;
	long dsth = mfd->new_y;

	if (dstw<16 || dsth<16)
		return 1;

	Resampler::eFilter fmode;

	switch(mfd->filter_mode) {
	case FILTER_NONE:			fmode = Resampler::eFilter::kPoint; break;
	case FILTER_BILINEAR:		fmode = Resampler::eFilter::kLinearInterp; break;
	case FILTER_BICUBIC:		fmode = Resampler::eFilter::kCubicInterp; break;
	case FILTER_TABLEBILINEAR:	fmode = Resampler::eFilter::kLinearDecimate; break;
	case FILTER_TABLEBICUBIC:	fmode = Resampler::eFilter::kCubicDecimate; break;
	}

	mfd->resampler = new Resampler();

#ifdef USE_INTERLACED
	if (mfd->fInterlaced)
		mfd->resampler->Init(fmode, fmode, dstw, dsth/2, fa->src.w, fa->src.h/2);
	else
#endif
		mfd->resampler->Init(fmode, fmode, dstw, dsth, fa->src.w, fa->src.h);

	return 0;
}

static int resize_stop(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	delete mfd->resampler;	mfd->resampler = NULL;

	return 0;
}

static void resize_script_config(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc)
{
	FilterActivation *fa = (FilterActivation *)lpVoid;

	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	char buff[255];

	mfd->new_x	= argv[0].asInt();
	mfd->new_y	= argv[1].asInt();
	mfd->fLetterbox = !!argv[2].asInt();
	mfd->filter_mode = argv[3].asInt();
	mfd->fInterlaced = false;
	if (mfd->filter_mode & 128) {
		mfd->fInterlaced = true;
		mfd->filter_mode &= 127;
	}
	mfd->new_xf = argv[4].asInt();
	mfd->new_yf = argv[5].asInt();
	mfd->rgbColor = revcolor(argv[6].asInt());
	strcpy(buff, *argv[7].asString());
	sscanf(buff, "%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d",
		   &mfd->enable, &mfd->num, &mfd->denom, &mfd->fit_mode,
		   &mfd->fitwidth, &mfd->fitheight, &mfd->percent, &mfd->no_enlarge, &mfd->no_reduce,
   		   &mfd->fixw, &mfd->fixh, &mfd->fixwval, &mfd->fixhval, &mfd->round_mode, &mfd->pad_mode);
}

static ScriptFunctionDef resize_func_defs[]={
	{ (ScriptFunctionPtr)resize_script_config, "Config", "0iiiiiiis" },
	{ NULL },
};

static CScriptObject resize_obj={
	NULL, resize_func_defs
};

static bool resize_script_line(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	int filtmode = mfd->filter_mode + (mfd->fInterlaced ? 128 : 0);
	char buff[255];

	sprintf(buff, "%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d",
		    mfd->enable, mfd->num, mfd->denom, mfd->fit_mode,
		    mfd->fitwidth, mfd->fitheight, mfd->percent, mfd->no_enlarge, mfd->no_reduce,
			mfd->fixw, mfd->fixh, mfd->fixwval, mfd->fixhval, mfd->round_mode, mfd->pad_mode);
	_snprintf(buf, buflen, "Config(%d,%d,%d,%d,%d,%d,0x%06x,\"%s\")",
		          mfd->new_x, mfd->new_y, mfd->fLetterbox,
		          filtmode, mfd->new_xf, mfd->new_yf, revcolor(mfd->rgbColor), buff);

	return true;
}

int resize_init(FilterActivation *fa, const FilterFunctions *ff);

FilterDefinition filterDef_resize={
	0,0,NULL,
	"smart resize (1.1)",
	"Resizes the image to a new size (based on internal resize filter by Avery Lee)."
	"\nSize calculator conceived by Jon C. Hodgson."
#ifdef USE_ASM
			"\n[Assembly optimized] [FPU optimized] [MMX optimized]"
#endif
			,
	"Donald Graft",
	NULL,
	sizeof(MyFilterData),
	resize_init,
	NULL,
	resize_run,
	resize_param,
	resize_config,
	resize_string,
	resize_start,
	resize_stop,

	&resize_obj,
	resize_script_line,
};

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_tutorial;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if (!(fd_tutorial = ff->addFilter(fm, &filterDef_resize, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_tutorial);
}

int resize_init(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	INITIALIZE_VTBLS;
	mfd->enable = false;
	mfd->num = 1;
	mfd->denom = 1;
	mfd->fit_mode = FIT_PERCENT;
	mfd->fitwidth = 320;
	mfd->fitheight = 240;
	mfd->percent = 100;
	mfd->no_enlarge = false;
	mfd->no_reduce = false;
	mfd->fixw = false;
	mfd->fixh = false;
	mfd->fixwval = 4;
	mfd->fixhval = 4;
	mfd->round_mode = NEAREST;
	mfd->pad_mode = PAD_ADJUST;

	return 0;
}
