/*
    Antiflicker Filter for VirtualDub -- removes temporal moire flickering.


	PROCESS PHASE
	*************

		Copyright (C) 2002-2004 Alessandro Malanca, 
		http://web.tiscali.it/minomala/

	original idea by:

		Copyright (C) 1999-2000 Donald A. Graft


    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <direct.h>
#include <math.h>
#include <crtdbg.h>
#include <commdlg.h>
#include <winuser.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"
#include "VdubGfxLib.h"

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
int StartProc(FilterActivation *fa, const FilterFunctions *ff);
int EndProc(FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int MyInitProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

#define MSGBLACK	0xFF000000
#define MSGWHITE	0xFFFFFFFF
#define MSGRED		0xFFFF0000
#define MSGBLUE		0xFF0000FF

#define MSG_W 80
#define MSG_H 16

#define MAX_KERNEL 100
#define MAX_SOFTENING 31

#define DEFAULTFILENAME	"filt.txt"

class LumFileWindow {

class Cache {

	#define CACHE_SIZE 50
	typedef  struct Elem { int key; int val; } Elem;
	Elem table [CACHE_SIZE];
	int size;
	int scanIndex;
	int firstOut;
	FILE * fp;

public:

	void init(FILE * f,int s) {
		this->size = (s>=CACHE_SIZE) ? CACHE_SIZE : s;
		this->fp = f;
		firstOut = 0;
		for ( int i=0; i<size; i++) {
			try {			
				table[i].val = getIframeLum (f,i);
				table[i].key = i;
			} catch(char*) {
				table[i].val = 
				table[i].key = -1;
			}
		}
		scanIndex=0;
	}

	int getValue(int key) {
		int k = scanIndex;
		while ((table[k].key != key) && (k != scanIndex)) k=next(k);
		if ( k != scanIndex ) {
			scanIndex = k;
		} else {
			table[firstOut].val = getIframeLum (fp,key); // throws (char*)
			table[firstOut].key = key;
			k = firstOut;
		}
		return table[k].val;
	}

private:

	int next(int i)  {
		if ( table[i].key < table[firstOut].key ) {firstOut = i;};
		i = (i+1) % size;
		return i;
	}
#undef CACHE_SIZE
}; // Cache class

// --------------------------------------
//      Start of class LumFileWindow
// --------------------------------------

protected:

	double	maxCorrection; // 0.01-0.99 %
	int		windowSize;
	FILE	*rfp;
	Cache	cache; 

	#define NumOfHeaderRecord 5

	#define RecordLength	  (3+2)
	bool	interlaced;
	int		suggestedWindow;


public:

	~LumFileWindow() {	done(); }

	LumFileWindow() {
		interlaced = false;
		suggestedWindow = 0;
		totalFramesCount = 0;
	}

	bool getInterlaced()	 { return interlaced; }

	int getWindowSuggested() { return suggestedWindow; }

	int getMaxCorrection()   { return int (maxCorrection*100.0); }
	long getTotalFramesCount() { return totalFramesCount; }

	void setMaxCorrection(int val) {	
		maxCorrection=(double)val/(double)100.0;
	}


	bool initRead ( char * fileName, int windowSize ) {

		this->windowSize = windowSize;
		done();
		try {
			readHeader (fileName, true);
			cache.init(rfp,50);
			return true;
		} catch( char*) {	
			return false;
		}
	}

	long totalFramesCount;

	void readHeader (	char * fileName,  bool heldOpen ) /*throw (char*)*/ {
		done();
		if ((rfp = fopen(fileName, "r")) == NULL) {
			throw("readHeader fopen");
		}
		try {
			int i,s;
			int nextRec = 0;
			i = readRec(rfp,nextRec++);
			s = readRec(rfp,nextRec++);

			int a,b,c;
			a = readRec(rfp,nextRec++);
			b = readRec(rfp,nextRec++);
			c = readRec(rfp,nextRec++);

			interlaced = (i==1);
			suggestedWindow = s;
			totalFramesCount = c<<16|b<<8|a;
			if (!heldOpen) done();

		} catch (char*) {
			throw ( "ReadHeader error");
		}
	}

	void done() {
		if ( rfp!=NULL )  { fclose(rfp); rfp=NULL; };
	}

	bool getLum ( long frame, double * lumValue ) /* throw char*/ {

		long first = frame - windowSize/2;
		long last = first + windowSize-1;		
		first = (first>=0) ? first : 0;
		long currLum = cache.getValue(frame);

		long media = 0;
		long tLum;
		int tSize=0;
		bool exceeded = false;
		for ( int k= first; k<=last; k++ ) {
			try {
				tLum = cache.getValue(k);
				if ( fabs( ((double)tLum / (double)currLum) - 1.0) >= maxCorrection ) { 
					// discard values exceeding max correction !
					media += currLum;
					exceeded = true;				
					tSize++;
				} else {
					media += tLum;
					tSize++;
				}				
			} catch (char*) {
				// ignore missing !
			}
		}

		*lumValue = ((double)media/(double)tSize)/(double)currLum;
		return exceeded;
	}

	static long getIframeLum (FILE * f, long frame)  {
		return readRec(f, frame + NumOfHeaderRecord);
	}


private:

	static long readRec (FILE * f, long offset) /*throw(char*)*/ {
		if( ! fseek(f,(long)(RecordLength*offset),SEEK_SET)) {
			char tmp[10];
			if (fgets(tmp, 10, f) != NULL) {	
				return atoi(tmp);
			}		
		}			
		throw "readRec Error";
	}

	void buildFilePath(char * dest, char * path, char * name ) {
		strcpy(dest, path);
		strcat(dest, name);
	}

#undef NumOfHeaderRecord	
#undef RecordLength
};


typedef struct MyFilterData {

	int 				window;
	int					softening;
	char				processFileName[MAX_PATH];
    unsigned char		*old_data;
    long				size;
	bool				error;
	LumFileWindow		lumWindow;
	bool				correctionPreview;

} MyFilterData;

void fileString( char * d, char * s) {
	while( *s != 0 ) {
		if (*s=='\\') *d++ = '\\';
		*d=*s;
		d++;s++;
	}
	*d=0;
}

bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	char ss[MAX_PATH];
	fileString( ss, mfd->processFileName);

	_snprintf(buf, buflen, "Config(%d, %d, %d, \"%s\")",
		mfd->window, mfd->softening, 
		mfd->lumWindow.getMaxCorrection(), ss
	);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->window		= argv[0].asInt();
	mfd->softening	= argv[1].asInt();
	mfd->lumWindow.setMaxCorrection(argv[2].asInt());
	strcpy(mfd->processFileName,*argv[3].asString());
}

ScriptFunctionDef func_defs[]={
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0iiis" },
	{ NULL },
};

CScriptObject script_obj={
	NULL, func_defs
};

struct FilterDefinition filterDef_tutorial = {

	NULL, NULL, NULL,					// next, prev, module
	"deflickProcess(A.2.0)",			// name
	"Remove temporal frame luminance variations (flicker)\nUses a file produced with the filter Deflick Prepare",
										// desc
	"Ale Malanca/Donald Graft", 		// maker
	NULL,								// private_data
	sizeof(MyFilterData),				// inst_data_size
	MyInitProc,							// initProc
	NULL,								// deinitProc
	RunProc,							// runProc
	ParamProc,							// paramProc
	ConfigProc, 						// configProc
	StringProc, 						// stringProc
	StartProc,							// startProc
	EndProc,							// endProc
	&script_obj,						// script_obj
	FssProc,							// fssProc
};


long ParamProc(FilterActivation *fa, const FilterFunctions *ff) {
    return FILTERPARAM_SWAP_BUFFERS;
}


int StartProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	// ------------------------
	// Process Video
	//-------------------------
	// Make buffer and clear it.

	mfd->size = fa->src.Size();
	if ((mfd->old_data = new unsigned char [mfd->size]) == 0)
		return 1;

	memset (mfd->old_data, 0, mfd->size);

	mfd->error = ! mfd->lumWindow.initRead(mfd->processFileName, mfd->window);

	return ( mfd->error ? (1) : (0) );
}


int EndProc(FilterActivation *fa, const FilterFunctions *ff) {

	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	if ( mfd->old_data != NULL ) {
		delete[] mfd->old_data;
		mfd->old_data = NULL;
	}
	mfd->lumWindow.done();

	return 0;
}

///////////////////////////////////////////////////////////////////////////

void	msgOut(int x, int y, const FilterActivation *fa, char * msg, Pixel32 color)	{
		fa->dst.RectFill(x, y,  MSG_W, MSG_H,  MSGWHITE);
		VGL_DrawString(fa, x+5, y+5, msg, color);
}


void	msgOut(const FilterActivation *fa, char * msg, Pixel32 color)	{
		int x = (fa->dst.w - MSG_W) /2;
		int y = (fa->dst.h - MSG_H) /2;
		msgOut(x,y,fa,msg,color);
		//fa->dst.RectFill(x, y,  MSG_W, MSG_H,  MSGBACKG);
		//VGL_DrawString(fa, x+5, y+5, msg, MSGCOLOR);
}


int RunProc(const FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	const PixDim	w = fa->src.w;
	const PixDim	h = fa->src.h;
	Pixel32 *src, *dst;
	int field, x, y;

	double scale;

	int r, g, b, max;
	FilterStateInfo *pfsi = fa->pfsi;
    int current;
	Pixel32 inPix;

	int FIELDS = (mfd->lumWindow.getInterlaced())? (2) : (1) ;

	// -----------------------
	//   Process video
	// -----------------------
	double saveScale[2];

	Pixel32 color;

	for (field = 0; field <= FIELDS - 1; field++)
	{
		current = FIELDS * pfsi->lCurrentSourceFrame + field;
		try {
			if(mfd->lumWindow.getLum ( current, &scale )) {
				// max correction exeeded
				saveScale[field]=scale;
				color = MSGBLUE;
			} else {	
				saveScale[field]=scale;
				color = MSGBLACK;
			}

		} catch(char*) {
			saveScale[field]=scale=1.0;
			//		if (mfd->correctionPreview) {							
			msgOut(fa,"File Err!",MSGRED);
			return -1;
			//		}	
		}

		// Luminance adjustment phase. 
		src = (Pixel32 *)((char *)fa->src.data + field * fa->src.pitch);
		dst = (Pixel32 *)((char *)fa->dst.data + field * fa->dst.pitch);
		
		int scaleInt = (int) (256.0 * scale);
		int deltaScaleInt = scaleInt - 256;
		int s;

		for (y = field; y < h; y += FIELDS)
		{
			for (x = 0; x < w; x++)
			{
				inPix = src[x];

				r = (inPix >> 16) & 0xff;
				g = (inPix >> 8) & 0xff;
				b = inPix & 0xff;

				max = r;
				if (g>max) max = g;
				if (b>max) max = b;
				if ((max+deltaScaleInt)>255) s = 256+(255-max);
				else s = scaleInt;
				
				r = (r * s) >> 8;  
				g = (g * s) >> 8;  
				b = (b * s) >> 8; 

				dst[x] = (r << 16) | (g << 8) | b;
			}
			src	= (Pixel32 *)((char *)src + FIELDS * fa->src.pitch);
			dst	= (Pixel32 *)((char *)dst + FIELDS * fa->dst.pitch);
		}
	}

	// Temporal softening phase. Adapted from code by Steven Don. 
	unsigned char *src1, *src2;
	long diff, ofs, sum;

	if (mfd->softening != 0) {
		ofs = mfd->size;
		src1 = (unsigned char *) fa->dst.data;
		src2 = (unsigned char *) mfd->old_data;
		do	{
			diff = abs(*src1 - *src2);
			if (diff < mfd->softening) {
				if (diff > (mfd->softening >> 1)) {
					sum = *src1 + *src1 + *src2;
					*src2 = sum / 3;
				}
			} else {
				*src2 = *src1;
			}
			*src1 = *src2; src1++; src2++;
		} while (--ofs);
	}

	if (mfd->correctionPreview) {

		if (mfd->error) msgOut(fa,"Error!",MSGBLACK);
		// output correction ...

		char tmp[20];
		if ( ! mfd->lumWindow.getInterlaced() ) {
			sprintf(tmp,"%+6.3f",scale-1.0);
			msgOut(5,5,fa,tmp,color);
		} else {
			sprintf(tmp,"a: %+6.3f",saveScale[0]-1.0);
			msgOut(5,5,fa,tmp,color);
			sprintf(tmp,"b: %+6.3f",saveScale[1]-1.0);
			msgOut(5,10+MSG_H,fa,tmp,color);
		}
	}

	return 0;
}

extern "C" __declspec(dllexport) int VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" __declspec(dllexport) void VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_tutorial;

int VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if ((fd_tutorial = ff->addFilter(fm, &filterDef_tutorial, sizeof(FilterDefinition))) == 0)
		return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}


void VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_tutorial);
}


int MyInitProc(FilterActivation *fa, const FilterFunctions *ff) {

	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->window = 25;
	mfd->softening = 0; // default no softening !
	mfd->error = false;
	mfd->correctionPreview = false;
	mfd->lumWindow.setMaxCorrection(15);

	char prog[MAX_PATH]; // path[MAX_PATH]
	LPTSTR ptr;
	char	folderPath[MAX_PATH];
 
	GetModuleFileName(NULL, prog, MAX_PATH);
	GetFullPathName(prog, MAX_PATH, folderPath, &ptr);
	*ptr = 0; // ???

	strcat(folderPath, "plugins\\tmp\\");

	if (fopen(folderPath,"r") == NULL) {
		// folder do not exist
		mkdir(folderPath);
	}

	strcpy(mfd->processFileName, folderPath);
	strcat(mfd->processFileName, DEFAULTFILENAME);

	return 0;
}


OPENFILENAME prepOpFlNm(char * filename, HWND hdlg) {

	char filter[] = "All Files(*,*)\0*.*\0Text Files(*.txt)\0*.txt\0\0";

	OPENFILENAME ofname;
	ZeroMemory(&ofname,sizeof(ofname));

	ofname.hwndOwner = hdlg; 
	ofname.hInstance = NULL; 
	ofname.lStructSize = sizeof(ofname);
	ofname.lpstrFilter =  filter;
	ofname.nFilterIndex = 2; 
	ofname.lpstrDefExt = "txt";
	ofname.Flags= OFN_HIDEREADONLY|OFN_FILEMUSTEXIST|OFN_EXPLORER; //|OFN_CREATEPROMPT;
	ofname.nMaxFile = sizeof(char)*MAX_PATH;
	ofname.lpstrFile = filename;
	return ofname;
}


bool tryToLoadFile (char * fileName, HWND hdlg, MyFilterData * mfd) {

	try {
		mfd->lumWindow.readHeader(fileName,false); 
		SetDlgItemText(hdlg, IDC_SOURCE, ( (mfd->lumWindow.getInterlaced()) ? ("interlaced") : ("progressive") ));
		SetDlgItemInt(hdlg, IDC_TOTAL_FRAMES, mfd->lumWindow.getTotalFramesCount(),FALSE);
		SetDlgItemInt(hdlg, IDC_SUGG_WINDOW, mfd->lumWindow.getWindowSuggested(),FALSE);
		return true;
	} catch(char*) {
		SetDlgItemText(hdlg, IDC_SOURCE, ( "undefined" ));
		SetDlgItemInt(hdlg, IDC_TOTAL_FRAMES, 0, FALSE);
		SetDlgItemInt(hdlg, IDC_SUGG_WINDOW, 0, FALSE);
		return false;
	}	
}


BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	char filter[] = "All Files(*,*)\0*.*\0Text Files(*.txt)\0*.txt\0\0";

	switch(msg) {

		case WM_INITDIALOG:

			SetWindowLong(hdlg, DWL_USER, lParam);
			mfd = (MyFilterData *)lParam;

			SetDlgItemInt(hdlg, IDC_WINDOW, mfd->window, FALSE);
			SetDlgItemInt(hdlg, IDC_SOFTENING, mfd->softening, FALSE);
			SetDlgItemText(hdlg, IDC_PROC_FILENAME2, mfd->processFileName);
			CheckDlgButton(hdlg, IDC_CORR_PREVIEW, mfd->correctionPreview ? BST_CHECKED : BST_UNCHECKED);
			SetDlgItemInt(hdlg, IDC_MAX_ADAPTION, mfd->lumWindow.getMaxCorrection(),FALSE);

			tryToLoadFile (mfd->processFileName, hdlg, mfd);

			return TRUE;

		case WM_COMMAND:
			switch(LOWORD(wParam)) {
			case IDOK:
				try {
					mfd->lumWindow.readHeader(mfd->processFileName, false);
					EndDialog(hdlg, 0);
					return TRUE;
				} catch(char*) {						
					// processing file doesn't exits
					SetDlgItemText(hdlg, IDC_PROC_FILENAME2, "");
				}
				break;

			case IDHELP:
				{
				char prog[256];
				char path[256];
				LPTSTR ptr;

				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\DeflickProc.htm");
				ShellExecute(hdlg, "open", path, NULL, NULL, SW_SHOWNORMAL);
				return TRUE;
				}

			case IDCANCEL:
				EndDialog(hdlg, 1);
				return TRUE;

			case IDC_WINDOW:
				mfd->window = GetDlgItemInt(hdlg, IDC_WINDOW, &mfd->window, FALSE);
				if (mfd->window > MAX_KERNEL) {
					mfd->window = MAX_KERNEL;
					SetDlgItemInt(hdlg, IDC_WINDOW, mfd->window, FALSE);
				} else if (mfd->window < 1) {
					mfd->window = 1;
					SetDlgItemInt(hdlg, IDC_WINDOW, mfd->window, FALSE);
				}
				break;

			case IDC_SOFTENING:
				mfd->softening = GetDlgItemInt(hdlg, IDC_SOFTENING, &mfd->softening, FALSE);
				if (mfd->softening > MAX_SOFTENING)
				{
					mfd->softening = MAX_SOFTENING;
					SetDlgItemInt(hdlg, IDC_SOFTENING, mfd->softening, FALSE);
				}
				break;

			case IDC_PROCFILE_BUTTON: {
				char filename[MAX_PATH];
				strcpy(filename,mfd->processFileName);
				OPENFILENAME ofname = prepOpFlNm(filename,hdlg);
				if ( GetOpenFileName(&ofname) != 0 ) { 

					if ( tryToLoadFile(filename,hdlg,mfd) ) {
						SetDlgItemText(hdlg, IDC_PROC_FILENAME2 ,filename);
						strcpy(mfd->processFileName,filename);			
					}
				}
			}	
				break;

			case IDC_PROC_FILENAME2: {
				char filename[MAX_PATH];
				GetDlgItemText(hdlg, IDC_PROC_FILENAME2, filename, MAX_PATH);
				if ( strcmp(filename,mfd->processFileName) != 0 ) {
					strcpy(mfd->processFileName,filename);
				}
				} break;

			case IDC_CORR_PREVIEW:
				mfd->correctionPreview = !mfd->correctionPreview ;
				break;

			case IDC_MAX_ADAPTION:
				int cc = GetDlgItemInt(hdlg, IDC_MAX_ADAPTION, NULL, FALSE);
				if (cc<1) cc=1;
				else if (cc>99) cc = 99;
				if (cc != mfd->lumWindow.getMaxCorrection()) {
					mfd->lumWindow.setMaxCorrection(cc);
					SetDlgItemInt(hdlg, IDC_MAX_ADAPTION, cc, FALSE);
				}
				break;
			}
			break;
	}

	return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
	MyFilterData *mfd = (MyFilterData *) fa->filter_data;
	MyFilterData mfd_old = *mfd;

    extern void Doit(void);

	if (DialogBoxParam(fa->filter->module->hInstModule,
			MAKEINTRESOURCE(IDD_FILTER), hwnd,
			(DLGPROC) ConfigDlgProc, (LPARAM) mfd))
	{
		*mfd = mfd_old;
		return TRUE;
	} 

	return FALSE;
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

		sprintf(str, " ( wind %d, soft %d, max %d file %s )",
			mfd->window,	mfd->softening,
			mfd->lumWindow.getMaxCorrection(),
			mfd->processFileName
		);
}
