/*
    Logo Filter for VirtualDub -- Add a bitmap logo to a video 
	with selectable position, alpha blending, and transparent color.
	Copyright (C) 1999-2000 Donald A. Graft

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	The author can be contacted at:
	Donald Graft
	neuron2@home.com.
*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <crtdbg.h>
#include <math.h>
#include <stdlib.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
int StartProc(FilterActivation *fa, const FilterFunctions *ff);
int EndProc(FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);
BOOL LoadBitmapFromBMPFile(LPTSTR szFileName, HBITMAP *phBitmap, HPALETTE *phPalette);

///////////////////////////////////////////////////////////////////////////

typedef struct MyFilterData {
	IFilterPreview		*ifp;
	unsigned char		*bitmap, *bits;
	long				bwidth, bheight, bpitch;
	unsigned int		depth;
	char				filename[1024];
	int					animate;
	int					x, y, alpha;
	int                 transparent, xr, xg, xb, tolerance;
	int					nextBMP;
	unsigned long		start, count, loop_count;
	unsigned long		fadeinlen, fadeoutend, fadeoutlen;
	unsigned long		duration, loops;
} MyFilterData;

bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
    char save[255];
	int i, j;

	i = j = 0;
	while(mfd->filename[i])
	{
		if (mfd->filename[i] == '\\')
		{
			save[j++] = '\\';
			save[j++] = '\\';
			i++;
		}
		else
		{
			save[j++] = mfd->filename[i++];
		}
	}
	save[j] = 0;
	_snprintf(buf, buflen, "Config(%d, %d, %d, %d, %d, %d, %d, %d, \"%s\", %d, %d, %d, %d, %d, %d, %d)",
		mfd->x,
		mfd->y,
		mfd->alpha,
		mfd->transparent,
		mfd->xr,
		mfd->xg,
		mfd->xb,
		mfd->tolerance,
		save,
		mfd->animate,
		mfd->start,
		mfd->duration,
		mfd->loops,
		mfd->fadeinlen,
		mfd->fadeoutend,
		mfd->fadeoutlen);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->x				= argv[0].asInt();
	mfd->y				= argv[1].asInt();
	mfd->alpha			= argv[2].asInt();
	mfd->transparent	= !!argv[3].asInt();
	mfd->xr				= argv[4].asInt();
	mfd->xg				= argv[5].asInt();
	mfd->xb				= argv[6].asInt();
	mfd->tolerance		= argv[7].asInt();
	strcpy(mfd->filename, *argv[8].asString());
	mfd->animate		= !!argv[9].asInt();
	mfd->start			= argv[10].asInt();
	mfd->duration		= argv[11].asInt();
	mfd->loops			= argv[12].asInt();
	mfd->fadeinlen		= argv[13].asInt();
	mfd->fadeoutend		= argv[14].asInt();
	mfd->fadeoutlen     = argv[15].asInt();
}

ScriptFunctionDef func_defs[]={
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0iiiiiiiisiiiiiii" },
	{ NULL },
};

CScriptObject script_obj={
	NULL, func_defs
};

struct FilterDefinition filterDef_tutorial = {

	NULL, NULL, NULL,		// next, prev, module
	"logo (1.4)",			// name
	"Overlay a bitmap with alpha blending and transparency.",
							// desc
	"Donald Graft", 		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	InitProc,				// initProc
	NULL,					// deinitProc
	RunProc,				// runProc
	NULL,					// paramProc
	ConfigProc, 			// configProc
	StringProc, 			// stringProc
	StartProc,				// startProc
	EndProc,				// endProc

	&script_obj,			// script_obj
	FssProc,				// fssProc

};

void Load(MyFilterData *mfd)
{
	FILE *fp;
	char buf[6];
	DWORD size;
    char *p;
	int found;
	char path[255];

	/* Load bitmap and get pointer to bitmap bits. */
	mfd->depth = 0;
	strcpy(path, mfd->filename);
	p = path + strlen(path);
	while (p >= path && *p != '.') p--;
	if (*p != '.') return;
	if (mfd->animate)
	{
		found = 0;
		while (*(p-1) >= '0' && *(p-1) <= '9')
		{
			p--;
			found = 1;
			if (mfd->nextBMP == -1) mfd->nextBMP = 0;
		}
		if (found)
		{
			if (mfd->nextBMP != 0)
			{
				sprintf(p, "%04d.bmp", mfd->nextBMP);
			}
			mfd->nextBMP++;
		}
	}
	if ((fp = fopen(path, "rb")) == NULL)
	{
		if (mfd->animate && mfd->nextBMP >= 0)
		{
			mfd->count = 1;
			mfd->nextBMP = -1;
			Load(mfd);
			++mfd->loop_count;
		}
		return;
	}
	fread(buf, 1, 2, fp);
	if (buf[0] != 'B' || buf[1] != 'M') return;
	fread(&size, 1, 4, fp);
	fclose(fp);
	if (mfd->bitmap) free(mfd->bitmap);
	mfd->bitmap = (unsigned char *) malloc((size_t) size);
    if ((fp = fopen(path, "rb")) == NULL)
		return;
	fread(mfd->bitmap, 1, size, fp);
	fclose(fp);
	mfd->bits = mfd->bitmap + *((DWORD *)(mfd->bitmap + 10));
	mfd->bwidth = *((LONG *)(mfd->bitmap + 18));
	mfd->bheight = *((DWORD *)(mfd->bitmap + 22));
	mfd->bpitch = 3 * mfd->bwidth;
	if (mfd->bpitch & 0x3)
	{
		mfd->bpitch = (mfd->bpitch & 0xfffffffc) + 4;
	}
	mfd->depth = *((unsigned int *)(mfd->bitmap + 28));
}

int StartProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	/* Load bitmap and get pointer to bitmap bits. */
	Load(mfd);

	return 0;
}

int EndProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	free(mfd->bitmap);
	mfd->bitmap = NULL;
	return 0;
}

int RunProc(const FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	const PixDim	width = fa->src.w;
	const PixDim	height = fa->src.h;
	Pixel32 *src, *dst;
	int x, y;
	unsigned long frame;
	int r, g, b, rb, gb, bb;
	unsigned char *bp;
	int X = mfd->x;
	int Y = mfd->y;
	double ALPHA = mfd->alpha / 255.0;
	double in = mfd->fadeinlen;
	double out = mfd->fadeoutlen;
	int XR = mfd->xr;
	int XG = mfd->xg;
	int XB = mfd->xb;
	bool doit;

	if (mfd->depth != 24) return(0);
	if (X > width - mfd->bwidth) X = width - mfd->bwidth;
	if (Y > height - mfd->bheight) Y = height - mfd->bheight;

	src = fa->src.data;
	dst = fa->dst.data;
	for (y = 0; y < height; y++)
	{
		for (x = 0; x < width; x++)
		{
			dst[x] = src[x];
		}
		src	= (Pixel *)((char *)src + fa->src.pitch);
		dst	= (Pixel *)((char *)dst + fa->dst.pitch);
	}
	frame = fa->pfsi->lCurrentSourceFrame;
//	frame = fa->pfsi->lCurrentFrame;
	if (frame == 0)
	{
		mfd->count = 0;
		mfd->loop_count = 0;
		mfd->nextBMP = -1;
		Load(mfd);
	}
    if (frame < mfd->start) return 0;
	if (mfd->nextBMP >= 0 && mfd->loops && mfd->loop_count >= mfd->loops) return 0;
	if ((mfd->nextBMP == -1) &&
		(mfd->duration && frame >= mfd->start + mfd->duration)) return 0;
	if ((mfd->nextBMP >= 0) &&
		(mfd->duration && ++mfd->count > mfd->duration))
	{
		mfd->count = 1;
		Load(mfd);
	}
	src	= (Pixel *)((char *)fa->src.data + (height - Y - mfd->bheight) * fa->src.pitch);
	dst	= (Pixel *)((char *)fa->dst.data + (height - Y - mfd->bheight) * fa->dst.pitch);
	for (y = 0; y < mfd->bheight; y++)
	{
		bp = mfd->bits + y * mfd->bpitch;
		for (x = X; x < X + mfd->bwidth; x++)
		{
			r = ((src[x] & 0xff0000) >> 16);
			g = ((src[x] & 0xff00) >> 8);
			b = (src[x] & 0xff);
			bb = *bp++;
			gb = *bp++;
			rb = *bp++;
			if ((mfd->fadeinlen > 0) && (frame >= mfd->start) && (frame < (mfd->start + in)))
			{
				ALPHA = ((frame - mfd->start) / in) * (mfd->alpha / 255.0);
			}
			if ((mfd->fadeoutlen > 0) && (frame > mfd->fadeoutend - out) /*&&
				(frame <= mfd->fadeoutend)*/)
			{
				if (frame < mfd->fadeoutend)
				{
					ALPHA = ((mfd->fadeoutend - frame) / out) * (mfd->alpha / 255.0);
				}
				else
				{
					ALPHA = 0.0;
				}
			}
			doit = false;
			if (mfd->transparent == 0)
				doit = true;
			else if ((mfd->transparent == 1) && (mfd->tolerance == 0) &&
				     (bb != XB || gb != XG || rb != XR))
				doit = true;
			else if (mfd->transparent == 1 && mfd->tolerance != 0)
			{
				if (abs(bb - XB) > mfd->tolerance ||
					abs(gb - XG) > mfd->tolerance ||
					abs(rb - XR) > mfd->tolerance)
					doit = true;
			}
			if (doit == true)
			{
				bb = (int) (ALPHA * bb + (1 - ALPHA) * b);			 
				gb = (int) (ALPHA * gb + (1 - ALPHA) * g);			 
				rb = (int) (ALPHA * rb + (1 - ALPHA) * r);			 
				dst[x] = ((rb << 16) | (gb << 8) | (bb));
			}
		}
		src	= (Pixel *)((char *)src + fa->src.pitch);
		dst	= (Pixel *)((char *)dst + fa->dst.pitch);
	}
	return 0;
}

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_tutorial;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if (!(fd_tutorial = ff->addFilter(fm, &filterDef_tutorial, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_tutorial);
}

int InitProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
   	char prog[256];
	char path[256];
	LPTSTR ptr;

	mfd->x = 0;
	mfd->y = 0;
	mfd->alpha = 128;
	mfd->transparent = 1;
	mfd->animate = 0;
	mfd->xr = 0;
	mfd->xg = 0;
	mfd->xb = 0xff;
	mfd->tolerance = 0;
	mfd->start = mfd->count = mfd->loop_count = 0;
	mfd->fadeinlen = mfd->fadeoutend = mfd->fadeoutlen = 0;
	mfd->duration = 0;
	mfd->loops = 0;
	mfd->nextBMP = -1;

	GetModuleFileName(NULL, prog, 255);
	GetFullPathName(prog, 255, path, &ptr);
	*ptr = 0;
	strcat(path, "plugins\\Logo.bmp");
	strcpy(mfd->filename, path);
	return 0;
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg) {
		case WM_INITDIALOG:
			SetWindowLong(hdlg, DWL_USER, lParam);
			mfd = (MyFilterData *)lParam;

			CheckDlgButton(hdlg, IDC_ENABLE, mfd->transparent ? BST_CHECKED : BST_UNCHECKED);
			CheckDlgButton(hdlg, IDC_ANIMATE, mfd->animate ? BST_CHECKED : BST_UNCHECKED);
			SetWindowText(GetDlgItem(hdlg, IDC_FILENAME), mfd->filename);

			SendMessage(GetDlgItem(hdlg, IDC_XSLIDER), TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 1200));
			SendMessage(GetDlgItem(hdlg, IDC_XSLIDER), TBM_SETPOS, (WPARAM)TRUE, mfd->x);
			SetDlgItemInt(hdlg, IDC_XOFF, mfd->x, FALSE);
			SendMessage(GetDlgItem(hdlg, IDC_YSLIDER), TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 800));
			SendMessage(GetDlgItem(hdlg, IDC_YSLIDER), TBM_SETPOS, (WPARAM)TRUE, mfd->y);
			SetDlgItemInt(hdlg, IDC_YOFF, mfd->y, FALSE);
			
			SetDlgItemInt(hdlg, IDC_RED, mfd->xr, FALSE);
			SetDlgItemInt(hdlg, IDC_ALPHA, mfd->alpha, FALSE);
			SetDlgItemInt(hdlg, IDC_GREEN, mfd->xg, FALSE);
			SetDlgItemInt(hdlg, IDC_BLUE, mfd->xb, FALSE);
			SetDlgItemInt(hdlg, IDC_TOLERANCE, mfd->tolerance, FALSE);
			SetDlgItemInt(hdlg, IDC_START, mfd->start, FALSE);
			SetDlgItemInt(hdlg, IDC_DURATION, mfd->duration, FALSE);
			SetDlgItemInt(hdlg, IDC_LOOPS, mfd->loops, FALSE);
			SetDlgItemInt(hdlg, IDC_FADEINLEN, mfd->fadeinlen, FALSE);
			SetDlgItemInt(hdlg, IDC_FADEOUTEND, mfd->fadeoutend, FALSE);
			SetDlgItemInt(hdlg, IDC_FADEOUTLEN, mfd->fadeoutlen, FALSE);
			mfd->ifp->InitButton(GetDlgItem(hdlg, IDPREVIEW));
			return TRUE;

		case WM_HSCROLL:
			if ((HWND) lParam == GetDlgItem(hdlg, IDC_XSLIDER))
			{
				int x = SendMessage(GetDlgItem(hdlg, IDC_XSLIDER), TBM_GETPOS, 0, 0);
				if (x != mfd->x)
				{
					mfd->x = x;
					SetDlgItemInt(hdlg, IDC_XOFF, mfd->x, FALSE);
					mfd->ifp->RedoFrame();
				}
			}
		case WM_VSCROLL:
			if ((HWND) lParam == GetDlgItem(hdlg, IDC_YSLIDER))
			{
				int y = SendMessage(GetDlgItem(hdlg, IDC_YSLIDER), TBM_GETPOS, 0, 0);
				if (y != mfd->y)
				{
					mfd->y = y;
					SetDlgItemInt(hdlg, IDC_YOFF, mfd->y, FALSE);
					mfd->ifp->RedoFrame();
				}
			}
			break;

		case WM_COMMAND:
			switch(LOWORD(wParam)) {

			case IDPREVIEW:
				mfd->ifp->Toggle(hdlg);
				break;

			case IDOK:
				EndDialog(hdlg, 0);
				return TRUE;
			case IDHELP:
				{
				char prog[256];
				char path[256];
				LPTSTR ptr;

				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\Logo.html");
				ShellExecute(hdlg, "open", path, NULL, NULL, SW_SHOWNORMAL);
				return TRUE;
				}

			case IDC_BROWSE:
				{
				OPENFILENAME ofn;
				mfd->filename[0] = NULL;
				ofn.lStructSize = sizeof(OPENFILENAME);
				ofn.hwndOwner = hdlg;
				ofn.hInstance = NULL;
				ofn.lpTemplateName = NULL;
				ofn.lpstrFilter = "All Files\0*.*\0\0";
				ofn.lpstrCustomFilter = NULL;
				ofn.nMaxCustFilter = 0;
				ofn.nFilterIndex = 1;
				ofn.lpstrFile = mfd->filename;
				ofn.nMaxFile = 1024;
				ofn.lpstrFileTitle = NULL;
				ofn.nMaxFileTitle = 0;
				ofn.lpstrInitialDir = NULL;
				ofn.lpstrTitle = "Choose Logo Bitmap File";
				ofn.Flags = OFN_CREATEPROMPT;
				ofn.nFileOffset = 0;
				ofn.nFileExtension = 0;
				ofn.lpstrDefExt = NULL;
				ofn.lCustData = 0;
				ofn.lpfnHook = NULL;
				GetOpenFileName(&ofn);
				if (mfd->filename[0] != 0)
				{
					SetWindowText(GetDlgItem(hdlg, IDC_FILENAME), mfd->filename);
					Load(mfd);
					mfd->ifp->UndoSystem();
					mfd->ifp->RedoSystem();
				}
				break;
				}


			case IDC_ALPHA:
				mfd->alpha = GetDlgItemInt(hdlg, IDC_ALPHA, 0, FALSE);
				mfd->ifp->RedoFrame();
				break;
			case IDC_ENABLE:
				mfd->transparent = !!IsDlgButtonChecked(hdlg, IDC_ENABLE);
				mfd->ifp->RedoFrame();
				break;
			case IDC_ANIMATE:
				mfd->animate = !!IsDlgButtonChecked(hdlg, IDC_ANIMATE);
				mfd->ifp->RedoFrame();
				break;
			case IDC_RED:
				mfd->xr = GetDlgItemInt(hdlg, IDC_RED, 0, FALSE);
				mfd->ifp->RedoFrame();
				break;
			case IDC_GREEN:
				mfd->xg = GetDlgItemInt(hdlg, IDC_GREEN, 0, FALSE);
				mfd->ifp->RedoFrame();
				break;
			case IDC_BLUE:
				mfd->xb = GetDlgItemInt(hdlg, IDC_BLUE, 0, FALSE);
				mfd->ifp->RedoFrame();
				break;
			case IDC_TOLERANCE:
				mfd->tolerance = GetDlgItemInt(hdlg, IDC_TOLERANCE, 0, FALSE);
				mfd->ifp->RedoFrame();
				break;
			case IDC_START:
				mfd->start = GetDlgItemInt(hdlg, IDC_START, 0, FALSE);
				mfd->ifp->RedoFrame();
				break;
			case IDC_FADEINLEN:
				mfd->fadeinlen = GetDlgItemInt(hdlg, IDC_FADEINLEN, 0, FALSE);
				mfd->ifp->RedoFrame();
				break;
			case IDC_FADEOUTEND:
				mfd->fadeoutend = GetDlgItemInt(hdlg, IDC_FADEOUTEND, 0, FALSE);
				mfd->ifp->RedoFrame();
				break;
			case IDC_FADEOUTLEN:
				mfd->fadeoutlen = GetDlgItemInt(hdlg, IDC_FADEOUTLEN, 0, FALSE);
				mfd->ifp->RedoFrame();
				break;
			case IDC_DURATION:
				mfd->duration = GetDlgItemInt(hdlg, IDC_DURATION, 0, FALSE);
				mfd->ifp->RedoFrame();
				break;
			case IDC_LOOPS:
				mfd->loops = GetDlgItemInt(hdlg, IDC_LOOPS, 0, FALSE);
				mfd->ifp->RedoFrame();
				break;
			case IDC_XPLUS:
				if (mfd->x < 1200)
				{
					mfd->x += 1;
					SetDlgItemInt(hdlg, IDC_XOFF, mfd->x, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_XSLIDER), TBM_SETPOS, (WPARAM)TRUE, mfd->x);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_XMINUS:
				if (mfd->x > 0)
				{
					mfd->x -= 1;
					SetDlgItemInt(hdlg, IDC_XOFF, mfd->x, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_XSLIDER), TBM_SETPOS, (WPARAM)TRUE, mfd->x);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_YPLUS:
				if (mfd->y < 800)
				{
					mfd->y += 1;
					SetDlgItemInt(hdlg, IDC_YOFF, mfd->y, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_YSLIDER), TBM_SETPOS, (WPARAM)TRUE, mfd->y);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_YMINUS:
				if (mfd->y > 0)
				{
					mfd->y -= 1;
					SetDlgItemInt(hdlg, IDC_YOFF, mfd->y, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_YSLIDER), TBM_SETPOS, (WPARAM)TRUE, mfd->y);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDCANCEL:
				EndDialog(hdlg, 1);
				return TRUE;
			}
			break;
	}

	return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	MyFilterData mfd_old = *mfd;
	int ret;

	mfd->ifp = fa->ifp;

	if (DialogBoxParam(fa->filter->module->hInstModule, MAKEINTRESOURCE(IDD_FILTER),
		hwnd, ConfigDlgProc, (LPARAM)mfd))
	{
		*mfd = mfd_old;
		ret = TRUE;
	}
    else
	{
		ret = FALSE;
	}
	return(ret);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
}
