// VirtualDub filter: deinterlace - PAL Interpolate
// by tHE fISH (fish@everymail.net)
//
// heavily based on code
// by Gunnar Thalin (guth@home.se)

#include <windows.h>
#include <stdio.h>
#include <commctrl.h>
#include <math.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

#define maxwidth 2000

///////////////////////////////////////////////////////////////////////////

int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

typedef struct MyFilterData {
    bool     bSwap;
	int		 thresh;
	bool     showInter;
	int      minwidth;
	int      edge;
	int		 edgeth;
	bool	 blend;
} MyFilterData;


bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	_snprintf(buf, buflen, "Config(%i,%i,%i,%i,%i,%i,%i)", mfd->bSwap, mfd->thresh, mfd->showInter, mfd->minwidth, mfd->edge, mfd->edgeth, mfd->blend);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->bSwap = !!argv[0].asInt();
	mfd->thresh = argv[1].asInt();
	mfd->showInter = !!argv[2].asInt();
	mfd->minwidth = argv[3].asInt();
	mfd->edge = argv[4].asInt();
	mfd->edgeth = argv[5].asInt();
	mfd->blend = !!argv[6].asInt();

}

ScriptFunctionDef func_defs[]={
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0iiiiiii" },
	{ NULL },
};

CScriptObject script_obj={
	NULL, func_defs
};


struct FilterDefinition filterDef = {

	NULL, NULL, NULL,			// next, prev, module
	"deinterlace - PAL/Interpolate v1.0b1",	// name
	"Almost the same as Deinterlace PAL by Gunnar Thalin, except that this filter checks the output for remaining interlace lines which are then removed by interpolation.",	// desc
	"tHE fISH",		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	InitProc,		// initProc
	NULL,			// deinitProc
	RunProc,		// runProc
	ParamProc,		// paramProc
	ConfigProc,		// configProc
	StringProc,		// stringProc
	NULL,			// startProc
	NULL,			// endProc

	&script_obj,	// script_obj
	FssProc,		// fssProc

};

///////////////////////////////////////////////////////////////////////////

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd;

///////////////////////////////////////////////////////////////////////////

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
    if (!(fd = ff->addFilter(fm, &filterDef, sizeof(FilterDefinition))))
        return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

    return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
    ff->removeFilter(fd);
}

///////////////////////////////////////////////////////////////////////////

int InitProc(FilterActivation *fa, const FilterFunctions *ff) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->bSwap = false;
	mfd->thresh = 14;
	mfd->showInter = false;
	mfd->minwidth = 2;
    mfd->edge = 18;
    mfd->edgeth = 30;
	mfd->blend = false;

	return 0;
}

int RunProc(const FilterActivation *fa, const FilterFunctions *ff) {
	Pixel32 *src, *dst, *last;
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	src = (Pixel32 *)fa->src.data;
	dst = (Pixel32 *)fa->dst.data;
	last = (Pixel32 *)fa->last->data;
	int iOddEven = mfd->bSwap ? 0 : 1;
	int thresh = mfd->thresh;

	for(int y = 0; y < fa->src.h; y++) {
		bool nstat [maxwidth];
		if((y % 2) == iOddEven)	{	 // Take line from the previous frame.
			Pixel32 *pprevrow = (y - 1 >= 0 ? (Pixel32 *)((char *)src - fa->src.pitch) : src);
			Pixel32 *pnextrow = (Pixel32 *)(y + 1 < fa->src.h ? (Pixel32 *)((char *)src + fa->src.pitch) : pprevrow);
			if(pprevrow == src)
				pprevrow = pnextrow;
			Pixel32 prevrow, nextrow, prpp, prnp, nrpp, nrnp;
			int nodr = 0; int nodg = 0; int nodb = 0; int ppan = 0; Pixel32 *wert = 0;
			Pixel32 intp [maxwidth];
			Pixel32 deip [maxwidth];
			bool stat [maxwidth];
			for(int x = 0; x < fa->src.w; x++)
			{
				prevrow = *pprevrow++;
				nextrow = *pnextrow++;

				Pixel32 ifv=0;
				
				int pR = ((prevrow >> 16) & 0xff);
				int pG = ((prevrow >> 8) & 0xff);
				int pB = (prevrow & 0xff);
				int nR = ((nextrow >> 16) & 0xff);
				int nG = ((nextrow >> 8) & 0xff);
				int nB = (nextrow & 0xff);
					
				wert = last + x;
				nodr=nodg=nodb=0;
					
				int wR = (*wert >> 16) & 0xff;
				int wG = (*wert >> 8) & 0xff;
				int wB = *wert & 0xff;

				if(fa->pfsi->lCurrentFrame == 0) { wR=0; wG=0; wB=0; nodr=1;nodg=1;nodb=1;}
				if (mfd->blend) nodr++;	
				
                if ( ((pR-thresh > wR) && (nR-thresh > wR)) || ((pR+thresh < wR) && (nR+thresh < wR)) ) nodr++;
				else if ( ((pG-thresh > wG) && (nG-thresh > wG)) || ((pG+thresh < wG) && (nG+thresh < wG)) ) nodg++;
				else if ( ((pB-thresh > wB) && (nB-thresh > wB)) || ((pB+thresh < wB) && (nB+thresh < wB)) ) nodb++;
				if (stat[x-1] == 1) {
					Pixel32 cpx = intp[x-1];
					int ccR = (cpx >> 16) & 0xff;
					int ccG = (cpx >> 8) & 0xff;
					int ccB = cpx & 0xff;
					if ( (ccR-thresh > wR) && (ccR+thresh < wR) && (ccG-thresh > wG) && (ccG+thresh < wG) && (ccB-thresh > wB) && (ccB+thresh < wB) ) {
						ppan++;
					}
				}

				if ((nodr > 0) || (nodg > 0) || (nodb > 0) || (ppan > 0)) {
					// use pixel above and below (*4) + their neighbours (*1) for interpolation
					prpp = *pprevrow-1;
					prnp = *pprevrow+1;
					nrpp = *pnextrow-1;
					nrnp = *pnextrow+1;
					int ppR = ((prpp >> 16) & 0xff); int pnR = ((prnp >> 16) & 0xff);
					int ppG = ((prpp >> 8) & 0xff); int pnG = ((prnp >> 8) & 0xff);
					int ppB = (prpp & 0xff); int pnB = (prnp & 0xff);
					int npR = ((nrpp >> 16) & 0xff); int nnR = ((nrnp >> 16) & 0xff);
					int npG = ((nrpp >> 8) & 0xff); int nnG = ((nrnp >> 8) & 0xff);
					int npB = (nrpp & 0xff); int nnB = (nrnp & 0xff);
					if (!(mfd->blend)) {
						if (mfd->edge > 0) {
						   Pixel32 curpix = *(src + x);	
						   int cR=((curpix >> 16) & 0xff);
						   int cG=((curpix >> 8) & 0xff);
	   					   int cB=(curpix & 0xff);
						   int Spp=ppR+ppG+ppB;
	                       int Spt=pR+pG+pB;
		                   int Spn=pnR+pnG+pnB;
			               int Snp=npR+npG+npB;
				           int Snt=nR+nG+nB;
	                       int Snn=nnR+nnG+nnB;
						   int MW = (Spp+(4*Spt)+Spn+Snp+(4*Snt)+Snn) / 12;
						   int zR=0; int zG=0; int zB=0; int divider=0;
						   if ( (Spp>(MW-(5*mfd->edge))) && (Spp<(MW+(5*mfd->edge))) ) { zR=zR+ppR; zG=zG+ppG; zB=zB+ppB; divider+=1; }
						   if ( (Spt>(MW-(5*mfd->edge))) && (Spt<(MW+(5*mfd->edge))) ) { zR=zR+(4*pR); zG=zG+(4*pG); zB=zB+(4*pB); divider+=4; }
						   if ( (Spn>(MW-(5*mfd->edge))) && (Spn<(MW+(5*mfd->edge))) ) { zR=zR+pnR; zG=zG+pnG; zB=zB+pnB; divider=divider+1; }
						   if ( (Snp>(MW-(5*mfd->edge))) && (Snp<(MW+(5*mfd->edge))) ) { zR=zR+npR; zG=zG+npG; zB=zB+npB; divider+=1; }
						   if ( (Snt>(MW-(5*mfd->edge))) && (Snt<(MW+(5*mfd->edge))) ) { zR=zR+(4*nR); zG=zG+(4*nG); zB=zB+(4*nB); divider+=4; }
						   if ( (Snn>(MW-(5*mfd->edge))) && (Snn<(MW+(5*mfd->edge))) ) { zR=zR+nnR; zG=zG+nnG; zB=zB+nnB; divider=divider+1; }
						   if (divider>0) {
//								if ( (((zB/divider)-mfd->edgeth) < cB) && (((zB/divider)+mfd->edgeth) > cB) && (((zG/divider)-mfd->edgeth) < cG) && (((zG/divider)+mfd->edgeth) > cB) && (((zR/divider)-mfd->edgeth) < cR) && (((zR/divider)+mfd->edgeth) > cR) ) {
								if ( ( ( ((cR > (pR-mfd->edgeth)) && (cR < (pR+mfd->edgeth))) || ((cR < (nR+mfd->edgeth)) && (cR > (nR-mfd->edgeth))) ) &&
									   ( ((cG > (pG-mfd->edgeth)) && (cG < (pG+mfd->edgeth))) || ((cG < (nG+mfd->edgeth)) && (cG > (nG-mfd->edgeth))) ) &&
									   ( ((cB > (pB-mfd->edgeth)) && (cB < (pB+mfd->edgeth))) || ((cB < (nB+mfd->edgeth)) && (cB > (nB-mfd->edgeth))) ) ) && (divider<12) ) {
									ifv = curpix;
								} else {
									ifv = ( (int)(zB/divider) + (int)((zG/divider) << 8) + (int)((zR/divider) << 16) );
								}
						   } else {
							   // edge detection panic !
							   // most probably we have a hard horizontal edge.
								Pixel32 intpix = ( ((pB+nB)/2) + 
												  (((pG+nG)/2) << 8) +
												  (((pR+nR)/2) << 16) );
								if ( ( ( ((cR > (pR-mfd->edgeth)) && (cR < (pR+mfd->edgeth))) || ((cR < (nR+mfd->edgeth)) && (cR > (nR-mfd->edgeth))) ) &&
									   ( ((cG > (pG-mfd->edgeth)) && (cG < (pG+mfd->edgeth))) || ((cG < (nG+mfd->edgeth)) && (cG > (nG-mfd->edgeth))) ) &&
									   ( ((cB > (pB-mfd->edgeth)) && (cB < (pB+mfd->edgeth))) || ((cB < (nB+mfd->edgeth)) && (cB > (nB-mfd->edgeth))) ) ) ||
   									   ( (cR > ((pR+nR)/2)-mfd->edgeth) && (cR < ((pR+nR)/2)+mfd->edgeth) && (cG > ((pG+nG)/2)-mfd->edgeth) && (cG < ((pG+nG)/2)+mfd->edgeth) && (cB > ((pB+nB)/2)-mfd->edgeth) && (cB < ((pB+nB)/2)+mfd->edgeth) ) ) {
									ifv = curpix;
								} else {
									ifv = intpix;
								}
						   }
						} else {
							ifv = (
									( ( (((pB+nB)/2)*4) + ((ppB+npB)/2) + ((pnB+nnB)/2) )/6 )+
									( ( ((((pG+nG)/2)*4) + ((ppG+npG)/2) + ((pnG+nnG)/2) )/6) << 8)+
									( ( ((((pR+nR)/2)*4) + ((ppR+npR)/2) + ((pnR+nnR)/2) )/6) << 16)
								);
						}
						if (mfd->showInter)	ifv = 0xffffff;
						intp[x]=ifv;
						stat[x]=1;
					} else { // blend
						Pixel32 curpix = *(src + x);	
						int cR=((curpix >> 16) & 0xff);
						int cG=((curpix >> 8) & 0xff);
	   					int cB=(curpix & 0xff);
						int dR, dG, dB;
						dR=(pR+nR+(2*cR))/4;
						dG=(pG+nG+(2*cG))/4;
						dB=(pB+nB+(2*cB))/4;
						if (mfd->showInter) {
							intp[x]=0xffffff;
						} else {
							intp[x]=dB + (dG << 8) + (dR << 16);
						}
						if ( ((pR-thresh < cR) && (nR-thresh < cR)) && ((pR+thresh > cR) && (nR+thresh > cR)) &&
							 ((pG-thresh < cG) && (nG-thresh < cG)) && ((pG+thresh > cG) && (nG+thresh > cG)) &&
							 ((pB-thresh < cB) && (nB-thresh < cB)) && ((pB+thresh > cB) && (nB+thresh > cB)) &&
							 ( (dR-thresh < wR) && (dR+thresh > wR) && (dG-thresh < wG) && (dG+thresh > wG) && (dB-thresh < wB) && (dB+thresh > wB)) ) {
							stat[x]=0;
						} else {
							stat[x]=1;
						}
					}
				} else {
					intp[x]=*wert;
					stat[x]=0;
				}
				deip[x]=*wert;
			}
			for(x = 0; x < fa->src.w; x++) {
				if (stat[x] == 1) {
					if ((mfd->minwidth == 1)) {
						*(dst+x)=intp[x];
						nstat[x]=1;
					} else if (mfd->minwidth == 2) {
						if ((stat[x-1] == 1) || (stat[x+1] == 1)) {
							*(dst+x)=intp[x];
							nstat[x]=1;
						} else {
							*(dst+x)=deip[x];
							nstat[x]=0;
						}
					} else if (mfd->minwidth == 3) {
						if ( ((stat[x-2] == 1) && (stat[x-1] == 1)) || ((stat[x-1] == 1) && (stat[x+1] == 1)) || ((stat[x+1] == 1) && (stat[x+2] == 1)) ) {
							*(dst+x)=intp[x];
							nstat[x]=1;
						} else {
							*(dst+x)=deip[x];
							nstat[x]=0;
						}
					} else if (mfd->minwidth == 4) {
						if ( ((stat[x-3] == 1) && (stat[x-2] == 1) && (stat[x-1] == 1)) ||
							 ((stat[x-2] == 1) && (stat[x-1] == 1) && (stat[x+1] == 1)) ||
							 ((stat[x-1] == 1) && (stat[x+1] == 1) && (stat[x+2] == 1)) ||
							 ((stat[x+1] == 1) && (stat[x+2] == 1) && (stat[x+3] == 1)) ) {
							*(dst+x)=intp[x];
							nstat[x]=1;
						} else {
							*(dst+x)=deip[x];
							nstat[x]=0;
						}
					}
				} else {
					*(dst+x)=deip[x];
					nstat[x]=0;
				}
			}
		} else
			// Take line from the this frame.
			if (mfd->blend) {
				Pixel32 *pprevrow = (y - 1 >= 0 ? (Pixel32 *)((char *)src - fa->src.pitch) : src);
				Pixel32 *pthisrow = (Pixel32 *)((char *)src);
				Pixel32 *pnextrow = (Pixel32 *)(y + 1 < fa->src.h ? (Pixel32 *)((char *)src + fa->src.pitch) : pprevrow);
				if(pprevrow == src)	pprevrow = pnextrow;
				Pixel32 prevrow, nextrow, thisrow;
				for(int x = 0; x < fa->src.w; x++) {
					prevrow = *pprevrow++;
					thisrow = *pthisrow++;
					nextrow = *pnextrow++;
					if (nstat[x] == 0) {
						*(dst+x)=thisrow;
					} else {
						int dR, dG, dB;
						int pR = ((prevrow >> 16) & 0xff);
						int pG = ((prevrow >> 8) & 0xff);
						int pB = (prevrow & 0xff);
						int cR=((thisrow >> 16) & 0xff);
						int cG=((thisrow >> 8) & 0xff);
	   					int cB=(thisrow & 0xff);
						int nR = ((nextrow >> 16) & 0xff);
						int nG = ((nextrow >> 8) & 0xff);
						int nB = (nextrow & 0xff);
						dR=(pR+nR+(2*cR))/4;
						dG=(pG+nG+(2*cG))/4;
						dB=(pB+nB+(2*cB))/4;
						*(dst+x)=dB + (dG << 8) + (dR << 16);
					}
				}
			} else {
				memcpy(dst, src, fa->src.w * sizeof(Pixel32));
			}

		src = (Pixel32 *)((char *)src + fa->src.pitch);
		dst = (Pixel32 *)((char *)dst + fa->dst.pitch);
		if(last != NULL)
			last = (Pixel32 *)((char *)last + fa->last->pitch);
	}

	return 0;
}

long ParamProc(FilterActivation *fa, const FilterFunctions *ff) {
    return FILTERPARAM_NEEDS_LAST | FILTERPARAM_SWAP_BUFFERS;
}


BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
    MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

    switch(msg) {
        case WM_INITDIALOG:
            SetWindowLong(hdlg, DWL_USER, lParam);
            mfd = (MyFilterData *)lParam;

            CheckDlgButton(hdlg, IDC_SWAP, mfd->bSwap?BST_CHECKED:BST_UNCHECKED);
            CheckDlgButton(hdlg, IDC_INTERPOL, mfd->showInter?BST_CHECKED:BST_UNCHECKED);
            CheckDlgButton(hdlg, IDC_BLEND, mfd->blend?BST_CHECKED:BST_UNCHECKED);
            SendDlgItemMessage (hdlg, IDC_WIDTH, TBM_SETTICFREQ, 1, 0);
            SendDlgItemMessage (hdlg, IDC_WIDTH, TBM_SETRANGE, false, MAKELONG (1, 4));
            SendDlgItemMessage (hdlg, IDC_WIDTH, TBM_SETPOS, true, mfd->minwidth);
            char widtht [35];
            sprintf (widtht, "min. width for interpolation: %d", mfd->minwidth);
            SetDlgItemText (hdlg, IDC_WIDTHTEXT, widtht);
			char str[3];
			sprintf(str, "%i", mfd->thresh);
			SetDlgItemText(hdlg, ID_THRESH, str);
			sprintf(str, "%i", mfd->edge);
			SetDlgItemText(hdlg, IDC_EDGE, str);
			sprintf(str, "%i", mfd->edgeth);
			SetDlgItemText(hdlg, IDC_EDGETH, str);

            return TRUE;

       case WM_NOTIFY:
            {
                int nwidth = SendDlgItemMessage (hdlg, IDC_WIDTH, TBM_GETPOS, 0, 0);
                char widtht [35];
                sprintf (widtht, "min. width for interpolation: %d", nwidth);
                SetDlgItemText (hdlg, IDC_WIDTHTEXT, widtht);
			}

        case WM_COMMAND:
            switch(LOWORD(wParam)) {
            case IDOK:
                mfd->bSwap = !!IsDlgButtonChecked(hdlg, IDC_SWAP);
				mfd->showInter = !!IsDlgButtonChecked(hdlg, IDC_INTERPOL);
				mfd->blend = !!IsDlgButtonChecked(hdlg, IDC_BLEND);
                mfd->minwidth = SendDlgItemMessage (hdlg, IDC_WIDTH, TBM_GETPOS, 0, 0);
				mfd->edge = !!IsDlgButtonChecked(hdlg, IDC_EDGE);
 				char str[3];
				GetDlgItemText(hdlg, ID_THRESH, str, 10);
				mfd->thresh = atoi(str);
				GetDlgItemText(hdlg, IDC_EDGE, str, 10);
				mfd->edge = atoi(str);
				GetDlgItemText(hdlg, IDC_EDGETH, str, 10);
				mfd->edgeth = atoi(str);
                EndDialog(hdlg, 0);
                return TRUE;
            case IDCANCEL:
                EndDialog(hdlg, 1);
                return FALSE;
			}
            break;
    }

    return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd) {
    return DialogBoxParam(fa->filter->module->hInstModule,
            MAKEINTRESOURCE(IDD_FILTER_DEINTERLACE_PALINTERPOLATION), hwnd,
            (DLGPROC)ConfigDlgProc, (LPARAM)fa->filter_data);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
    char stat[3]; sprintf (stat, "no");
	if (((MyFilterData *)(fa->filter_data))->showInter) sprintf(stat, "yes");
    char sta2[2]; sprintf (sta2, "");
	if (((MyFilterData *)(fa->filter_data))->blend) sprintf(sta2, "B,");
	if(((MyFilterData *)(fa->filter_data))->bSwap)
		sprintf(str, " (%sswap,Th:%i,Wh:%s,Wi:%i,Ed:%i,%i)", sta2, ((MyFilterData *)(fa->filter_data))->thresh, stat, ((MyFilterData *)(fa->filter_data))->minwidth, ((MyFilterData *)(fa->filter_data))->edge, ((MyFilterData *)(fa->filter_data))->edgeth);
	else
		sprintf(str, " (%snormal,Th:%i,Wh:%s,Wi:%i,Ed:%i,%i)", sta2, ((MyFilterData *)(fa->filter_data))->thresh, stat, ((MyFilterData *)(fa->filter_data))->minwidth, ((MyFilterData *)(fa->filter_data))->edge, ((MyFilterData *)(fa->filter_data))->edgeth);
}
