/*
    Halftone Filter for VirtualDub -- Create large single coloured
      blocks based on the average of the input pixels.
      Copyright (C) 2001 Tom Ford

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    The author can be contacted at:
      Tom Ford
      flend@compsoc.net

    This work includes code from:
      Colorize Filter for VirtualDub -- Change a video to a monochrome video
        with selectable base color. Copyright (C) 1999-2000 Donald A. Graft

      By 
        Donald Graft
        neuron2@home.com.
*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <crtdbg.h>
#include <math.h>
#include <stdlib.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

#define BLOCK_SIZE_MAX 100
#define INTENSITY_MAX 20

int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int StartProc(FilterActivation *fa, const FilterFunctions *ff);
int StopProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

void RGBtoHSL(Pixel32 src, double *H, double *S, double *L);
void HLStoRGB(double H, double L, double S, int *r, int *g, int *b);
void Pixel32RGB(Pixel32 src, int *r, int *g, int *b);
double Lum(Pixel32 src);

void AntiAliasCircle(int x_l, int y_b, unsigned int block_size, unsigned int radius, unsigned int mag, bool darker, const FilterActivation *fa);
void AntiAliasCopyBlock(const FilterActivation *fa, Pixel32 *dst, Pixel32 *src, unsigned int src_block_size, unsigned int dst_block_size, unsigned int mag, int radius);
void CopyBlockLightness(const FilterActivation *fa, Pixel32 *src, int x_l, int y_b, unsigned int block_size, bool darker);

void PlotCircle(const FilterActivation *fa, Pixel32 *src, unsigned int block_size, unsigned int radius);
void Plot8CirclePoints(const FilterActivation *fa, Pixel32 *src, unsigned int block_size, int x_orig, int y_orig, int x, int y);
void PlotPoint(const FilterActivation *fa, Pixel32 *src, unsigned int block_size, unsigned int x, unsigned int y);



typedef struct FilterDataStruct {

  HBRUSH spot_brush;
  HBRUSH background_brush;

  COLORREF crCustColors[16];

  COLORREF spot_colour;
  COLORREF background_colour;

  CHOOSECOLOR spot_cc;
  CHOOSECOLOR back_cc;

  int spot_red, spot_green, spot_blue;
  int background_red, background_green, background_blue;

  IFilterPreview *ifp;
  int block_size;
  int cache_size;
  int intensity;
  Pixel32 **cache;
} FilterData;

void PreviewParameterChanged(FilterData *filter_data);
int BuildCache(FilterData *fa);
int ClearCache(FilterData *fa);
void SetColours(FilterData *fa, COLORREF spot_colour, COLORREF back_colour);
void ShowColour(HWND hdlg, FilterData *filter_data, COLORREF spot_colour, COLORREF back_colour);


int BuildCache(FilterData *filter_data) {

  //Setup cache
  filter_data->cache_size = filter_data->block_size;

  filter_data->cache = (Pixel32 **)malloc(filter_data->block_size * sizeof(Pixel32 *));

  int i;
  for(i=0;i<filter_data->block_size;i++) {
    filter_data->cache[i] = 0;
  }
  
  return 0;
}

int ClearCache(FilterData *filter_data) {

  int i;
  for(i=0;i<filter_data->cache_size;i++) {
    if(filter_data->cache[i] != 0) {
      free(filter_data->cache[i]);
    }
  }

  free(filter_data->cache);
  
  return 0;
}

int StartProc(FilterActivation *fa, const FilterFunctions *ff) {

  return BuildCache((FilterData *)fa->filter_data);
}

int StopProc(FilterActivation *fa, const FilterFunctions *ff) {

  return ClearCache((FilterData *)fa->filter_data);
}


bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
  FilterData *filter_data = (FilterData *)fa->filter_data;

  _snprintf(buf, buflen, "Config(%d, %d, %d, %d)", filter_data->spot_colour, filter_data->background_colour, filter_data->block_size, filter_data->intensity);
  return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {

  FilterActivation *fa = (FilterActivation *)lpVoid;
  FilterData *filter_data = (FilterData *)fa->filter_data;

  SetColours(filter_data, argv[0].asInt(), argv[1].asInt());
  filter_data->block_size = argv[2].asInt();
  filter_data->intensity = argv[3].asInt();
}

ScriptFunctionDef func_defs[]={
  { (ScriptFunctionPtr)ScriptConfig, "Config", "0iiii" },
  { NULL },
};

CScriptObject script_obj={
  NULL, func_defs
};

struct FilterDefinition filterdef = {

  NULL, NULL, NULL,     // next, prev, module
  "halftone",           // name
  "Approximates an image in two colours with appropriately sized monochrome circles.",           // desc
  "Tom Ford",           // author
  NULL,                 // private_data
  sizeof(FilterData),   // inst_data_size

  InitProc,             // initProc
  NULL,                 // deinitProc
  RunProc,              // runProc
  NULL,                 // paramProc
  ConfigProc,           // configProc
  StringProc,           // stringProc
  StartProc,            // startProc
  NULL,                 // endProc

  &script_obj,          // script_obj
  FssProc,              // fssProc
};

int RunProc(const FilterActivation *fa, const FilterFunctions *ff) {

  FilterData *filter_data = (FilterData *)fa->filter_data;
  FilterStateInfo *info = fa->pfsi;
  const PixDim width = fa->src.w;
  const PixDim height = fa->src.h;
  Pixel32 *src, *dst;
  int x, y;
  int r, g, b;
  int i,j;
  int dst_block_size = (int)ceil(filter_data->block_size / (double)2);

  src = fa->src.data;
  dst = fa->dst.data;

  //clear the canvas

  r = filter_data->background_red;
  g = filter_data->background_green;
  b = filter_data->background_blue;

  for (y = 0; y < height; y++) {
    for (x = 0; x < width; x++) {
      dst[x] = (r << 16) | (g << 8) | (b);
    }

    src = (Pixel *)((char *)src + fa->src.pitch);
    dst = (Pixel *)((char *)dst + fa->dst.pitch);
  }

  //decide if spots are darker or lighter than the background
  
  int sr = filter_data->spot_red;
  int sg = filter_data->spot_green;
  int sb = filter_data->spot_blue;

  Pixel32 back_lum = (r << 16) | (g << 8) | (b);                   
  Pixel32 spot_lum = (sr << 16) | (sg << 8) | (sb);                   

  bool darker = true;

  if(Lum(spot_lum) > Lum(back_lum))
    darker = false;
  
  src = fa->src.data;
  dst = fa->dst.data;

  int start_y = -1 * (dst_block_size - ((height % dst_block_size) >> 1));
  int start_x = -1 * ((width % dst_block_size) >> 1);

  for (y = start_y; y < height; y+=dst_block_size) {
    for (x = start_x; x < width; x+=dst_block_size) {

      //Calc mean lightness, saturation and median hue and set block to appropriate rgb 
        
      double mean_l;
      int clip_modifier;

      mean_l = 0;
      clip_modifier = 0;

      int start_i = 0; int end_i = dst_block_size;
      int start_j = 0; int end_j = dst_block_size;

      if(x < 0)
        start_i = -1 * x;

      if(y < 0)
        start_j = -1 * y;

      if(x + dst_block_size > width)
        end_i = width - x;

      if(y + dst_block_size > height)
        end_j = height - y;

      //account for missed pixels
      clip_modifier = (end_i - start_i) * (end_j - start_j);

      for(i = start_i; i < end_i; i++) {
        for(j = start_j; j < end_j; j++) {

          Pixel32 *pixel_to_read;
          pixel_to_read = src + x + i;
          pixel_to_read = (Pixel32 *)((char *)pixel_to_read + (fa->src.pitch * (j+start_y)));

          mean_l += Lum(*pixel_to_read);
        }
      }

      //mean lightness

      mean_l /= clip_modifier;

      //compensate for rounding errors

      if(mean_l > 1.0)
        mean_l = 1.0;

      //write output block

      int radius = (int)floor(filter_data->block_size * (1 - mean_l) * (filter_data->intensity / 20.0));
      
      AntiAliasCircle(x, y, filter_data->block_size, radius, 2, darker, fa);
      //PlotCircle(x + filter_data->block_size/2, y + filter_data->block_size/2, radius, fa);
    }

    src = (Pixel32 *)((char *)src + (dst_block_size * fa->src.pitch));
    dst = (Pixel32 *)((char *)dst + (dst_block_size * fa->dst.pitch));   
  }


  return 0;
}

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_halftone;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
  if (!(fd_halftone = ff->addFilter(fm, &filterdef, sizeof(FilterDefinition))))
    return 1;

  vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
  vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

  return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
  ff->removeFilter(fd_halftone);
}

int InitProc(FilterActivation *fa, const FilterFunctions *ff)
{
  FilterData *filter_data = (FilterData *)fa->filter_data;

  filter_data->block_size = 15;
  filter_data->intensity = 14;

  SetColours(filter_data, RGB(0,0,0), RGB(247,224,271)); //historical reasons!

  filter_data->spot_cc.rgbResult = filter_data->spot_colour;
  filter_data->back_cc.rgbResult = filter_data->background_colour;

  return 0;
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
  FilterData *filter_data = (FilterData *)GetWindowLong(hdlg, DWL_USER);

  switch(msg) {
    HWND hWnd;

  case WM_PAINT:
    return 0;

  case WM_INITDIALOG:
    SetWindowLong(hdlg, DWL_USER, lParam);
    filter_data = (FilterData *)lParam;

    hWnd = GetDlgItem(hdlg, IDC_BLOCKSIZE);
    SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(1, BLOCK_SIZE_MAX));
    SendMessage(hWnd, TBM_SETTICFREQ, 10 , 0);
    SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, filter_data->block_size);
    SetDlgItemInt(hdlg, IDC_EBLOCKSIZE, filter_data->block_size, FALSE);
    hWnd = GetDlgItem(hdlg, IDC_DARK);
    SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, INTENSITY_MAX));
    SendMessage(hWnd, TBM_SETTICFREQ, 5 , 0);
    SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, filter_data->intensity);
    SetDlgItemInt(hdlg, IDC_EDARK, filter_data->intensity, FALSE);
    
    filter_data->spot_brush = CreateSolidBrush(filter_data->spot_colour);
    filter_data->background_brush = CreateSolidBrush(filter_data->background_colour);
    ShowColour(hdlg, filter_data, filter_data->spot_colour, filter_data->background_colour);

    filter_data->ifp->InitButton(GetDlgItem(hdlg, IDPREVIEW));
    return TRUE;

  case WM_HSCROLL:
    {
      int block, dark;
      HWND hwnd = GetDlgItem(hdlg, IDC_BLOCKSIZE);
      block = SendMessage(hwnd, TBM_GETPOS, 0, 0);
      SetDlgItemInt(hdlg, IDC_EBLOCKSIZE, block, FALSE);

      hwnd = GetDlgItem(hdlg, IDC_DARK);
      dark = SendMessage(hwnd, TBM_GETPOS, 0, 0);
      SetDlgItemInt(hdlg, IDC_EDARK, dark, FALSE);

      if (block != filter_data->block_size || dark != filter_data->intensity) {
        filter_data->block_size = block;
        filter_data->intensity = dark;
        PreviewParameterChanged(filter_data);
      }      
      break;
    }

  case WM_COMMAND:
    switch(LOWORD(wParam)) {
    case IDPREVIEW:
      filter_data->ifp->Toggle(hdlg);
      break;
    case IDOK:
      EndDialog(hdlg, 0);
      return TRUE;
    case IDHELP:
      {
        MessageBox(NULL, "This filter should be pretty straightforward.", "Halftone 'help'.", MB_OK | MB_ICONINFORMATION);
        /*char prog[256];
          char path[256];
          LPTSTR ptr;

          GetModuleFileName(NULL, prog, 255);
          GetFullPathName(prog, 255, path, &ptr);
          *ptr = 0;
          strcat(path, "plugins\\Colorize.html");
          ShellExecute(hdlg, "open", path, NULL, NULL, SW_SHOWNORMAL);*/
        return TRUE;
      }
    case IDCANCEL:
      EndDialog(hdlg, 1);
      return TRUE;

    case IDC_BLOCKSIZEPLUS:
   
      if (filter_data->block_size < BLOCK_SIZE_MAX) {

        filter_data->block_size++;
        PreviewParameterChanged(filter_data);
        SetDlgItemInt(hdlg, IDC_EBLOCKSIZE, filter_data->block_size, FALSE);
        SendMessage(GetDlgItem(hdlg, IDC_BLOCKSIZE), TBM_SETPOS, (WPARAM)TRUE, filter_data->block_size);

        filter_data->ifp->RedoFrame();
      }
      break;

    case IDC_BLOCKSIZEMINUS:

      if (filter_data->block_size > 1) {

        filter_data->block_size--;
        PreviewParameterChanged(filter_data);
        SetDlgItemInt(hdlg, IDC_EBLOCKSIZE, filter_data->block_size, FALSE);
        SendMessage(GetDlgItem(hdlg, IDC_BLOCKSIZE), TBM_SETPOS, (WPARAM)TRUE, filter_data->block_size);

        filter_data->ifp->RedoFrame();
      }
      break;

    case IDC_DARKPLUS:
   
      if (filter_data->intensity < INTENSITY_MAX) {

        filter_data->intensity++;
        PreviewParameterChanged(filter_data);
        SetDlgItemInt(hdlg, IDC_EDARK, filter_data->intensity, FALSE);
        SendMessage(GetDlgItem(hdlg, IDC_DARK), TBM_SETPOS, (WPARAM)TRUE, filter_data->intensity);

        filter_data->ifp->RedoFrame();
      }
      break;

    case IDC_DARKMINUS:

      if (filter_data->intensity > 0) {

        filter_data->intensity--;
        PreviewParameterChanged(filter_data);
        SetDlgItemInt(hdlg, IDC_EDARK, filter_data->intensity, FALSE);
        SendMessage(GetDlgItem(hdlg, IDC_DARK), TBM_SETPOS, (WPARAM)TRUE, filter_data->intensity);

        filter_data->ifp->RedoFrame();
      }
      break;

    case IDC_SPOTPICK:

      filter_data->spot_cc.lStructSize = sizeof(CHOOSECOLOR);
      filter_data->spot_cc.hwndOwner = NULL;
      filter_data->spot_cc.hInstance = NULL;
      filter_data->spot_cc.lpCustColors = filter_data->crCustColors;
      filter_data->spot_cc.Flags = CC_RGBINIT | CC_FULLOPEN;
      filter_data->spot_cc.lCustData = 0L;
      filter_data->spot_cc.lpfnHook = NULL;
      filter_data->spot_cc.lpTemplateName = NULL;
      ChooseColor(&filter_data->spot_cc);
      filter_data->spot_colour = filter_data->spot_cc.rgbResult;
      SetColours(filter_data, filter_data->spot_cc.rgbResult, filter_data->background_colour);
      ShowColour(hdlg, filter_data, filter_data->spot_colour, filter_data->background_colour);

      PreviewParameterChanged(filter_data);
      filter_data->ifp->RedoFrame();
      return TRUE;
      break;

    case IDC_BACKPICK:

      filter_data->back_cc.lStructSize = sizeof(CHOOSECOLOR);
      filter_data->back_cc.hwndOwner = NULL;
      filter_data->back_cc.hInstance = NULL;
      filter_data->back_cc.lpCustColors = filter_data->crCustColors;
      filter_data->back_cc.Flags = CC_RGBINIT | CC_FULLOPEN;
      filter_data->back_cc.lCustData = 0L;
      filter_data->back_cc.lpfnHook = NULL;
      filter_data->back_cc.lpTemplateName = NULL;
      ChooseColor(&filter_data->back_cc);
      SetColours(filter_data, filter_data->spot_colour, filter_data->back_cc.rgbResult);
      ShowColour(hdlg, filter_data, filter_data->spot_colour, filter_data->background_colour);

      PreviewParameterChanged(filter_data);
      filter_data->ifp->RedoFrame();
      return TRUE;

    }
    break;

  case WM_CTLCOLORSTATIC:
    if (GetWindowLong((HWND)lParam, GWL_ID) == IDC_SPOTCOLOUR)
      return (BOOL)filter_data->spot_brush;

    if (GetWindowLong((HWND)lParam, GWL_ID) == IDC_BACKCOLOUR)
      return (BOOL)filter_data->background_brush;

    break;
    
  }

  return FALSE;
}

void PreviewParameterChanged(FilterData *filter_data) {

  filter_data->ifp->UndoSystem();
  ClearCache(filter_data);
  BuildCache(filter_data);
  filter_data->ifp->RedoSystem();
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
  FilterData *filter_data = (FilterData *)fa->filter_data;
  FilterData filter_data_old = *filter_data;
  int ret;

  filter_data->ifp = fa->ifp;

  ret = DialogBoxParam(fa->filter->module->hInstModule, MAKEINTRESOURCE(IDD_FILTER),
                       hwnd, ConfigDlgProc, (LPARAM) filter_data);
  if (ret) {
    *filter_data = filter_data_old;
    ret = TRUE;
  }
  else {
    ret = FALSE;
  }
  return(ret);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str)
{
  FilterData *filter_data = (FilterData *)fa->filter_data;

  sprintf(str, " (spot %d, back %d, block %d, int %d)", filter_data->spot_colour, filter_data->background_colour, filter_data->block_size, filter_data->intensity);
}

void RGBtoHSL(Pixel32 src, double *H, double *S, double *L)
{
  double delta;
  double r = (double)((src & 0xff0000) >> 16)/255;
  double g = (double)((src & 0xff00) >> 8)/255;
  double b = (double)(src & 0xff)/255;
  double cmax = max(r,max(g,b));
  double cmin = min(r,min(g,b));

  *L=(cmax+cmin)/2.0;
  if(cmax==cmin) {
    *S = 0;
    *H = 0; // it's really undefined
  } else {
    if(*L < 0.5) 
      *S = (cmax-cmin)/(cmax+cmin);
    else
      *S = (cmax-cmin)/(2.0-cmax-cmin);
    delta = cmax - cmin;
    if(r==cmax)
      *H = (g-b)/delta;
    else if(g==cmax)
      *H = 2.0 +(b-r)/delta;
    else
      *H=4.0+(r-g)/delta;
    *H /= 6.0;
    if(*H < 0.0)
      *H += 1;
  }
}

double Lum(Pixel32 src) {

  double r = (double)((src & 0xff0000) >> 16)/255;
  double g = (double)((src & 0xff00) >> 8)/255;
  double b = (double)(src & 0xff)/255;
  double cmax = max(r,max(g,b));
  double cmin = min(r,min(g,b));

  return (cmax+cmin)/2.0;
}


int HuetoRGB(double m1, double m2, double h)
{
  if( h < 0 ) h += 1.0;
  if( h > 1 ) h -= 1.0;
  if( 6.0*h < 1 )
    return ((int) (255 * (m1+(m2-m1)*h*6.0)));
  if( 2.0*h < 1 )
    return ((int) (255 * m2));
  if( 3.0*h < 2.0 )
    return ((int) (255 * (m1+(m2-m1)*((2.0/3.0)-h)*6.0)));
  return ((int) (255 * m1));
}

void HLStoRGB(double H, double L, double S, int *r, int *g, int *b)
{
  double m1, m2;

  if(S==0) {
    *r = *g = *b = (int) (255 * L);
  } else {
    if(L <=0.5)
      m2 = L*(1.0+S);
    else
      m2 = L+S-L*S;
    m1 = 2.0*L-m2;
    *r = HuetoRGB(m1,m2,H+1.0/3.0);
    *g = HuetoRGB(m1,m2,H);
    *b = HuetoRGB(m1,m2,H-1.0/3.0);
  }
}

void AntiAliasCircle(int x_l, int y_b, unsigned int block_size, unsigned int radius, unsigned int mag, bool darker, const FilterActivation *fa) {

  unsigned int i;
  FilterData *filter_data = (FilterData *)fa->filter_data;

  //Ensure at least a spot is drawn

  if(!radius)
    radius = 1;

  //Calculate coords and size of block to contain entire circle
  int offset = block_size - (radius * 2);

  int x_l2 = x_l + (offset / (2 * (int)mag));
  int y_b2 = y_b + (offset / (2 * (int)mag));

  block_size = (radius * 2) + 2;

  Pixel32 *dst;

  unsigned int dst_block_size = (int)ceil(block_size / (double)mag);

  if(filter_data->cache[radius - 1] == 0) {
  
    unsigned int canvas_length = block_size * block_size;

    //Create larger canvas

    Pixel32 *src = (Pixel32 *)malloc(canvas_length * sizeof(Pixel32));
    unsigned int dst_length = dst_block_size * dst_block_size;
    dst = (Pixel32 *)malloc((dst_length) * sizeof(Pixel32));

    int r = filter_data->background_red;
    int g = filter_data->background_green;
    int b = filter_data->background_blue;
    
    Pixel32 colour = (r << 16) | (g << 8) | (b);                   

    //Clear canvases to base colour

    for (i = 0; i < canvas_length; i++) {
      src[i] = colour;
    }

    for (i = 0; i < dst_length; i++) {
      dst[i] = colour;
    }
  
    //Plot large (magnified) circle on large canvas
    PlotCircle(fa, src, block_size, radius);

    //Create antialiased copy of circle at normal size 
    AntiAliasCopyBlock(fa, dst, src, block_size, dst_block_size, mag, radius);

    //Cache the antialiased block
    filter_data->cache[radius - 1] = dst;

    free(src);
  }
  
  dst = filter_data->cache[radius - 1];

  //Copy antialiased block onto actual canvas, don't copy lighter pixels
  CopyBlockLightness(fa, dst, x_l2, y_b2, dst_block_size, darker);
}

void Pixel32RGB(Pixel32 src, int *r, int *g, int *b) {

  *r = (src & 0xff0000) >> 16;
  *g = (src & 0xff00) >> 8;
  *b = src & 0xff;
}

void PlotCircle(const FilterActivation *fa, Pixel32 *src, unsigned int block_size, unsigned int radius) {

  int x, y;
  int x_change, y_change;
  int radius_error;

  unsigned int x_orig = (block_size >> 1) - 1;
  unsigned int y_orig = (block_size >> 1) - 1;

  x = radius;
  y = 0;

  x_change = 1 - 2 * radius;
  y_change = 1;
  radius_error = 0;

  while (x >= y) {

    Plot8CirclePoints(fa, src, block_size, x_orig, y_orig, x, y);

    y++;
    radius_error += y_change;
    y_change += 2;

    if(radius_error * 2 + x_change > 0) {
      x--;
      radius_error += x_change;
      x_change += 2;
    }
  }
}

void Plot8CirclePoints(const FilterActivation *fa, Pixel32 *src, unsigned int block_size, int x_orig, int y_orig, int x, int y) {

  int i;

  i = x_orig - x;
  while(i <= x_orig + x) {
    PlotPoint(fa, src, block_size, i, y_orig + y);
    i++;
  }

  i = x_orig - x;
  while(i <= x_orig + x) {
    PlotPoint(fa, src, block_size, i, y_orig - y);
    i++;
  }

  i = x_orig - y;
  while(i <= x_orig + y) {
    PlotPoint(fa, src, block_size, i, y_orig + x);
    i++;
  }

  i = x_orig - y;
  while(i <= x_orig + y) {
    PlotPoint(fa, src, block_size, i, y_orig - x);
    i++;
  }
}

void PlotPoint(const FilterActivation *fa, Pixel32 *src, unsigned int block_size, unsigned int x, unsigned int y) {
  
  FilterData *filter_data = (FilterData *)fa->filter_data;
  Pixel32 *pixel_to_write;

  pixel_to_write = src + x;
  pixel_to_write = pixel_to_write + (block_size * y);

  int r = filter_data->spot_red;
  int g = filter_data->spot_green;
  int b = filter_data->spot_blue;

  *(pixel_to_write) = (r << 16) | (g << 8) | (b);                   
}

void CopyBlockLightness(const FilterActivation *fa, Pixel32 *src, int x_l, int y_b, unsigned int block_size, bool darker) {

  int x, y;
  
  const PixDim dst_width = fa->src.w;
  const PixDim dst_height = fa->src.h;

  int d_width = dst_width;
  int d_height = dst_height;

  int b_size = block_size;

  Pixel32 src_pixel, *dst_pixel;

  int dst_x, dst_y;
  int max_x = block_size;
  int max_y = block_size;
  int min_x = 0;
  int min_y = 0;

  if(x_l + b_size > d_width) {
    max_x = d_width - x_l;
  }

  if(y_b + b_size > d_height) {
    max_y = d_height - y_b;
  }

  if(x_l < 0) {
    min_x = -1 * x_l;
  }
  
  if(y_b < 0) {
    min_y = -1 * y_b;
  }

  for(x = min_x; x < max_x; x++) {
    for(y = min_y; y < max_y; y++) {

      //Position of destination pixel
      dst_x = x_l + x;
      dst_y = y_b + y;

      src_pixel = *(src + x + (y * block_size));
      dst_pixel = fa->dst.data + dst_x;
      dst_pixel = (Pixel32 *)((char *)dst_pixel + (fa->src.pitch * dst_y));
      
      //Check if what's already there is darker, if so do not draw
      double dst_l, src_l;
      dst_l = Lum(*dst_pixel);
      src_l = Lum(src_pixel);

      if(darker && (dst_l > src_l)) {
        (*dst_pixel) = src_pixel;
      }
      else if(!darker && (dst_l < src_l)) {
        (*dst_pixel) = src_pixel;
      }
    }
  }
}

void AntiAliasCopyBlock(const FilterActivation *fa, Pixel32 *dst, Pixel32 *src, unsigned int src_block_size, unsigned int dst_block_size, unsigned int mag, int radius) {

  FilterData *filter_data = (FilterData *)fa->filter_data;
    
  int x, y;
  int i, j;

  Pixel32 *pixel_to_write;

  int rb = filter_data->background_red;
  int gb = filter_data->background_green;
  int bb = filter_data->background_blue;

  int max_block = src_block_size;

  if(src_block_size / mag >= dst_block_size) {
    max_block = dst_block_size * mag;
  }

  for(x = 0; x < max_block; x += mag) {
    for(y = 0; y < max_block; y += mag) {

      //Position of destination pixel
      unsigned int dst_x = (x / mag);
      unsigned int dst_y = (y / mag);
      
      int r, g, b;
      unsigned int r_mean = 0;
      unsigned int g_mean = 0;
      unsigned int b_mean = 0;

      int max_mag_block = mag;
      
      if(x + mag >= src_block_size)
        max_mag_block = src_block_size - x;

      unsigned int src_anti_block_area = max_mag_block * max_mag_block;

      //Take mean of mag * mag block for destination pixel RGB
      for(i = 0; i < max_mag_block; i++) {
        for(j = 0; j < max_mag_block; j++) {

          unsigned int src_x = x + i;
          unsigned int src_y = y + j;
          
          Pixel32RGB(*(src + src_x + (src_y * src_block_size)), &r, &g, &b);

          r_mean += r;
          g_mean += g;
          b_mean += b;
        }
      }
      
      r_mean /= src_anti_block_area;
      g_mean /= src_anti_block_area;
      b_mean /= src_anti_block_area;

      Pixel32 src_pixel = (r_mean << 16) | (g_mean << 8) | (b_mean);

      //Write output pixel

      pixel_to_write = dst + dst_x;
      pixel_to_write = pixel_to_write + (dst_block_size * dst_y);
      *(pixel_to_write) = src_pixel;

    }
  }
}

void SetColours(FilterData *filter_data, COLORREF spot_colour, COLORREF back_colour) {

  filter_data->spot_colour = spot_colour;

  filter_data->spot_red = GetRValue(spot_colour);
  filter_data->spot_green = GetGValue(spot_colour);
  filter_data->spot_blue = GetBValue(spot_colour);

  filter_data->background_colour = back_colour;

  filter_data->background_red = GetRValue(back_colour);
  filter_data->background_green = GetGValue(back_colour);
  filter_data->background_blue = GetBValue(back_colour);
}

void ShowColour(HWND hdlg, FilterData *filter_data, COLORREF spot_colour, COLORREF back_colour) {

  DeleteObject(filter_data->spot_brush);
  DeleteObject(filter_data->background_brush);
  filter_data->spot_brush = CreateSolidBrush(spot_colour);
  filter_data->background_brush = CreateSolidBrush(back_colour);
  RedrawWindow(GetDlgItem(hdlg, IDC_SPOTCOLOUR), NULL, NULL, RDW_ERASE|RDW_INVALIDATE|RDW_UPDATENOW);
  RedrawWindow(GetDlgItem(hdlg, IDC_BACKCOLOUR), NULL, NULL, RDW_ERASE|RDW_INVALIDATE|RDW_UPDATENOW);
}

