/**************************************************************************
*  Image process tool box                                                 *
*     by Yang Yudong                                                      *
*     yangyd@yahoo.com                                                    *
*                                                                         *
***************************************************************************
*  Copyright (C) 1992-1999, Yang Yudong, All rights reserved.             *
*  This file is part of Yang Yudong's image processing software package.  *
*  If you use this software, you agree to the following:                  *
*  This program package is purely experimental, and is licensed "as is".  *
*  Permission is granted to use, modify, and distribute this program      *
*  without charge for any purpose, provided this license/ disclaimer      *
*  notice appears in the copies.  No warranty or maintenance is given,    *
*  either expressed or implied.  In no event shall the author(s) be       *
*  liable to you or a third party for any special, incidental,            *
*  consequential, or other damages, arising out of the use or inability   *
*  to use the program for any purpose (or the loss of data), even if we   *
*  have been advised of such possibilities.  Any public reference or      *
*  advertisement of this source code should refer to it as Yang Yudong's  *
*  orignal.                                                               *
**************************************************************************/
/******************************************************************
*  Image process tool box
*     by Yang Yudong
*
* File : filepcx.c
* Description:  PCX 8 bits indexed color image file I/O
* Create Date:  1996. 10. 1
* Modification(date/where): 1996.11.24 Make the integer byte order system independent
* Modification(date/where): 1997.5.30  Add SavePCX8b
*******************************************************************/

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#define _IMG_LIBBUILD_
#include "image.h"
#include "imgfile.h"

/*  These remarks are written in Chinese*/

/*   PCX File

   ļͷݣΪ PCXHeader Ϊ 128 ֽ)
   ͼ
   ɫݣҪʹõɫ壬ûС Ϊ769ֽڣ
 */




/* PCX  header*/
typedef struct   {
             IBYTE       PCXFlag;         /* PCXļı־ ȡֵΪ 10                */
             IBYTE       PCXVersion;      /* PCX汾Ϣֻа汾5ſеɫϢ */
             IBYTE       RunLengthEncode; /* г̳ȱ־ȡֵΪ 1               */
             IBYTE       BitsPerPixel;    /* ÿصĶλ ȡֵΪ 1248   */
             IWORD       XStart, YStart;  /* ʾʼ                             */
             IWORD       XEnd,   YEnd;    /* ʾꢡ           */
             IWORD       HorResolution;   /* ˮƽֱ                               */
             IWORD       VerResolution;   /* ֱֱ                               */
             IMyRGB      ColorMap[16];    /* ûеɫPCXļ                  */
             IBYTE       Reserved;        /* ֽ                                 */
             IBYTE       NumPlanes;       /* λƽ                                 */
             IWORD       BytesPerLine;    /* ÿֽӦ4ı              */
             IWORD       PaletteInterp;   /* ɫ˵ϢȡֵΪ1ɫ2Ҷȼ */
             IBYTE       AlsoReserved[58];/* ֽ                                 */
             }  PCXHeader;


typedef struct   {
             IBYTE PaletteFlag;       /* ɫЧ־ȡֵΪ 12              */
             IMyRGB  Palette[256];      /* 256ĵɫ                      */
             }  PaletteTable;


/* PCXļһЩ:
A.   PCX
     1. һλƽͼ                                   MSB    LSB
          ÿ1λʱ, 8Ϊһֽ, Ӹ.      12345678
          ÿ2λʱ, 4Ϊһֽ, ͬǰ.         11223344
          ÿ4λʱ, 2Ϊһֽ, ͬǰ.         11112222
          ÿ8λʱ, 1Ϊһֽ.                 11111111
           2. λƽͼ
          صֽͬһλƽ, ǲͬλƽͬһҪһ.
          ʽΪ: λƽ1ĵm λƽ2ĵm   λƽnĵm,
          ڼÿеֽʱ, һλƽһ.


B.    PCXļֽг̳ȱĹ

   1.   г̵󳤶63ֽ.
   2.   г̱벻һȥ.
   3.   г̳ĸʽǳȱǰ, ֽں.
   4.   г̳ȱȡֵΧ192255, 063.
   5.   г̵ĳ1
              ֽڵֵС192ʱҪȱֵ
              ֽڵֵڻ192Ҫȱֵ
   6.   ÿеֽӦĵı, 㲹.

   :

     ֵֽ:1  1A 2C 2A 2A C0 C1 C9 E8 3C F0 F0 F0 F0
              2  F0 F0 C2 A8 13 13 24 68 37 A0 48 48 48

      :      1A 2C C2 2A C1 C0 C1 C1 C1 C9 C1 E8 3C C4 F0 C3
                         00 C2 F0 C1 C2 A8 C2 13 24 68 37 A0 C2 48 C3 00


 */




IBOOL LoadPCX8b(const char * fname, ImageDes *img, IBOOL ping)
{
  FILE * fd;
  int xsize, ysize, bpl;
  int i, j;
  int pix, run;

  PCXHeader ph;
  char pl[769];

  IBYTE *buf;

    fd=fopen(fname, "rb");
    if(fd == NULL) {
              return IFALSE;
    }
    ph.PCXFlag = (IBYTE)fgetc(fd);
    ph.PCXVersion = (IBYTE)fgetc(fd);
    ph.RunLengthEncode = (IBYTE)fgetc(fd);
    ph.BitsPerPixel = (IBYTE)fgetc(fd);
    ph.XStart = fgetWord(fd, ITRUE);
    ph.YStart = fgetWord(fd, ITRUE);
    ph.XEnd = fgetWord(fd, ITRUE);
    ph.YEnd = fgetWord(fd, ITRUE);
    ph.HorResolution = fgetWord(fd, ITRUE);
    ph.VerResolution = fgetWord(fd, ITRUE);
    fread(ph.ColorMap, 3, 16, fd);
    ph.Reserved = (IBYTE)fgetc(fd);
    ph.NumPlanes = (IBYTE)fgetc(fd);
    ph.BytesPerLine = fgetWord(fd, ITRUE);
    ph.PaletteInterp = fgetWord(fd, ITRUE);
    fread(ph.AlsoReserved, 1, 58, fd);
    if(ph.PCXFlag != 10  || ph.BitsPerPixel != 8 || ph.NumPlanes != 1 ) 
    { /* invalid 8 bits PCX file*/
              fclose(fd);
              return IFALSE;
    }

    if(fseek(fd,-769L, SEEK_END)) 
    {   /* Get Palette*/
              fclose(fd);
              return IFALSE;
    }
    if(fread(pl, 1, 769, fd) != 769)
    {
              fclose(fd);
              return IFALSE;
    }
    if(pl[768] != 12) 
    { /* invalid palette*/
              fclose(fd);
              return IFALSE;
    }
    
    xsize=ph.XEnd-ph.XStart+1;
    ysize=ph.YEnd-ph.YStart+1;
    bpl=ph.BytesPerLine;    
    
	if(ping)
	{
		fclose(fd);
		return InitPicture(img, xsize, ysize, IColor256, IFALSE, 256);
	}
    
    /*Allocate memory*/
    if(!AllocPicture(img, xsize, ysize, IColor256, IFALSE, 256))
    {
       fclose(fd); return IFALSE;  /* can not allocate enough memory*/
    }

    __Img_Busy(999);
    buf = img->r;

    /* fill out the palette*/
	for(i=0; i<256; i++)
	{
		img->pal[i].r = pl[i*3+0];
		img->pal[i].g = pl[i*3+1];
		img->pal[i].b = pl[i*3+2];
	}

    fseek(fd, sizeof(PCXHeader), SEEK_SET);        /* Prepare for load */
    for(i=0; i<ysize; i++) 
    {
       __Img_Busy(i*100/img->ysize);
       run = 0;
       j=0;
       while(j < bpl)   
       {
            if(run==0)   
            {
                if((pix=fgetc(fd)) == EOF)
                   {i=ysize;break;}
                if(pix > 192)   
                {
                   run=pix-192;
                   if((pix=fgetc(fd)) == EOF)
                       {i=ysize;break;}
                }
                else
                  run = 1;
            }
            for(;j<xsize && run > 0; j++, run--) 
            {
                 *buf = (IBYTE)pix;
                 buf ++;
            }
            for(;j<bpl && run > 0; j++, run--);
       }
    }

    fclose(fd);
   
    img->load = ITRUE;
    __Img_Busy(0);
    return ITRUE;
}


IBOOL SavePCX8b(const char * fname, ImageDes img)
{
  FILE * fd;
  int bpl;
  int i, j, k;
  int pix, run, tail;
  PCXHeader ph;
  IBYTE *buf, *obuf;
  IBYTE *p;
  ImageDes timg;

    if(!img.load ) return IFALSE;
    timg.alloc = IFALSE;
    __Img_Busy(999);
    if(img.imagetype == ITrueColor) 
    {
       if(!img_color24to8(&timg, img, 256, 0)) 
       {
          __Img_Busy(0);
          return(IFALSE);
       }
       buf = timg.r;
    }
    else buf = img.r;  

    ph.PCXFlag = 10;
    ph.PCXVersion = 5;
    ph.RunLengthEncode = 1;
    ph.BitsPerPixel = 8;
    ph.XStart = 0;
    ph.YStart = 0;
    ph.XEnd = img.xsize-1;
    ph.YEnd = img.ysize-1;
    ph.HorResolution = 100;
    ph.VerResolution = 100;
    ph.Reserved = 0;
    ph.NumPlanes = 1;
    ph.BytesPerLine = bpl = (img.xsize+3)&0xfffc;
    ph.PaletteInterp = (img.imagetype == IGray ? 2 : 1);

    obuf = malloc(bpl*2 > 768 ? bpl*2 : 768);  // for pallete of grey scale image and run length coding buffer
    if(obuf == NULL)
    {
        FreePicture(&timg); 
        __Img_Busy(0);
        return IFALSE;
    }

    fd=fopen(fname, "wb");
    if(fd == NULL)
    {
        free(obuf);
        FreePicture(&timg); 
        __Img_Busy(0);
        return IFALSE;
    }

    fputc(ph.PCXFlag, fd);
    fputc(ph.PCXVersion, fd);
    fputc(ph.RunLengthEncode, fd);
    fputc(ph.BitsPerPixel, fd);
    fputWord(ph.XStart, fd, ITRUE);
    fputWord(ph.YStart, fd, ITRUE);
    fputWord(ph.XEnd, fd, ITRUE);
    fputWord(ph.YEnd, fd, ITRUE);
    fputWord(ph.HorResolution, fd, ITRUE);
    fputWord(ph.VerResolution, fd, ITRUE);
    fwrite(ph.ColorMap, 3, 16, fd);
    fputc(ph.Reserved, fd);
    fputc(ph.NumPlanes, fd);
    fputWord(ph.BytesPerLine, fd, ITRUE);
    fputWord(ph.PaletteInterp, fd, ITRUE);
    fwrite(ph.AlsoReserved, 1, 58, fd);

    tail = bpl-img.xsize;
    for(i=0; i<img.ysize; i++)
    {
       __Img_Busy(i*100/img.ysize);
       p = buf+img.xsize*i;
       for(j=0,k=0; j<bpl;) 
       {
          if(j<img.xsize) // in case it'll run over the boundary of memory block
          {
             pix = p[j++];  
             for(run=1; run<63 && j<img.xsize && pix==p[j]; run++,j++);  // run coding
             if(j == img.xsize && run+tail <= 63) 
             {
               run += tail; j+= tail; 
             }
          }  
          else
          {
             pix = p[img.xsize-1];
             run = tail; j = bpl;
          }   
          if(run > 1 || pix >= 192) 
            { obuf[k]=run+192; obuf[k+1]=pix; k+=2;} 
          else if(pix <192) obuf[k++] = pix;  
       }
       fwrite(obuf, 1, k, fd);
    }
    
    fputc(12, fd);  // palette flag
    if(img.imagetype == IGray)
    {
      for(i=0; i<256; i++) obuf[i*3] = obuf[i*3+1] = obuf[i*3+2] = i;
    }
    else if(img.imagetype == ITrueColor)
	{
      for(i=0; i<256; i++) 
	  {
		  obuf[i*3+0] = timg.pal[i].r;
		  obuf[i*3+1] = timg.pal[i].g;
		  obuf[i*3+2] = timg.pal[i].b;
	  }
	}
    else 
	{
      for(i=0; i<256; i++) 
	  {
		  obuf[i*3+0] = img.pal[i].r;
		  obuf[i*3+1] = img.pal[i].g;
		  obuf[i*3+2] = img.pal[i].b;
	  }
	}
    fwrite(obuf, 256, 3, fd);
    fclose(fd);
    free(obuf);
    FreePicture(&timg); 
    __Img_Busy(0);
    return ITRUE;
}

