/**************************************************************************
*  Image process tool box                                                 *
*     by Yang Yudong                                                      *
*     yangyd@yahoo.com                                                    *
*                                                                         *
***************************************************************************
*  Copyright (C) 1992-1999, Yang Yudong, All rights reserved.             *
*  This file is part of Yang Yudong's image processing software package.  *
*  If you use this software, you agree to the following:                  *
*  This program package is purely experimental, and is licensed "as is".  *
*  Permission is granted to use, modify, and distribute this program      *
*  without charge for any purpose, provided this license/ disclaimer      *
*  notice appears in the copies.  No warranty or maintenance is given,    *
*  either expressed or implied.  In no event shall the author(s) be       *
*  liable to you or a third party for any special, incidental,            *
*  consequential, or other damages, arising out of the use or inability   *
*  to use the program for any purpose (or the loss of data), even if we   *
*  have been advised of such possibilities.  Any public reference or      *
*  advertisement of this source code should refer to it as Yang Yudong's  *
*  orignal.                                                               *
**************************************************************************/
/******************************************************************/
/*  Image process tool box*/
/*     by Yang Yudong*/
/**/
/* File : fileraw.c*/
/* Description: Raw format image file I/O */
/* Create Date: 1996. 9. 25*/
/* Modification(date/where):*/
// Modification(date/where): 1996.11.23 Make the integer byte order system independent
/**/
/******************************************************************/

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#define _IMG_LIBBUILD_
#include "image.h"
#include "imgfile.h"


/* save image as a raw 8 bits grey scale image*/
/*  conversion will be added if needed.*/
/* return ITRUE if successful*/

IBOOL SaveRawGrey(const char * fname, ImageDes img)
 {
  IBYTE * cr;
  IBYTE * cg;
  IBYTE * cb;
  IBYTE *buf;
  int  i, j;
  int  offset;
  FILE   *fd;
  IBYTE  px;
  
  if(!img.load) return IFALSE;

  fd=fopen(fname, "wb");
  if(fd == NULL) {
     return IFALSE; 
   }

  cr = img.r;
  if(img.imagetype != IGrey) {
    if((buf = (IBYTE *)malloc(img.xsize)) == NULL) {
      fclose(fd);
      return IFALSE;
     } 
    if(img.imagetype == ITrueColor) {
      cg = img.g;
      cb = img.b;
      for(i=0, offset=0; i < img.ysize;  i++ ) {
        __Img_Busy(i*100/img.ysize);
        for(j=0; j<img.xsize; j++, offset++) 
          /* Y = 0.30*R + 0.59*G + 0.11*B*/
          buf[j] = (IBYTE)((IWORD)((IWORD)cr[offset]*77
                                +(IWORD)cg[offset]*151
                                +(IWORD)cb[offset]*28 )>>8);
        fwrite(buf, img.xsize, 1, fd);
      }
    }
    else {
      for(i=0, offset=0; i < img.ysize;  i++ ) {
        __Img_Busy(i*100/img.ysize);
        for(j=0; j<img.xsize; j++, offset++) {
          px = cr[offset];  
          /* Y = 0.30*R + 0.59*G + 0.11*B*/
          buf[j] = (IBYTE)((IWORD)((IWORD)img.pal[px].r*77
                                +(IWORD)img.pal[px].g*151
                                +(IWORD)img.pal[px].b*28 )>>8);
        }
        fwrite(buf, img.xsize, 1, fd);
      }
    }
    free(buf);
  }    
  else 
     fwrite(cr, img.xsize*img.ysize, 1, fd);
   
  fputWord((IWORD)img.xsize, fd, ITRUE);
  fputWord((IWORD)img.ysize, fd, ITRUE);
  fclose(fd);
  return ITRUE;
 }


/* load image as a raw 8 bits grey scale image*/
/*   if xsize or ysize is not a positive number */
/*   the image size will be defined by the two 16 bit integers at end of file*/
/* return ITRUE if successful*/

IBOOL LoadRawGrey(const char * fname, ImageDes *img, int xsize, int ysize)
 {
  FILE * fd;
/*  int xsize=0, ysize=0;*/
  int fsz;
    
  fd=fopen(fname, "rb");
  if(fd == NULL)
     return IFALSE;
  
  if(xsize <= 0 || ysize <= 0) {  /* raw data contains a size defination*/
    if(fseek(fd, -4, SEEK_END)) 
       {fclose(fd); return IFALSE;}    /* seek error*/

    fsz = ftell(fd);
    xsize = fgetWord(fd, ITRUE);
    ysize = fgetWord(fd, ITRUE);
    if(xsize <= 0 || ysize <= 0 || xsize*ysize > fsz) 
       {fclose(fd); return IFALSE;}    /* read error*/
    if(fseek(fd, 0, SEEK_SET))
      {fclose(fd); return IFALSE;}    /* seek error*/
  }
  
  if(!AllocPicture(img, xsize, ysize, IGrey, IFALSE, 256)){
       fclose(fd); return IFALSE;  /* can not allocate enough memory*/
     }

  __Img_Busy(999);
  fsz = fread(img->r, 1, xsize*ysize, fd);
  fclose(fd);
  if(fsz != xsize*ysize) {
    FreePicture(img);
    __Img_Busy(0);
    return IFALSE;
  }
  img->load = ITRUE;
  __Img_Busy(0);
  return ITRUE;
}


/* load image as a raw 24 bits RGB color image as RRRRR... GGGGG... BBBBB...*/
/*   if xsize or ysize is not a positive number */
/*   the image size will be defined by the two 16 bit integers at end of file*/
/* return ITRUE if successful*/

IBOOL LoadRawColor(const char * fname, ImageDes *img, int xsize, int ysize)
 {
  FILE * fd;
/*  int xsize=0, ysize=0;*/
  int fsz;
    
  fd=fopen(fname, "rb");
  if(fd == NULL)
     return IFALSE;
  
  if(xsize <= 0 || ysize <= 0) {  /* raw data contains a size defination*/
    if(fseek(fd, -4, SEEK_END)) 
       {fclose(fd); return IFALSE;}    /* seek error*/

    fsz = ftell(fd);
    xsize = fgetWord(fd, ITRUE);
    ysize = fgetWord(fd, ITRUE);
    if(xsize <= 0 || ysize <= 0 || xsize*ysize*3 > fsz) 
       {fclose(fd); return IFALSE;}    /* read error*/
    if(fseek(fd, 0, SEEK_SET))
      {fclose(fd); return IFALSE;}    /* seek error*/
  }
  
  if(!AllocPicture(img, xsize, ysize, ITrueColor, IFALSE, 0)){
       fclose(fd); return IFALSE;  /* can not allocate enough memory*/
     }

  __Img_Busy(999);
  fsz  = fread(img->r, 1, xsize*ysize, fd);
  fsz += fread(img->g, 1, xsize*ysize, fd);
  fsz += fread(img->b, 1, xsize*ysize, fd);
  fclose(fd);
  if(fsz != xsize*ysize*3) {
    FreePicture(img);
    __Img_Busy(0);
    return IFALSE;
  }
  img->load = ITRUE;
  __Img_Busy(0);
  return ITRUE;
}


/* save image as a raw 24 bits Color image as RRRRR... GGGGG... BBBBB...*/
/*  conversion will be added if needed.*/
/* return ITRUE if successful*/

IBOOL SaveRawColor(const char * fname, ImageDes img)
 {
  IBYTE *buf;
  int  i, j;
  int  offset;
  FILE   *fd;
  
  if(!img.load) return IFALSE;

  fd=fopen(fname, "wb");
  if(fd == NULL) {
     return IFALSE; 
   }

  if(img.imagetype == IColor256 || img.imagetype == IIndexedColor) {
    if((buf = (IBYTE *)malloc(img.xsize)) == NULL) {
      fclose(fd);
      return IFALSE;
    } 
    for(i=0, offset=0; i < img.ysize;  i++ ) {
        __Img_Busy(i*100/img.ysize/3);
        for(j=0; j<img.xsize; j++, offset++) 
          buf[j] = img.pal[img.r[offset]].r;
        fwrite(buf, img.xsize, 1, fd);
    }
    for(i=0, offset=0; i < img.ysize;  i++ ) {
        __Img_Busy(i*100/img.ysize/3);
        for(j=0; j<img.xsize; j++, offset++) 
          buf[j] = img.pal[img.r[offset]].g;
        fwrite(buf, img.xsize, 1, fd);
    }
    for(i=0, offset=0; i < img.ysize;  i++ ) {
        __Img_Busy(i*100/img.ysize/3);
        for(j=0; j<img.xsize; j++, offset++) 
          buf[j] = img.pal[img.r[offset]].b;
        fwrite(buf, img.xsize, 1, fd);
    }
    free(buf);
    __Img_Busy(0);
  }    
  else {
     fwrite(img.r, img.xsize*img.ysize, 1, fd);
     if(img.imagetype == IGray) {
       fwrite(img.r, img.xsize*img.ysize, 1, fd);
       fwrite(img.r, img.xsize*img.ysize, 1, fd);
     }
     else {
       fwrite(img.g, img.xsize*img.ysize, 1, fd);
       fwrite(img.b, img.xsize*img.ysize, 1, fd);
     }
  }   
  fputWord((IWORD)img.xsize, fd, ITRUE);
  fputWord((IWORD)img.ysize, fd, ITRUE);
  fclose(fd);
  return ITRUE;
 }


IBOOL SaveRawASCII(const char * fname, ImageDes img)
 {
  int offset;
  IBYTE *buf;
  int  i, j;
  FILE *fd;

  if(!img.load) return IFALSE;

  fd=fopen(fname, "wb");
  if(fd == NULL) {
     return IFALSE; 
   }

  if((buf=malloc(img.xsize*4+3)) == NULL) {
      fclose(fd);
      return IFALSE;
  } 

  for(i=0, offset=0; i < img.ysize;  i++ ) {
       for(j=0; j<img.xsize; j++, offset++) 
          sprintf((char *)buf+j*4, "%3u ", img.r[offset]);
       buf[img.xsize*4] = 0x0d; buf[img.xsize*4+1] = 0x0a;
       fwrite(buf, img.xsize*4+2, 1, fd);
  }
  if(img.imagetype == ITrueColor) {
      for(i=0, offset=0; i < img.ysize;  i++ ) {
           for(j=0; j<img.xsize; j++, offset++) 
              sprintf((char *)buf+j*4, "%3u ", img.g[offset]);
           buf[img.xsize*4] = 0x0d; buf[img.xsize*4+1] = 0x0a;
           fwrite(buf, img.xsize*4+2, 1, fd);
      }
      for(i=0, offset=0; i < img.ysize;  i++ ) {
           for(j=0; j<img.xsize; j++, offset++) 
              sprintf((char *)buf+j*4, "%3u ", img.b[offset]);
           buf[img.xsize*4] = 0x0d; buf[img.xsize*4+1] = 0x0a;
           fwrite(buf, img.xsize*4+2, 1, fd);
      }
  }
  free(buf);
  fclose(fd);
  return ITRUE;
 }


/* load image as a raw 8 bits grey scale image or 3 plane raw RGB color */
/*   if xsize or ysize is not a positive number */
/*   the image size will be defined by the two 16 bit integers at end of file*/
/* return ITRUE if successful*/
IBOOL LoadRaw(const char * fname, ImageDes *img, int xsize, int ysize)
 {
  FILE * fd;
/*  int xsize=0, ysize=0;*/
  int fsz;
  IBOOL colors=IFALSE;
    
  fd=fopen(fname, "rb");
  if(fd == NULL)
     return IFALSE;
  
  if(xsize <= 0 || ysize <= 0)
  {  /* raw data contains a size defination*/
    if(fseek(fd, -4, SEEK_END)) 
       {fclose(fd); return IFALSE;}    /* seek error*/
    fsz = ftell(fd);
    xsize = fgetWord(fd, ITRUE);
    ysize = fgetWord(fd, ITRUE);
    if(fseek(fd, 0, SEEK_SET))
      {fclose(fd); return IFALSE;}    /* seek error*/
    if(xsize <= 0 || ysize <= 0)
       {fclose(fd); return IFALSE;}    /* read error*/
  }
  else
  {
    if(fseek(fd, 0, SEEK_END)) 
       {fclose(fd); return IFALSE;}    /* seek error*/
    fsz = ftell(fd);
    if(fseek(fd, 0, SEEK_SET))
      {fclose(fd); return IFALSE;}    /* seek error*/
  }  
  
  if(xsize*ysize > fsz) 
      {fclose(fd); return IFALSE;}    /* seek error*/
  else if(xsize*ysize*3 <= fsz) colors = ITRUE;
    
  if(colors)
  {
     if(!AllocPicture(img, xsize, ysize, ITrueColor, IFALSE, 0)){
       fclose(fd); return IFALSE;  /* can not allocate enough memory*/
     }  
     __Img_Busy(999);
     fsz  = fread(img->r, 1, xsize*ysize, fd);
     fclose(fd);
     if(fsz != xsize*ysize)
     {
        FreePicture(img);
        __Img_Busy(0);
        return IFALSE;
     }
  }
  else
  {
     if(!AllocPicture(img, xsize, ysize, ITrueColor, IFALSE, 0)){
       fclose(fd); return IFALSE;  /* can not allocate enough memory*/
     }  
     fsz  = fread(img->r, 1, xsize*ysize, fd);
     fsz += fread(img->g, 1, xsize*ysize, fd);
     fsz += fread(img->b, 1, xsize*ysize, fd);
     fclose(fd);
     if(fsz != xsize*ysize*3) {
       FreePicture(img);
       __Img_Busy(0);
       return IFALSE;
     }
  }

  img->load = ITRUE;
  __Img_Busy(0);
  return ITRUE;
}


/* load image as a raw 8 bits grey scale image*/
/*   if xsize or ysize is not a positive number */
/*   the image size will be defined by the two 16 bit integers at end of file*/
/* return ITRUE if successful*/

IBOOL LoadRawGreyWithHeader(const char * fname, ImageDes *img, int xsize, int ysize, int offset)
 {
  FILE * fd;
/*  int xsize=0, ysize=0;*/
  int fsz;
    
  if(offset < 0) return IFALSE;
  fd=fopen(fname, "rb");
  if(fd == NULL)
     return IFALSE;
  
  if(xsize <= 0 || ysize <= 0) return IFALSE;
  if(fseek(fd, 0, SEEK_END)) 
       {fclose(fd); return IFALSE;}    /* seek error*/
  fsz = ftell(fd);
  if(xsize*ysize > fsz-offset) 
       {fclose(fd); return IFALSE;}    /* read error*/

  if(fseek(fd, offset, SEEK_SET))
      {fclose(fd); return IFALSE;}    /* seek error*/
  
  if(!AllocPicture(img, xsize, ysize, IGrey, IFALSE, 256)){
       fclose(fd); return IFALSE;  /* can not allocate enough memory*/
     }

  __Img_Busy(999);
  fsz = fread(img->r, 1, xsize*ysize, fd);
  fclose(fd);
  if(fsz != xsize*ysize) {
    FreePicture(img);
    __Img_Busy(0);
    return IFALSE;
  }
  img->load = ITRUE;
  __Img_Busy(0);
  return ITRUE;
}
