/**************************************************************************
*  Image process tool box                                                 *
*     by Yang Yudong                                                      *
*     yangyd@yahoo.com                                                    *
*                                                                         *
***************************************************************************
*  Copyright (C) 1992-1999, Yang Yudong, All rights reserved.             *
*  This file is part of Yang Yudong's image processing software package.  *
*  If you use this software, you agree to the following:                  *
*  This program package is purely experimental, and is licensed "as is".  *
*  Permission is granted to use, modify, and distribute this program      *
*  without charge for any purpose, provided this license/ disclaimer      *
*  notice appears in the copies.  No warranty or maintenance is given,    *
*  either expressed or implied.  In no event shall the author(s) be       *
*  liable to you or a third party for any special, incidental,            *
*  consequential, or other damages, arising out of the use or inability   *
*  to use the program for any purpose (or the loss of data), even if we   *
*  have been advised of such possibilities.  Any public reference or      *
*  advertisement of this source code should refer to it as Yang Yudong's  *
*  orignal.                                                               *
**************************************************************************/
// ****************************************************************
//  Image process tool box
//     by Yang Yudong
//
// File : ibitbilt.c
// Description:  do bitbilt of images from one image to next image
//               two images can be the same one
// Create Date:
// Modification(date/where):  98/12/15  fixed aplha blending errors
//
// ****************************************************************
#include <malloc.h>
#include <string.h>
#include <stdio.h>


#define _IMG_LIBBUILD_
#include "imgdraw.h"
#include "routines.h"

//   do bitbilt (rectangle block copy) of src to des
//   des must be allocated and must be the same type as src
//    (xsts,ysts) is the start point in source image (can be negtive)
//    (xstd, ystd) is the destination location (can be negtive)
//    (xblk, yblk) is the size of expected rectangle block
// return ITRUE if successful

IBOOL img_bitbilt(ImageDes des, ImageDes src, 
                  int xsts, int ysts, int xblk, int yblk, int xstd, int ystd)
{
   int j;
   IBYTE *ss, *dd;
   
   if(!src.load || !des.alloc || xblk<=0 || yblk<=0 || src.imagetype != des.imagetype) return IFALSE;
   
   if(!__clip_rect(src.xsize, src.ysize, des.xsize, des.ysize, 
           &xsts, &ysts, &xstd, &ystd, &xblk, &yblk))  return ITRUE;

   if(src.alpha && des.alpha) 
   {
	   ss = src.a + xsts + ysts*src.xsize;
	   dd = des.a + xstd + ystd*des.xsize;
	   for(j = 0; j < yblk; j++, ss += src.xsize, dd += des.xsize) memmove(dd, ss, xblk);
   }

   ss = src.r + xsts + ysts*src.xsize;
   dd = des.r + xstd + ystd*des.xsize;
   for(j = 0; j < yblk; j++, ss += src.xsize, dd += des.xsize) memmove(dd, ss, xblk);

   if(src.imagetype != ITrueColor) return ITRUE;

   ss = src.g + xsts + ysts*src.xsize;
   dd = des.g + xstd + ystd*des.xsize;
   for(j = 0; j < yblk; j++, ss += src.xsize, dd += des.xsize) memmove(dd, ss, xblk);

   ss = src.b + xsts + ysts*src.xsize;
   dd = des.b + xstd + ystd*des.xsize;
   for(j = 0; j < yblk; j++, ss += src.xsize, dd += des.xsize) memmove(dd, ss, xblk);
   
   return ITRUE;       
}



//   do transparent bitbilt (rectangle block copy) of src to des
//   des must be allocated and must be the same type as src
//   where (xs,ys) is the start point of source image block (can be negtive)
//         (x, y) is the size of source image block
//         (dx, dy) is the destination location (can be negtive)
//         (tr, tg, tb) is the color that is transparent, 
//            only tr is used when image is not RGB true color.
// return ITRUE if successful
IBOOL img_bitbilt_transparent(ImageDes des, ImageDes src, 
     int xsts, int ysts, int xblk, int yblk, int xstd, int ystd, IBYTE tr, IBYTE tg, IBYTE tb)
{
   int i,j;
   IBYTE *sr, *dr, *sg, *dg, *sb, *db, *sa, *da;
   
   if(!src.load || !des.alloc || xblk<=0 || yblk<=0 || src.imagetype != des.imagetype) return IFALSE;

   if(!__clip_rect(src.xsize, src.ysize, des.xsize, des.ysize, 
           &xsts, &ysts, &xstd, &ystd, &xblk, &yblk))  return ITRUE;


   if(src.imagetype != ITrueColor) {
	   sr = src.r + xsts + ysts*src.xsize;
	   dr = des.r + xstd + ystd*des.xsize;
	   for(j = 0; j < yblk; j++, sr += src.xsize, dr += des.xsize) 
		   for(i=0; i<xblk; i++) if(*(sr+i) != tr) *(dr+i) = *(sr+i);
		   
	   if(src.alpha && des.alpha) 
	   {
		   sr = src.r + xsts + ysts*src.xsize;
		   dr = des.r + xstd + ystd*des.xsize;
		   sa = src.a + xsts + ysts*src.xsize;
		   da = des.a + xstd + ystd*des.xsize;
		   for(j = 0; j < yblk; j++, sr += src.xsize, dr += des.xsize, sa+= src.xsize, da+=des.xsize) 
			   for(i=0; i<xblk; i++) if(*(sr+i) != tr) *(da+i) = *(sa+i);
			   return ITRUE;
	   }
	   return ITRUE;
   }
   

   sr = src.r + xsts + ysts*src.xsize;
   sg = src.g + xsts + ysts*src.xsize;
   sb = src.b + xsts + ysts*src.xsize;
   dr = des.r + xstd + ystd*des.xsize;
   dg = des.g + xstd + ystd*des.xsize;
   db = des.b + xstd + ystd*des.xsize;
   for(j = 0; j < yblk; j++, sr += src.xsize, dr += des.xsize, sg += src.xsize, dg += des.xsize, sb += src.xsize, db += des.xsize) 
     for(i=0; i<xblk; i++) 
        if(*(sr+i) != tr || *(sg+i) != tg || *(sb+i) != tb) {
           *(dr+i) = *(sr+i);  *(dg+i) = *(sg+i);  *(db+i) = *(sb+i);
        }

   if(src.alpha && des.alpha) 
   {
	   sr = src.r + xsts + ysts*src.xsize;
	   sg = src.g + xsts + ysts*src.xsize;
	   sb = src.b + xsts + ysts*src.xsize;
	   dr = des.r + xstd + ystd*des.xsize;
	   dg = des.g + xstd + ystd*des.xsize;
	   db = des.b + xstd + ystd*des.xsize;
	   sa = src.a + xsts + ysts*src.xsize;
	   da = des.a + xstd + ystd*des.xsize;
	   for(j = 0; j < yblk; j++, sr += src.xsize, dr += des.xsize, sg += src.xsize, dg += des.xsize, sb += src.xsize, db += des.xsize, sa+= src.xsize, da+=des.xsize) 
		   for(i=0; i<xblk; i++) 
			   if(*(sr+i) != tr || *(sg+i) != tg || *(sb+i) != tb) {
				   *(da+i) = *(sa+i);
			   }
   }		
   return ITRUE;       
}



//   do alpha bitbilt (rectangle block copy) of src to des
//   des must be allocated and must be the same type as src
//    (xsts,ysts) is the start point in source image (can be negtive)
//    (xstd, ystd) is the destination location (can be negtive)
//    (xblk, yblk) is the size of expected rectangle block
// return ITRUE if successful

IBOOL img_bitbilt_alpha(ImageDes des, ImageDes src, IBYTE *alpha,
                  int xsts, int ysts, int xblk, int yblk, int xstd, int ystd)
{
   int i, j;
   IBYTE *ss, *dd, *aa;
   
   if(!src.load || !des.alloc || xblk<=0 || yblk<=0 || src.imagetype != des.imagetype) return IFALSE;
   if(src.imagetype == IColor256  || src.imagetype == IIndexedColor) return IFALSE;
   
   if(!__clip_rect(src.xsize, src.ysize, des.xsize, des.ysize, 
           &xsts, &ysts, &xstd, &ystd, &xblk, &yblk))  return ITRUE;

   
   if(src.imagetype != ITrueColor) {
      ss = src.r + xsts + ysts*src.xsize;
      aa = alpha + xstd + ystd*des.xsize;
      dd = des.r + xstd + ystd*des.xsize;
      for(j = 0; j < yblk; j++, ss += src.xsize, dd += des.xsize, aa+=des.xsize)
        for(i=0; i<xblk; i++) {
			dd[i]=(IBYTE)(((ILONG)ss[i]-(ILONG)dd[i])*(ILONG)aa[i]/255L+(ILONG)dd[i]);
		}
   }
   else {
      ss = src.r + xsts + ysts*src.xsize;
      aa = alpha + xstd + ystd*des.xsize;
      dd = des.r + xstd + ystd*des.xsize;
      for(j = 0; j < yblk; j++, ss += src.xsize, dd += des.xsize, aa+=des.xsize)
        for(i=0; i<xblk; i++) {
			dd[i]=(IBYTE)(((ILONG)ss[i]-(ILONG)dd[i])*(ILONG)aa[i]/255L+(ILONG)dd[i]);
		}
      
      ss = src.g + xsts + ysts*src.xsize;
      aa = alpha + xstd + ystd*des.xsize;
      dd = des.g + xstd + ystd*des.xsize;
      for(j = 0; j < yblk; j++, ss += src.xsize, dd += des.xsize, aa+=des.xsize)
        for(i=0; i<xblk; i++) {
			dd[i]=(IBYTE)(((ILONG)ss[i]-(ILONG)dd[i])*(ILONG)aa[i]/255L+(ILONG)dd[i]);
		}

      ss = src.b + xsts + ysts*src.xsize;
      aa = alpha + xstd + ystd*des.xsize;
      dd = des.b + xstd + ystd*des.xsize;
      for(j = 0; j < yblk; j++, ss += src.xsize, dd += des.xsize, aa+=des.xsize)
        for(i=0; i<xblk; i++) {
			dd[i]=(IBYTE)(((ILONG)ss[i]-(ILONG)dd[i])*(ILONG)aa[i]/255L+(ILONG)dd[i]);
		}
   }
   return ITRUE;       
}



//   do transparent alpha bitbilt (rectangle block copy) of src to des
//   des must be allocated and must be the same type as src
//   where (xs,ys) is the start point of source image block (can be negtive)
//         (x, y) is the size of source image block
//         (dx, dy) is the destination location (can be negtive)
//         (tr, tg, tb) is the color that is transparent, 
//            only tr is used when image is not RGB true color.
// return ITRUE if successful
IBOOL img_bitbilt_transparent_alpha(ImageDes des, ImageDes src, IBYTE *alpha,
     int xsts, int ysts, int xblk, int yblk, int xstd, int ystd, IBYTE tr, IBYTE tg, IBYTE tb)
{
   int i,j;
   IBYTE *sr, *dr, *sg, *dg, *sb, *db, *aa;
   
   if(!src.load || !des.alloc || xblk<=0 || yblk<=0 || src.imagetype != des.imagetype) return IFALSE;
   if(src.imagetype == IColor256  || src.imagetype == IIndexedColor) return IFALSE;

   if(!__clip_rect(src.xsize, src.ysize, des.xsize, des.ysize, 
           &xsts, &ysts, &xstd, &ystd, &xblk, &yblk))  return ITRUE;

   if(src.imagetype != ITrueColor) {
      sr = src.r + xsts + ysts*src.xsize;
      aa = alpha + xstd + ystd*des.xsize;
      dr = des.r + xstd + ystd*des.xsize;
      for(j = 0; j < yblk; j++, sr += src.xsize, dr += des.xsize, aa+=des.xsize) 
        for(i=0; i<xblk; i++)
			if(*(sr+i) != tr)
				dr[i]=(IBYTE)(((ILONG)sr[i]-(ILONG)dr[i])*(ILONG)aa[i]/255L+(ILONG)dr[i]);
      return ITRUE;
   }
   

   sr = src.r + xsts + ysts*src.xsize;
   sg = src.g + xsts + ysts*src.xsize;
   sb = src.b + xsts + ysts*src.xsize;
   aa = alpha + xstd + ystd*des.xsize;
   dr = des.r + xstd + ystd*des.xsize;
   dg = des.g + xstd + ystd*des.xsize;
   db = des.b + xstd + ystd*des.xsize;
   for(j = 0; j < yblk; j++, sr += src.xsize, dr += des.xsize, sg += src.xsize, dg += des.xsize, sb += src.xsize, db += des.xsize, aa+=des.xsize) 
     for(i=0; i<xblk; i++) 
        if(*(sr+i) != tr || *(sg+i) != tg || *(sb+i) != tb) {
			dr[i]=(IBYTE)(((ILONG)sr[i]-(ILONG)dr[i])*(ILONG)aa[i]/255L+(ILONG)dr[i]);
			dg[i]=(IBYTE)(((ILONG)sg[i]-(ILONG)dg[i])*(ILONG)aa[i]/255L+(ILONG)dg[i]);
			db[i]=(IBYTE)(((ILONG)sb[i]-(ILONG)db[i])*(ILONG)aa[i]/255L+(ILONG)db[i]);
        }
   
   return ITRUE;       
}

