/**************************************************************************
*  Image process tool box                                                 *
*     by Yang Yudong                                                      *
*     yangyd@yahoo.com                                                    *
*                                                                         *
***************************************************************************
*  Copyright (C) 1992-1999, Yang Yudong, All rights reserved.             *
*  This file is part of Yang Yudong's image processing software package.  *
*  If you use this software, you agree to the following:                  *
*  This program package is purely experimental, and is licensed "as is".  *
*  Permission is granted to use, modify, and distribute this program      *
*  without charge for any purpose, provided this license/ disclaimer      *
*  notice appears in the copies.  No warranty or maintenance is given,    *
*  either expressed or implied.  In no event shall the author(s) be       *
*  liable to you or a third party for any special, incidental,            *
*  consequential, or other damages, arising out of the use or inability   *
*  to use the program for any purpose (or the loss of data), even if we   *
*  have been advised of such possibilities.  Any public reference or      *
*  advertisement of this source code should refer to it as Yang Yudong's  *
*  orignal.                                                               *
**************************************************************************/
// ****************************************************************
//  Image process tool box
//     by Yang Yudong
//
// File : icylind.c
// Description:  do planar to cylinder transform of an image 
// Create Date:  1997. 7. 2
// Modification(date/where): 
//
// ****************************************************************

#include <malloc.h>
#include <stdio.h>
#include <string.h>
#include <float.h>
#include <math.h>
#include <stdlib.h>
#define _IMG_LIBBUILD_
#include "imgproc.h"
#include "mathutil.h"
//#include "camera.h"


#define D2R  (3.1415926535897932385/180.0)
#ifndef DBL_EPSILON
 #define DBL_EPSILON	2.22045e-16
#endif


// x' = (pj[0]*x + pj[1]*y + pj[2])/(pj[6]*x + pj[7]*y + pj[8])
// y' = (pj[3]*x + pj[4]*y + pj[5])/(pj[6]*x + pj[7]*y + pj[8])

//  Xcyl  = f*atan(x'/f);          // arc length of horizontal
//  Ycyl  = y'*f/sqrt(x'^2+f^2);   // height 


static void __cylinder_size(int ix, int iy, double cam[4], double roll, double pitch, double yaw, int *rx, int *ry)
{
  double x1, y1, x2, y2, x3, y3, x4, y4, x, y, div, pj[9];
  double camb[9];
  
  camb[0] = 0; camb[1] = 0; camb[2] = cam[2]; camb[3] = cam[3];
  construct_project_matrix(pj, roll, pitch, yaw, cam, camb);

  x = (double)ix/2.0;  y = (double)iy/2.0;

  // camera basic point is at (cam[0], cam[1]) of the image center coordinate
  // but now the cylinder image's center is the camera center

  // (-x, -y)       
  div = (pj[6]*(-x) + pj[7]*(-y) + pj[8]); 
  if(div == 0) div = 1e-10; // this is for judgement of infinite points
  x1 = (pj[0]*(-x) + pj[1]*(-y) + pj[2])/div;
  y1 = (pj[3]*(-x) + pj[4]*(-y) + pj[5])/div;

  // (x, -y)
  div = (pj[6]*x + pj[7]*(-y) + pj[8]); 
  if(div == 0) div = 1e-10;  // this is for judgement of infinite points
  x2 = (pj[0]*x + pj[1]*(-y) + pj[2])/div;
  y2 = (pj[3]*x + pj[4]*(-y) + pj[5])/div;

  // (x, y)
  div = (pj[6]*x + pj[7]*y + pj[8]); 
  if(div == 0) div = 1e-10;  // this is for judgement of infinite points
  x3 = (pj[0]*x + pj[1]*y + pj[2])/div;
  y3 = (pj[3]*x + pj[4]*y + pj[5])/div;

  // (-x, y)
  div = (pj[6]*(-x) + pj[7]*y + pj[8]); 
  if(div == 0) div = 1e-10;  // this is for judgement of infinite points
  x4 = (pj[0]*(-x) + pj[1]*y + pj[2])/div;
  y4 = (pj[3]*(-x) + pj[4]*y + pj[5])/div;

  y1 = fabs(y1*cam[3]/sqrt(cam[2]*cam[2]+x1*x1));
  y2 = fabs(y2*cam[3]/sqrt(cam[2]*cam[2]+x2*x2));
  y3 = fabs(y3*cam[3]/sqrt(cam[2]*cam[2]+x3*x3));
  y4 = fabs(y4*cam[3]/sqrt(cam[2]*cam[2]+x4*x4));

  x1 = fabs(cam[2]*atan(x1/cam[2])); 
  x2 = fabs(cam[2]*atan(x2/cam[2])); 
  x3 = fabs(cam[2]*atan(x3/cam[2])); 
  x4 = fabs(cam[2]*atan(x4/cam[2])); 
  x = x1;
  if(x2 > x) x = x2;
  if(x3 > x) x = x3;
  if(x4 > x) x = x4;

  y = y1;
  if(y2 > y) y = y2;
  if(y3 > y) y = y3;
  if(y4 > y) y = y4;

  *rx = (int)ceil(2*x);
  *ry = (int)ceil(2*y);
}


// (x, y)   = src-center_src;
// (x', y') = intermediate;
// (Xcyl, Ycly) = des-center_des;
// rp = pj^-1

// x' = (pj[0]*x + pj[1]*y + pj[2])/(pj[6]*x + pj[7]*y + pj[8])
// y' = (pj[3]*x + pj[4]*y + pj[5])/(pj[6]*x + pj[7]*y + pj[8])

//  Xcyl  = f*atan(x'/f);          // arc length of horizontal
//  Ycyl  = y'*f/sqrt(x'^2+f^2);   // height 

//  x' = f*tan(Xcyl/f) = f*tan((dx-Cdx)/f);
//  y' = Ycyl*sqrt(x'^2+f^2)/f  = (dy-Cdy)/cos((dx-Cdx)/f);
//  

// x = (rp[0]*x' + rp[1]*y' + rp[2])/(rp[6]*x' + rp[7]*y' + rp[8])
// y = (rp[3]*x' + rp[4]*y' + rp[5])/(rp[6]*x' + rp[7]*y' + rp[8])

// assume:
//    ky[dx] = 1/cos((dx-Cdx)/f)
//  sx = (rp[0]*f*tan((dx-Cdx)/f) + rp[1]*(dy-Cdy)*ky[dx] + rp[2])/(rp[6]*f*tan((dx-Cdx)/f) + rp[7]*(dy-Cdy)*ky[dx] + rp[8]) + Csx
//  sy = (rp[3]*f*tan((dx-Cdx)/f) + rp[4]*(dy-Cdy)*ky[dx] + rp[5])/(rp[6]*f*tan((dx-Cdx)/f) + rp[7]*(dy-Cdy)*ky[dx] + rp[8]) + Csy

static void __cylinder_trans(IBYTE *src, int sx, int sy,  IBYTE *des, int rx, int ry, 
                             double cam[4], double roll, double pitch, double yaw, IBYTE fill)
{ 
  int x, y, t;
  double rp[9], pj[9], crx, cry, csx, csy, focal;
  double *m0, *m3, *m6,  m1, m4, m7, *ky;
  int x16, y16, xx, yy, xp, yp, dx, dy;
  IBYTE *pp;
  long p00, p10, p01, p11;
  double camb[9];
  
  camb[0] = 0; camb[1] = 0; camb[2] = cam[2]; camb[3] = cam[3];
  construct_project_matrix(pj, roll, pitch, yaw, cam, camb);

  if(!__inv_mat_3x3(rp, pj)) {
     t = rx*ry;
     for(x=0; x<t; x++) des[x] = fill;
     return;
  }

  m0 = malloc(rx*4*sizeof(double));
  if(m0 == NULL) {
     t = rx*ry;
     for(x=0; x<t; x++) des[x] = fill;
     return;
  }
  m3 = m0+rx;   m6 = m3+rx; 
  ky = m6+rx;
  
  csx = (double)sx/2;   csy = (double)sy/2;
  crx = (double)rx/2;   cry = (double)ry/2;
  focal = cam[2];
  for(x=0; x<rx; x++) {
    m0[x] = (rp[0]*focal*tan(((double)x-crx)/focal) + rp[2])*1024.0;                           
    m3[x] = (rp[3]*focal*tan(((double)x-crx)/focal) + rp[5])*1024.0;
    m6[x] = (rp[6]*focal*tan(((double)x-crx)/focal) + rp[8]);
    ky[x] = 1/cos(((double)x-crx)/focal);
  }
  
  csx *= 1024.0;  csy *= 1024.0;
  x16 = (sx-1)<<10;       y16 = (sy-1)<<10;

  for(y=0; y<ry; y++) {
    m1 = rp[1]*(y-cry)*1024.0;                           
    m4 = rp[4]*(y-cry)*1024.0;
    m7 = rp[7]*(y-cry);
    for(x=0; x<rx; x++, des++) {
      xx = (int)((m0[x]+m1*ky[x])/(m6[x]+m7*ky[x])+csx);
      yy = (int)((m3[x]+m4*ky[x])/(m6[x]+m7*ky[x])+csy);
      if(xx<0 || yy<0 || xx>x16 || yy>y16) *des = fill;
      else {
        xp = xx>>10; yp = yy>>10; dx = xx&0x3ff; dy = yy&0x3ff;
        pp = src+xp+yp*sx; 
        p00 = *pp;  
        if(xx < x16) {
          p10 = *(pp+1); 
          if(yy < y16) { p01 = *(pp+sx);  p11 = *(pp+sx+1);}
          else { p01 = *pp;  p11 = *(pp+1); }
        }
        else { // xx = x16
          p10 = *pp; 
          if(yy < y16) { p01 = *(pp+sx);  p11 = *(pp+sx); }
          else { p01 = *pp;   p11 = *pp; }
        }
        *des = (IBYTE)(((p11*dx+p01*(1024-dx))*dy + (p10*dx+p00*(1024-dx))*(1024-dy))>>20);
      }
    }
  }  
  free(m0);
  return ;
}  

static void __cylinder_trans_fast(IBYTE *src, int sx, int sy,  IBYTE *des, int rx, int ry, 
                             double cam[4], double roll, double pitch, double yaw, IBYTE fill)
{ 
  int x, y, t;
  double rp[9], pj[9], crx, cry, csx, csy, focal;
  double *m0, *m3, *m6,  m1, m4, m7, *ky;
  int x16, y16, xx, yy, xp, yp;
  double camb[9];
  
  camb[0] = 0; camb[1] = 0; camb[2] = cam[2]; camb[3] = cam[3];
  construct_project_matrix(pj, roll, pitch, yaw, cam, camb);

  if(!__inv_mat_3x3(rp, pj)) {
     t = rx*ry;
     for(x=0; x<t; x++) des[x] = fill;
     return;
  }

  m0 = malloc(rx*4*sizeof(double));
  if(m0 == NULL) {
     t = rx*ry;
     for(x=0; x<t; x++) des[x] = fill;
     return;
  }
  m3 = m0+rx;   m6 = m3+rx; 
  ky = m6+rx;
  
  csx = (double)sx/2;   csy = (double)sy/2;
  crx = (double)rx/2;   cry = (double)ry/2;
  focal = cam[2];
  for(x=0; x<rx; x++) {
    m0[x] = (rp[0]*focal*tan(((double)x-crx)/focal) + rp[2])*1024.0;                           
    m3[x] = (rp[3]*focal*tan(((double)x-crx)/focal) + rp[5])*1024.0;
    m6[x] = (rp[6]*focal*tan(((double)x-crx)/focal) + rp[8]);
    ky[x] = 1/cos(((double)x-crx)/focal);
  }
  
  csx *= 1024.0;  csy *= 1024.0;
  x16 = (sx-1)<<10;       y16 = (sy-1)<<10;

  for(y=0; y<ry; y++) {
    m1 = rp[1]*(y-cry)*1024.0;                           
    m4 = rp[4]*(y-cry)*1024.0;
    m7 = rp[7]*(y-cry);
    for(x=0; x<rx; x++, des++) {
      xx = (int)((m0[x]+m1*ky[x])/(m6[x]+m7*ky[x])+csx);
      yy = (int)((m3[x]+m4*ky[x])/(m6[x]+m7*ky[x])+csy);
      if(xx<0 || yy<0 || xx>x16 || yy>y16) *des = fill;
      else {
        xp = (xx+512)>>10; yp = (yy+512)>>10;
        *des = *(src+xp+yp*sx); 
      }
    }
  }
  free(m0);
  return ;
}  

// Do planar to cylinder transformation (image center) of img to get des, des is the same size as img
// Positive tx is translate right,  Positive ty is translate down 

IBOOL  image_cylinder_deform(ImageDes *des, ImageDes img, double cam[4], double roll, double pitch, double yaw, IDWORD fills, IInterpType interpolation)
{
  
  if(!img.load) return IFALSE;
  __Img_Busy(1);
    
  if(!AllocPicture(des, img.xsize, img.ysize, img.imagetype, img.alpha, img.numColors)) {__Img_Busy(0); return IFALSE;}
  des->load = ITRUE;
  des->transparent = img.transparent;
  des->trans = img.trans;
  des->background = img.background;
  des->back = img.back;
  des->gamma = img.gamma;
  if(img.imagetype == IColor256  || img.imagetype == IIndexedColor) {
	  interpolation = IInterp_fast;
	  memcpy(des->pal, img.pal, sizeof(IMyRGB)*256);
  }
  switch (interpolation) {
    case IInterp_normal:   // or bilinear interpolation
    case IInterp_fine  :   // or bicubic interpolation
	   if(des->alpha) __cylinder_trans(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, cam, roll, pitch, yaw, DWORD_COLOR_ALPHA(fills));
       __cylinder_trans(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, cam, roll, pitch, yaw, DWORD_COLOR_RED(fills));
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __cylinder_trans(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, cam, roll, pitch, yaw, DWORD_COLOR_GREEN(fills));
         __Img_Busy(66);
         __cylinder_trans(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, cam, roll, pitch, yaw, DWORD_COLOR_BLUE(fills));
         __Img_Busy(99);
       }
       break;
    case IInterp_fast  :   // or near neighbor interpolation (palette image can only do this)
	   if(des->alpha) __cylinder_trans_fast(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, cam, roll, pitch, yaw, DWORD_COLOR_ALPHA(fills));
       __cylinder_trans_fast(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, cam, roll, pitch, yaw, DWORD_COLOR_RED(fills));
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __cylinder_trans_fast(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, cam, roll, pitch, yaw, DWORD_COLOR_GREEN(fills));
         __Img_Busy(66);
         __cylinder_trans_fast(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, cam, roll, pitch, yaw, DWORD_COLOR_BLUE(fills));
         __Img_Busy(99);
       }
       break;
  }
  __Img_Busy(0);
  return ITRUE;
}  




// Do planar to cylinder transformation (image center) of img to get des, des contain the full result
// Positive tx is translate right,  Positive ty is translate down 
IBOOL  image_cylinder_deform_size(ImageDes *des, ImageDes img, double cam[4], double roll, double pitch, double yaw, IDWORD fills, IInterpType interpolation)
{
  int dx, dy;
  
  if(!img.load) return IFALSE;

  __Img_Busy(1);
    
  __cylinder_size(img.xsize, img.ysize, cam, roll, pitch, yaw, &dx, &dy);
  if(!AllocPicture(des, dx, dy, img.imagetype, img.alpha, img.numColors)) {__Img_Busy(0); return IFALSE;}

  des->load = ITRUE;
  des->transparent = img.transparent;
  des->trans = img.trans;
  des->background = img.background;
  des->back = img.back;
  des->gamma = img.gamma;

  if(img.imagetype == IColor256 || img.imagetype == IIndexedColor) {
	  interpolation = IInterp_fast;
	  memcpy(des->pal, img.pal, sizeof(IMyRGB)*256);
  }
  switch (interpolation) {
    case IInterp_normal:   // or bilinear interpolation
    case IInterp_fine  :   // or bicubic interpolation
	   if(des->alpha) __cylinder_trans(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, cam, roll, pitch, yaw, DWORD_COLOR_ALPHA(fills));
       __cylinder_trans(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, cam, roll, pitch, yaw, DWORD_COLOR_RED(fills));
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __cylinder_trans(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, cam, roll, pitch, yaw, DWORD_COLOR_GREEN(fills));
         __Img_Busy(66);
         __cylinder_trans(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, cam, roll, pitch, yaw, DWORD_COLOR_BLUE(fills));
         __Img_Busy(99);
       }
       break;
    case IInterp_fast  :   // or near neighbor interpolation (palette image can only do this)
	   if(des->alpha) __cylinder_trans_fast(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, cam, roll, pitch, yaw, DWORD_COLOR_ALPHA(fills));
       __cylinder_trans_fast(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, cam, roll, pitch, yaw, DWORD_COLOR_RED(fills));
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __cylinder_trans_fast(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, cam, roll, pitch, yaw, DWORD_COLOR_GREEN(fills));
         __Img_Busy(66);
         __cylinder_trans_fast(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, cam, roll, pitch, yaw, DWORD_COLOR_BLUE(fills));
         __Img_Busy(99);
       }
       break;
  }
  __Img_Busy(0);
  return ITRUE;
}  


