/**************************************************************************
*  Image process tool box                                                 *
*     by Yang Yudong                                                      *
*     yangyd@yahoo.com                                                    *
*                                                                         *
***************************************************************************
*  Copyright (C) 1992-1999, Yang Yudong, All rights reserved.             *
*  This file is part of Yang Yudong's image processing software package.  *
*  If you use this software, you agree to the following:                  *
*  This program package is purely experimental, and is licensed "as is".  *
*  Permission is granted to use, modify, and distribute this program      *
*  without charge for any purpose, provided this license/ disclaimer      *
*  notice appears in the copies.  No warranty or maintenance is given,    *
*  either expressed or implied.  In no event shall the author(s) be       *
*  liable to you or a third party for any special, incidental,            *
*  consequential, or other damages, arising out of the use or inability   *
*  to use the program for any purpose (or the loss of data), even if we   *
*  have been advised of such possibilities.  Any public reference or      *
*  advertisement of this source code should refer to it as Yang Yudong's  *
*  orignal.                                                               *
**************************************************************************/
//****************************************************************
//  Image process tool box
//     by Yang Yudong
//
// File : imgalloc.c
// Description: Allocate and free image memory 
// Create Date:  1996. 9. 25
// Modification(date/where):   1999. 12. 03  add limitted support for less than 16 indexed 
//                                           color and transparent image
//					Adam Hoult-2000, 03, 01  added ReAllocPicture to simply reallocate space
//											 to an image after resizing for example.
//											 useful if when calling resample the source and
//											 Destination Images are identical.
//
//****************************************************************

#include <malloc.h>
#include <string.h>
#include <stdio.h>

#define _IMG_LIBBUILD_
#include "image.h"

//AllocPicture is used to allocate image memory
//   It'll free an allocated image descriptor if it has allocated memories and they
//   does not fit the required or it'll adjust the existing ones to fit the needs.
//   The allocated buffer will be resized only if imagetype changes from grayscale/indexed
//   to true color. Other wise old contents are also not destroyed.
//   The size and type fields will be filled and load flag will be cleared
//Return ITRUE if successful, otherwise IFALSE.
 
IBOOL AllocPicture(ImageDes *picture, int xsize, int ysize, IImageType imagetype, IBOOL alpha, IDWORD numColor)
{
  #ifdef _DEBUG
     fprintf(stderr, "Allocating picture memory...");
  #endif
  if(xsize <= 0 || ysize <= 0 || numColor > 256) return IFALSE;
  if(picture->alloc) {
     IDWORD wantsize = (IDWORD)xsize*(IDWORD)ysize;
	 if(picture->_allocated_size >= wantsize ) // reusable
	 {
		 switch(imagetype)
		 {
			case IGrey:
				switch(picture->imagetype)
				{
				case IIndexedColor:
				case IColor256:	// fall through to IGrey
		            free(picture->pal);
					picture->pal = NULL;
				case IGrey:
					// just use the old one
					// picture->grey = (IBYTE *)realloc(picture->grey, xsize*ysize); // should not fail at all
					break;
				case ITrueColor:
					// picture->r = (IBYTE *)realloc(picture->r, xsize*ysize); // should not fail at all
					free(picture->g); picture->g = NULL;
					free(picture->b); picture->b = NULL;
					break;
				}
				break;
			case IIndexedColor:
			case IColor256:
				switch(picture->imagetype)
				{
				case IIndexedColor:
				case IColor256:
					break;
				case ITrueColor:  // fall through to IGrey
					free(picture->g); picture->g = NULL;
					free(picture->b); picture->b = NULL;
				case IGrey:
					// picture->index = (IBYTE *)realloc(picture->grey, xsize*ysize); // should not fail at all
					picture->pal = (IMyRGB *)malloc(sizeof(IMyRGB)*256);
					if(picture->pal == NULL) 
					{ 
						picture->alloc = IFALSE;
						if(picture->alpha) free(picture->a);
						free(picture->r /*index*/); 
						#ifdef _DEBUG
						fprintf(stderr, "Fail.\n");
						#endif
						return IFALSE; 
					}
					break;
				}
				break;
			case ITrueColor:
				switch(picture->imagetype)
				{
				case ITrueColor:
					break;
				case IIndexedColor:
				case IColor256:  // fall through to IGrey
		            free(picture->pal);
					picture->pal = NULL;
				case IGrey:
					// the buffer size will be changed if new channels are allocated
					picture->_allocated_size = (IDWORD)xsize*(IDWORD)ysize;
					picture->r = (IBYTE *)realloc(picture->r /*grey*/, picture->_allocated_size); // should not fail at all
					picture->g = (IBYTE *)malloc(xsize*ysize);
					if(picture->g == NULL) {
						picture->alloc = IFALSE;
						if(picture->alpha) free(picture->a);
						free(picture->r); 
						#ifdef _DEBUG
						fprintf(stderr, "Fail.\n");
						#endif
						return IFALSE;
					}
					picture->b = (IBYTE *)malloc(xsize*ysize);
					if(picture->b == NULL) {
						picture->alloc = IFALSE;
						if(picture->alpha) free(picture->a);
						free(picture->r);
						free(picture->g);
						#ifdef _DEBUG
						fprintf(stderr, "Fail.\n");
						#endif
						return IFALSE;
					}
					break;
				}
				break;
		 }
		 if(picture->alpha)
		 {
			 if(alpha) 
				 picture->a = (IBYTE *)realloc(picture->a, picture->_allocated_size);
			 else
			 {
				 free(picture->a);
				 picture->a = NULL;
				 picture->alpha = IFALSE;
			 }
		 }
		 else if(alpha) 
		 {
			 picture->a = (IBYTE *)malloc(picture->_allocated_size);
			 if(picture->a == NULL)
			 {
				 picture->alloc = IFALSE;
				 if(picture->imagetype == ITrueColor)
				 {
					free(picture->g); picture->g = NULL;
					free(picture->b); picture->b = NULL;
				 }
				 else if(picture->imagetype != IGrey) // indexed color
				 {
					 free(picture->pal); picture->pal = NULL;
				 }
				 free(picture->r);
				 #ifdef _DEBUG
				 fprintf(stderr, "Fail.\n");
				 #endif
				 return IFALSE;
			 }
		 }

		 picture->imagetype = imagetype;
		 
		 // this value will not change here
		 // picture->_allocated_size = (IDWORD)xsize*(IDWORD)ysize;

		 picture->load  = IFALSE;
		 picture->xsize = xsize;
		 picture->ysize = ysize;
		 if(imagetype == ITrueColor || imagetype == IColor256)   picture->numColors = 256;
		 else picture->numColors = numColor;
		 picture->transparent = IFALSE;
		 picture->background = IFALSE;
		 picture->gamma = 0;  // no gamma
		 //if(imagetype == IColor256 || imagetype == IIndexedColor) 
		 //	 memset(picture->pal, 0, sizeof(IMyRGB)*256);
		 return ITRUE;
	 } //	 if(picture->_allocated_size >= wantsize ) // reusable

     FreePicture(picture);
  }

  picture->load  = IFALSE;
  picture->alloc = IFALSE;
  picture->imagetype  = imagetype;
  picture->xsize = xsize;
  picture->ysize = ysize;
  picture->r = picture->g = picture->b = NULL;

  picture->alpha = alpha;
  picture->a = NULL;
  if(imagetype == ITrueColor || imagetype == IColor256)   picture->numColors = 256;
  else picture->numColors = numColor;
  picture->transparent = IFALSE;
  picture->background = IFALSE;
  picture->gamma = 0;  // no gamma
  picture->_allocated_size = (IDWORD)xsize*(IDWORD)ysize;

  if(picture->alpha)
  {
     picture->a = (IBYTE *)malloc(xsize*ysize);
     if(picture->a == NULL) 
	 {
		 return IFALSE;
	 }
  }

  picture->r = (IBYTE *)malloc(xsize*ysize);
  if(picture->r == NULL) 
  {
	  if(picture->alpha) free(picture->a);
	  return IFALSE;
  }
  if(imagetype == IGrey) {
    picture->alloc = ITRUE;
    #ifdef _DEBUG
      fprintf(stderr, "Succeed.\n");
    #endif
    return ITRUE;
  }


  if(imagetype == IColor256 || imagetype == IIndexedColor) {
    picture->pal = (IMyRGB *)malloc(sizeof(IMyRGB)*256); // just allocate maximized color palette
    if(picture->pal == NULL) 
	{ 
	    if(picture->alpha) free(picture->a);
		free(picture->r); 
	    #ifdef _DEBUG
	      fprintf(stderr, "Fail.\n");
	    #endif
		return IFALSE; 
	}
    picture->alloc = ITRUE;
    #ifdef _DEBUG
      fprintf(stderr, "Succeed.\n");
    #endif
	memset(picture->pal, 0, sizeof(IMyRGB)*256);
    return ITRUE;
  }
  picture->pal = NULL;
  picture->g = (IBYTE *)malloc(xsize*ysize);
  if(picture->g == NULL) {
    if(picture->alpha) free(picture->a);
	free(picture->r); 
    #ifdef _DEBUG
      fprintf(stderr, "Fail.\n");
    #endif
    return IFALSE;
  }
  picture->b = (IBYTE *)malloc(xsize*ysize);
  if(picture->b == NULL) {
    if(picture->alpha) free(picture->a);
    free(picture->r);
    free(picture->g);
    #ifdef _DEBUG
      fprintf(stderr, "Fail.\n");
    #endif
    return IFALSE;
  }
  picture->alloc = ITRUE;
  #ifdef _DEBUG
    fprintf(stderr, "Succeed.\n");
  #endif
  return ITRUE;
}


//FreePicture is used to free image memory
//   It'll free the image descriptor if it has an allocated memory
//   All fields will be cleaned.
//Return none.

void FreePicture(ImageDes *picture)
{
  if(!picture->alloc) return;
  #ifdef _DEBUG
     fprintf(stderr, "Freeing picture memory...");
  #endif
  switch(picture->imagetype) {
      case IColor256:
	  case IIndexedColor:
          free(picture->pal);
		  free(picture->r /*index*/);
		  break; 
      case IGrey:
		  free(picture->r /*grey*/);
          break;
      case ITrueColor:
          free(picture->r);
          free(picture->g);
          free(picture->b);
          break;
  }
  picture->r = picture->g = picture->b = NULL;
  picture->xsize = picture->ysize = 0;
  picture->alloc = picture->load = IFALSE;
  picture->_allocated_size = 0;
  picture->pal = NULL;
  if(picture->alpha) free(picture->a);
  picture->a = NULL;
  picture->alpha = IFALSE;
  picture->transparent = IFALSE;
  picture->background = IFALSE;
  //picture->index = NULL;
  #ifdef _DEBUG
     fprintf(stderr, "Done.\n");
  #endif
  return;
}



//InitPicture is used to initialize a picture header
//   It'll free an allocated image descriptor
//   The size and type fields will be filled and load flag will be cleared
//Return ITRUE if successful, otherwise IFALSE.
IBOOL InitPicture(ImageDes *picture, int xsize, int ysize, IImageType imagetype, IBOOL alpha, IDWORD numColor)
{
  #ifdef _DEBUG
     fprintf(stderr, "Initializing picture header ...");
  #endif
  if(xsize <= 0 || ysize <= 0 || numColor > 256) return IFALSE;
  if(picture->alloc) {
     FreePicture(picture);
  }
  picture->load  = IFALSE;
  picture->alloc = IFALSE;
  picture->imagetype  = imagetype;
  picture->xsize = xsize;
  picture->ysize = ysize;
  picture->r = picture->g = picture->b = NULL;
  picture->pal = NULL;
  picture->_allocated_size = 0;

  picture->alpha = alpha;
  picture->a = NULL;
  if(imagetype == ITrueColor || imagetype == IColor256)   picture->numColors = 256;
  else picture->numColors = numColor;
  picture->transparent = IFALSE;
  picture->background = IFALSE;
  picture->gamma = 0;  // no gamma

  return ITRUE;
}


//	Adam Hoult-2000, 03, 01  added ReAllocPicture to simply reallocate space
//											 to an image after resizing for example.
//											 useful if when calling resample the source and
//											 Destination Images are identical.
IBOOL ReAllocPicture(ImageDes *picture, int xsize, int ysize, IImageType imagetype, IBOOL alpha)
{
  #ifdef _DEBUG
     fprintf(stderr, "Re-Allocating picture memory...");
  #endif
  if(xsize <= 0 || ysize <= 0) return IFALSE;
  if(picture->alloc) {
     IDWORD wantsize = (IDWORD)xsize*(IDWORD)ysize;
	 switch(imagetype)
	 {
		case IGrey:
			switch(picture->imagetype)
			{
			case IIndexedColor:
			case IColor256:	// fall through to IGrey
		        free(picture->pal);
				picture->pal = NULL;
			case IGrey:
				picture->r /*grey*/ = (IBYTE *)realloc(picture->r /*grey*/, xsize*ysize); // should not fail at all
				break;
			case ITrueColor:
				picture->r = (IBYTE *)realloc(picture->r, xsize*ysize); // should not fail at all
				free(picture->g); picture->g = NULL;
				free(picture->b); picture->b = NULL;
				break;
			}
			break;
		case IIndexedColor:
		case IColor256:
			switch(picture->imagetype)
			{
			case IIndexedColor:
			case IColor256:
				picture->r = (IBYTE *)realloc(picture->r, xsize*ysize); // should not fail at all
				break;
			case ITrueColor:  // fall through to IGrey
				picture->r = (IBYTE *)realloc(picture->r, xsize*ysize); // should not fail at all
				free(picture->g); picture->g = NULL;
				free(picture->b); picture->b = NULL;
			case IGrey:
				picture->r /*index*/ = (IBYTE *)realloc(picture->r /*grey*/, xsize*ysize); // should not fail at all
				picture->pal = (IMyRGB *)malloc(sizeof(IMyRGB)*256);
				if(picture->pal == NULL) 
				{ 
					picture->alloc = IFALSE;
					if(picture->alpha) free(picture->a);
					free(picture->r /*index*/); 
					#ifdef _DEBUG
					fprintf(stderr, "Fail.\n");
					#endif
					return IFALSE; 
				}
				break;
			}
			break;
		case ITrueColor:
			switch(picture->imagetype)
			{
			case ITrueColor:
				//Just simply resize the image buffer
				picture->r = (IBYTE *)realloc(picture->r, xsize*ysize); // should not fail at all
				picture->g = (IBYTE *)realloc(picture->g, xsize*ysize); // should not fail at all
				picture->b = (IBYTE *)realloc(picture->b, xsize*ysize); // should not fail at all
				break;
			case IIndexedColor:
			case IColor256:  // fall through to IGrey
				free(picture->pal);
				picture->pal = NULL;
			case IGrey:
				picture->r = (IBYTE *)realloc(picture->r /*grey*/, xsize*ysize); // should not fail at all
				picture->g = (IBYTE *)malloc(xsize*ysize);
				if(picture->g == NULL) {
					picture->alloc = IFALSE;
					if(picture->alpha) free(picture->a);
					free(picture->r); 
					#ifdef _DEBUG
					fprintf(stderr, "Fail.\n");
					#endif
					return IFALSE;
				}
				picture->b = (IBYTE *)malloc(xsize*ysize);
				if(picture->b == NULL) {
					picture->alloc = IFALSE;
					if(picture->alpha) free(picture->a);
					free(picture->r);
					free(picture->g);
					#ifdef _DEBUG
					fprintf(stderr, "Fail.\n");
					#endif
					return IFALSE;
				}
				break;
			}
			break;
	 }
	 if(picture->alpha)
	 {
		 if(alpha) 
			 picture->a = (IBYTE *)realloc(picture->a, xsize*ysize);
		 else
		 {
			 free(picture->a);
			 picture->a = NULL;
			 picture->alpha = IFALSE;
		 }
	 }
	 else if(alpha) 
	 {
		 picture->a = (IBYTE *)malloc(xsize*ysize);
		 if(picture->a == NULL)
		 {
			 picture->alloc = IFALSE;
			 if(picture->imagetype == ITrueColor)
			 {
				free(picture->g); picture->g = NULL;
				free(picture->b); picture->b = NULL;
			 }
			 else if(picture->imagetype != IGrey) // indexed color
			 {
				 free(picture->pal); picture->pal = NULL;
			 }
			 free(picture->r);
			 #ifdef _DEBUG
			 fprintf(stderr, "Fail.\n");
			 #endif
			 return IFALSE;
		 }
	 }

	 picture->imagetype = imagetype;
	 picture->_allocated_size = (IDWORD)xsize*(IDWORD)ysize;

	 picture->xsize = xsize;
	 picture->ysize = ysize;
	 return ITRUE;
  }
  return IFALSE;
}


