/**************************************************************************
*  Image process tool box                                                 *
*     by Yang Yudong                                                      *
*     yangyd@yahoo.com                                                    *
*                                                                         *
***************************************************************************
*  Copyright (C) 1992-1999, Yang Yudong, All rights reserved.             *
*  This file is part of Yang Yudong's image processing software package.  *
*  If you use this software, you agree to the following:                  *
*  This program package is purely experimental, and is licensed "as is".  *
*  Permission is granted to use, modify, and distribute this program      *
*  without charge for any purpose, provided this license/ disclaimer      *
*  notice appears in the copies.  No warranty or maintenance is given,    *
*  either expressed or implied.  In no event shall the author(s) be       *
*  liable to you or a third party for any special, incidental,            *
*  consequential, or other damages, arising out of the use or inability   *
*  to use the program for any purpose (or the loss of data), even if we   *
*  have been advised of such possibilities.  Any public reference or      *
*  advertisement of this source code should refer to it as Yang Yudong's  *
*  orignal.                                                               *
**************************************************************************/
// ****************************************************************
//  Image process tool box
//     by Yang Yudong
//
// File : iproject.c
// Description:  do full 2D projective transformation of an image 
// Create Date:  1997. 6. 29
// Modification(date/where): 
//
// ****************************************************************

#include <malloc.h>
#include <stdio.h>
#include <string.h>
#include <float.h>
#include <math.h>
#include <stdlib.h>
#define _IMG_LIBBUILD_
#include "imgproc.h"
#include "mathutil.h"

#define D2R  (3.1415926535897932385/180.0)
#ifndef DBL_EPSILON
 #define DBL_EPSILON	2.22045e-16
#endif

// x' = (pj[0]*x + pj[1]*y + pj[2])/(pj[6]*x + pj[7]*y + pj[8])
// y' = (pj[3]*x + pj[4]*y + pj[5])/(pj[6]*x + pj[7]*y + pj[8])




static void __project_size(int ix, int iy, double pj[9], int *rx, int *ry)
{
  double x1, y1, x2, y2, x3, y3, x4, y4, x, y, div;

  x = (double)ix/2.0;  y = (double)iy/2.0;

  // (-x, -y)
  div = fabs(pj[6]*(-x) + pj[7]*(-y) + pj[8]); 
  if(div == 0) div = 1e-10; // this is for judgement of infinite points
  x1 = fabs(pj[0]*(-x) + pj[1]*(-y) + pj[2])/div;
  y1 = fabs(pj[3]*(-x) + pj[4]*(-y) + pj[5])/div;

  // (x, -y)
  div = fabs(pj[6]*x + pj[7]*(-y) + pj[8]); 
  if(div == 0) div = 1e-10;  // this is for judgement of infinite points
  x2 = fabs(pj[0]*x + pj[1]*(-y) + pj[2])/div;
  y2 = fabs(pj[3]*x + pj[4]*(-y) + pj[5])/div;

  // (x, y)
  div = fabs(pj[6]*x + pj[7]*y + pj[8]); 
  if(div == 0) div = 1e-10;  // this is for judgement of infinite points
  x3 = fabs(pj[0]*x + pj[1]*y + pj[2])/div;
  y3 = fabs(pj[3]*x + pj[4]*y + pj[5])/div;

  // (-x, y)
  div = fabs(pj[6]*(-x) + pj[7]*y + pj[8]); 
  if(div == 0) div = 1e-10;  // this is for judgement of infinite points
  x4 = fabs(pj[0]*(-x) + pj[1]*y + pj[2])/div;
  y4 = fabs(pj[3]*(-x) + pj[4]*y + pj[5])/div;

  x = x1;
  if(x2 > x) x = x2;
  if(x3 > x) x = x3;
  if(x4 > x) x = x4;

  y = y1;
  if(y2 > y) y = y2;
  if(y3 > y) y = y3;
  if(y4 > y) y = y4;

  *rx = (int)ceil(2*x);
  *ry = (int)ceil(2*y);
}


// (x, y)   = src-center_src;
// (x', y') = des-center_des;
// rp = pj^-1

// x' = (pj[0]*x + pj[1]*y + pj[2])/(pj[6]*x + pj[7]*y + pj[8])
// y' = (pj[3]*x + pj[4]*y + pj[5])/(pj[6]*x + pj[7]*y + pj[8])

// x = (rp[0]*x' + rp[1]*y' + rp[2])/(rp[6]*x' + rp[7]*y' + rp[8])
// y = (rp[3]*x' + rp[4]*y' + rp[5])/(rp[6]*x' + rp[7]*y' + rp[8])

// assume: 
//    px = -rp[0]*Cdx - rp[1]*Cdy + rp[2]
//    py = -rp[3]*Cdx - rp[4]*Cdy + rp[5]
//    pk = -rp[6]*Cdx - rp[7]*Cdy + rp[8]                           
// then:
//  sx = (rp[0]*dx + rp[1]*dy + px)/(rp[6]*dx + rp[7]*dy + pk) + Csx
//  sy = (rp[3]*dx + rp[4]*dy + py)/(rp[6]*dx + rp[7]*dy + pk) + Csy

static void __proj_trans(IBYTE *src, int sx, int sy,  IBYTE *des, int rx, int ry, double pj[9], IBYTE fill)
{ 
  int x, y, t;
  double rp[9], px, py, pk, crx, cry, csx, csy, dp;
  double *m0, *m3, *m6,  m1, m4, m7;
  int x16, y16, xx, yy, xp, yp, dx, dy;
  IBYTE *pp;
  long p00, p10, p01, p11;

  
  if(!__inv_mat_3x3(rp, pj)) {
     t = rx*ry;
     for(x=0; x<t; x++) des[x] = fill;
     return;
  }

  m0 = malloc(rx*3*sizeof(double));
  if(m0 == NULL) {
     t = rx*ry;
     for(x=0; x<t; x++) des[x] = fill;
     return;
  }
  m3 = m0+rx;   m6 = m3+rx; 
  
  csx = (double)sx/2;   csy = (double)sy/2;
  crx = (double)rx/2;   cry = (double)ry/2;
  px = -rp[0]*crx - rp[1]*cry + rp[2];
  py = -rp[3]*crx - rp[4]*cry + rp[5];
  pk = -rp[6]*crx - rp[7]*cry + rp[8];
  for(x=0; x<rx; x++) {
    m0[x] = (rp[0]*(double)x + px)*1024.0;                           
    m3[x] = (rp[3]*(double)x + py)*1024.0;
    m6[x] = rp[6]*(double)x + pk;
  }
  
  csx *= 1024.0;  csy *= 1024.0;
  x16 = (sx-1)<<10;       y16 = (sy-1)<<10;

  for(y=0; y<ry; y++) {
    m1 = rp[1]*(double)y*1024.0;                           
    m4 = rp[4]*(double)y*1024.0;
    m7 = rp[7]*(double)y;
    for(x=0; x<rx; x++, des++) {
      dp = m6[x]+m7;
      if(fabs(dp) < 1000*DBL_EPSILON) continue;  // the projection have infinite points
      xx = (int)((m0[x]+m1)/dp+csx);
      yy = (int)((m3[x]+m4)/dp+csy);
      if(xx<0 || yy<0 || xx>x16 || yy>y16) *des = fill;
      else {
        xp = xx>>10; yp = yy>>10; dx = xx&0x3ff; dy = yy&0x3ff;
        pp = src+xp+yp*sx; 
        p00 = *pp;  
        if(xx < x16) {
          p10 = *(pp+1); 
          if(yy < y16) { p01 = *(pp+sx);  p11 = *(pp+sx+1);}
          else { p01 = *pp;  p11 = *(pp+1); }
        }
        else { // xx = x16
          p10 = *pp; 
          if(yy < y16) { p01 = *(pp+sx);  p11 = *(pp+sx); }
          else { p01 = *pp;   p11 = *pp; }
        }
        *des = (IBYTE)(((p11*dx+p01*(1024-dx))*dy + (p10*dx+p00*(1024-dx))*(1024-dy))>>20);
      }
    }
  }  
  free(m0);
  return ;
}  

static void __proj_trans_fast(IBYTE *src, int sx, int sy,  IBYTE *des, int rx, int ry, double pj[9], IBYTE fill)
{ 
  int x, y, t;
  double rp[9], px, py, pk, crx, cry, csx, csy, dp;
  double *m0, *m3, *m6,  m1, m4, m7;
  int x16, y16, xx, yy, xp, yp;
  
  if(!__inv_mat_3x3(rp, pj)) {
     t = rx*ry;
     for(x=0; x<t; x++) des[x] = fill;
     return;
  }

  m0 = malloc(rx*3*sizeof(double));
  if(m0 == NULL) {
     t = rx*ry;
     for(x=0; x<t; x++) des[x] = fill;
     return;
  }
  m3 = m0+rx;   m6 = m3+rx; 
  
  csx = (double)sx/2;   csy = (double)sy/2;
  crx = (double)rx/2;   cry = (double)ry/2;
  px = -rp[0]*crx - rp[1]*cry + rp[2];
  py = -rp[3]*crx - rp[4]*cry + rp[5];
  pk = -rp[6]*crx - rp[7]*cry + rp[8];
  for(x=0; x<rx; x++) {
    m0[x] = (rp[0]*(double)x + px)*1024.0;                           
    m3[x] = (rp[3]*(double)x + py)*1024.0;
    m6[x] = rp[6]*(double)x + pk;
  }
  
  csx *= 1024.0;  csy *= 1024.0;
  x16 = (sx-1)<<10;       y16 = (sy-1)<<10;

  for(y=0; y<ry; y++) {
    m1 = rp[1]*(double)y*1024.0;                           
    m4 = rp[4]*(double)y*1024.0;
    m7 = rp[7]*(double)y;
    for(x=0; x<rx; x++, des++) {
      dp = m6[x]+m7;
      if(fabs(dp) < 1000*DBL_EPSILON) continue;  // the projection have infinite points
      xx = (int)((m0[x]+m1)/dp+csx);
      yy = (int)((m3[x]+m4)/dp+csy);
      if(xx<0 || yy<0 || xx>x16 || yy>y16) *des = fill;
      else {
        xp = (xx+512)>>10; yp = (yy+512)>>10;
        *des = *(src+xp+yp*sx); 
      }
    }
  }
  free(m0);
  return ;
}  

// Do 2D projective transformation (image center) of img to get des, des is the same size as img
// Positive tx is translate right,  Positive ty is translate down 

IBOOL  image_2D_project(ImageDes *des, ImageDes img, double pj[9], IDWORD fills, IInterpType interpolation)
{
  
  if(!img.load) return IFALSE;
  if(!__mat_3x3_is_not_degenerate(pj)) return IFALSE;
  __Img_Busy(1);
    
  if(!AllocPicture(des, img.xsize, img.ysize, img.imagetype, img.alpha, img.numColors)) {__Img_Busy(0); return IFALSE;}
  des->load = ITRUE;
  des->transparent = img.transparent;
  des->trans = img.trans;
  des->background = img.background;
  des->back = img.back;
  des->gamma = img.gamma;
  if(img.imagetype == IColor256 || img.imagetype == IIndexedColor) {
	  interpolation = IInterp_fast;
	  memcpy(des->pal, img.pal, 768);
  }
  switch (interpolation) {
    case IInterp_normal:   // or bilinear interpolation
    case IInterp_fine  :   // or bicubic interpolation
	   if(des->alpha) __proj_trans(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, pj, DWORD_COLOR_ALPHA(fills));
       __proj_trans(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, pj, DWORD_COLOR_RED(fills));
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __proj_trans(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, pj, DWORD_COLOR_GREEN(fills));
         __Img_Busy(66);
         __proj_trans(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, pj, DWORD_COLOR_BLUE(fills));
         __Img_Busy(99);
       }
       break;
    case IInterp_fast  :   // or near neighbor interpolation (palette image can only do this)
	   if(des->alpha) __proj_trans_fast(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, pj, DWORD_COLOR_ALPHA(fills));
       __proj_trans_fast(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, pj, DWORD_COLOR_RED(fills));
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __proj_trans_fast(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, pj, DWORD_COLOR_GREEN(fills));
         __Img_Busy(66);
         __proj_trans_fast(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, pj, DWORD_COLOR_BLUE(fills));
         __Img_Busy(99);
       }
       break;
  }
  __Img_Busy(0);
  return ITRUE;
}  




// Do 2D projective transformation (image center) of img to get des, des contain the full result
// Positive tx is translate right,  Positive ty is translate down 
IBOOL  image_2D_project_size(ImageDes *des, ImageDes img, double pj[9], IDWORD fills, IInterpType interpolation)
{
  int dx, dy;
  
  if(!img.load) return IFALSE;
  if(!__mat_3x3_is_not_degenerate(pj)) return IFALSE;

  __Img_Busy(1);
    
  __project_size(img.xsize, img.ysize, pj, &dx, &dy);
  if(!AllocPicture(des, dx, dy, img.imagetype, img.alpha, img.numColors)) {__Img_Busy(0); return IFALSE;}
  des->load = ITRUE;
  des->transparent = img.transparent;
  des->trans = img.trans;
  des->background = img.background;
  des->back = img.back;
  des->gamma = img.gamma;
  if(img.imagetype == IColor256 || img.imagetype == IIndexedColor) {
	  interpolation = IInterp_fast;
	  memcpy(des->pal, img.pal, 768);
  }
  switch (interpolation) {
    case IInterp_normal:   // or bilinear interpolation
    case IInterp_fine  :   // or bicubic interpolation
	   if(des->alpha) __proj_trans(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, pj, DWORD_COLOR_ALPHA(fills));
       __proj_trans(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, pj, DWORD_COLOR_RED(fills));
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __proj_trans(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, pj, DWORD_COLOR_GREEN(fills));
         __Img_Busy(66);
         __proj_trans(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, pj, DWORD_COLOR_BLUE(fills));
         __Img_Busy(99);
       }
       break;
    case IInterp_fast  :   // or near neighbor interpolation (palette image can only do this)
	   if(des->alpha) __proj_trans_fast(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, pj, DWORD_COLOR_ALPHA(fills));
       __proj_trans_fast(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, pj, DWORD_COLOR_RED(fills));
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __proj_trans_fast(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, pj, DWORD_COLOR_GREEN(fills));
         __Img_Busy(66);
         __proj_trans_fast(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, pj, DWORD_COLOR_BLUE(fills));
         __Img_Busy(99);
       }
       break;
  }
  __Img_Busy(0);
  return ITRUE;
}  


