/**************************************************************************
*  Image process tool box                                                 *
*     by Yang Yudong                                                      *
*     yangyd@yahoo.com                                                    *
*                                                                         *
***************************************************************************
*  Copyright (C) 1992-1999, Yang Yudong, All rights reserved.             *
*  This file is part of Yang Yudong's image processing software package.  *
*  If you use this software, you agree to the following:                  *
*  This program package is purely experimental, and is licensed "as is".  *
*  Permission is granted to use, modify, and distribute this program      *
*  without charge for any purpose, provided this license/ disclaimer      *
*  notice appears in the copies.  No warranty or maintenance is given,    *
*  either expressed or implied.  In no event shall the author(s) be       *
*  liable to you or a third party for any special, incidental,            *
*  consequential, or other damages, arising out of the use or inability   *
*  to use the program for any purpose (or the loss of data), even if we   *
*  have been advised of such possibilities.  Any public reference or      *
*  advertisement of this source code should refer to it as Yang Yudong's  *
*  orignal.                                                               *
**************************************************************************/
// ****************************************************************
//  Image process tool box
//     by Yang Yudong
//
// File : irottran.c
// Description:  scale and rotate then translate an image in it orignal size 
// Create Date:  1996. 10. 28
// Modification(date/where): 
//
// ****************************************************************

#include <malloc.h>
#include <stdio.h>
#include <string.h>
#include <float.h>
#include <math.h>
#include <stdlib.h>
#define _IMG_LIBBUILD_
#include "imgproc.h"
#include "routines.h"

#define D2R  (3.1415926535897932385/180.0)

static void __rot_transize(int ix, int iy, double ang, double tx, double ty, double scale, int *rx, int *ry)
{
  double x1, y1, x2, y2, x, y;
  ang *= D2R;
  x = ix/2.0;  y = iy/2.0;
  x1 = fabs(x*cos(ang) - y*sin(ang));
  y1 = fabs(x*sin(ang) + y*cos(ang));
  x2 = fabs(x*cos(ang) + y*sin(ang));
  y2 = fabs(x*sin(ang) - y*cos(ang));
  *rx = (int)ceil((2*(int)max(x1, x2)+fabs(tx))*scale);
  *ry = (int)ceil((2*(int)max(y1, y2)+fabs(ty))*scale);
}

// (des-center_des) / scl = R*(src - center_src) + T
//   R = (c -s,  s  c),   negR = (c  s,  -s c)
//   src = negR*((des-center_des)/scl - T) + center_src
//   src = negR*des/scl - negR*(center_des/scl + T) + center_src
static void __rtbuild_table(long *lut, int dx, int dy, int sx, int sy, double ang, double tx, double ty, double scale)
{ int i;
  double sa, ca, ux, uy;

  ang *= D2R;
  sa = sin(ang)*1024.0; ca = cos(ang)*1024.0;
  ux = -( ((double)dx/2.0/scale+tx)*ca + ((double)dy/2.0/scale+ty)*sa) + (double)sx/2.0*1024.0; 
  uy = -(-((double)dx/2.0/scale+tx)*sa + ((double)dy/2.0/scale+ty)*ca) + (double)sy/2.0*1024.0; 
  
  for(i=0; i<dx; i++) {
    lut[i   ] = (long)( (double)i*ca/scale + ux);
    lut[i+dx] = (long)(-(double)i*sa/scale + uy);
  }
  for(i=0; i<dy; i++) {
    lut[i+2*dx   ] = (long)((double)i*sa/scale);
    lut[i+2*dx+dy] = (long)((double)i*ca/scale);
  }
}  



// Rotate (image center) then translate and scale img to get des, des is the same size as img
// Positive ang is CW rotation,  Positive tx is translate right,  Positive ty is translate down 
IBOOL  image_rotate_trans(ImageDes *des, ImageDes img, double ang, double tx, double ty, double scale, IDWORD fills, IInterpType interpolation)
{
  long *lut;
  
  if(!img.load) return IFALSE;
  __Img_Busy(1);
  if(!AllocPicture(des, img.xsize, img.ysize, img.imagetype, img.alpha, img.numColors)) {__Img_Busy(0); return IFALSE;}
  lut = (long *)malloc(sizeof(long)*4*(img.xsize+img.ysize));
  if(lut == NULL) {
     FreePicture(des);
     __Img_Busy(0); 
     return IFALSE;
  } 
  __rtbuild_table(lut, des->xsize, des->ysize, img.xsize, img.ysize, ang, tx, ty, scale);

  des->load = ITRUE;
  des->transparent = img.transparent;
  des->trans = img.trans;
  des->background = img.background;
  des->back = img.back;
  des->gamma = img.gamma;
  if(img.imagetype == IColor256 || img.imagetype == IIndexedColor) {
	  interpolation = IInterp_fast;
	  memcpy(des->pal, img.pal, 768);
  }

  switch (interpolation) {
    case IInterp_normal:   // or bilinear interpolation
	   if(des->alpha) __lut2d(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, lut, DWORD_COLOR_ALPHA(fills));
       __lut2d(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, lut, DWORD_COLOR_RED(fills));
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __lut2d(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, lut, DWORD_COLOR_GREEN(fills));
         __Img_Busy(66);
         __lut2d(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, lut, DWORD_COLOR_BLUE(fills));
         __Img_Busy(99);
       }
       break;
    case IInterp_fast  :   // or near neighbor interpolation (palette image can only do this)
	   if(des->alpha) __lut2d_fast(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, lut, DWORD_COLOR_ALPHA(fills));
       __lut2d_fast(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, lut, DWORD_COLOR_RED(fills));
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __lut2d_fast(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, lut, DWORD_COLOR_GREEN(fills));
         __Img_Busy(66);
         __lut2d_fast(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, lut, DWORD_COLOR_BLUE(fills));
         __Img_Busy(99);
       }
       break;
    case IInterp_fine  :   // or bicubic interpolation
	   if(des->alpha) __lut2d_fine(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, lut, DWORD_COLOR_ALPHA(fills));
       __lut2d_fine(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, lut, DWORD_COLOR_RED(fills));
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __lut2d_fine(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, lut, DWORD_COLOR_GREEN(fills));
         __Img_Busy(66);
         __lut2d_fine(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, lut, DWORD_COLOR_BLUE(fills));
         __Img_Busy(99);
       }
       break;
  }
  free(lut);
  __Img_Busy(0);
  return ITRUE;
}  


// Rotate (image center) then translate and scale img to get des, des is the same size as img
// Positive ang is CW rotation,  Positive tx is translate right,  Positive ty is translate down 
IBOOL  image_rotate_trans_keep_old(ImageDes *des, ImageDes img, double ang, double tx, double ty, double scale, IInterpType interpolation)
{
  long *lut;
  
  if(!img.load) return IFALSE;
  __Img_Busy(1);
  if(!AllocPicture(des, img.xsize, img.ysize, img.imagetype, img.alpha, img.numColors)) {__Img_Busy(0); return IFALSE;}
  lut = (long *)malloc(sizeof(long)*4*(img.xsize+img.ysize));
  if(lut == NULL) {
     FreePicture(des);
     __Img_Busy(0); 
     return IFALSE;
  } 
  __rtbuild_table(lut, des->xsize, des->ysize, img.xsize, img.ysize, ang, tx, ty, scale);

  des->load = ITRUE;
  des->transparent = img.transparent;
  des->trans = img.trans;
  des->background = img.background;
  des->back = img.back;
  des->gamma = img.gamma;
  if(img.imagetype == IColor256 || img.imagetype == IIndexedColor) {
	  interpolation = IInterp_fast;
	  memcpy(des->pal, img.pal, 768);
  }

  switch (interpolation) {
    case IInterp_normal:   // or bilinear interpolation
	   if(des->alpha) __lut2d_keep_old(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, lut);
       __lut2d_keep_old(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, lut);
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __lut2d_keep_old(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, lut);
         __Img_Busy(66);
         __lut2d_keep_old(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, lut);
         __Img_Busy(99);
       }
       break;
    case IInterp_fast  :   // or near neighbor interpolation (palette image can only do this)
	   if(des->alpha) __lut2d_fast_keep_old(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, lut);
       __lut2d_fast_keep_old(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, lut);
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __lut2d_fast_keep_old(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, lut);
         __Img_Busy(66);
         __lut2d_fast_keep_old(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, lut);
         __Img_Busy(99);
       }
       break;
    case IInterp_fine  :   // or bicubic interpolation
	   if(des->alpha) __lut2d_fine_keep_old(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, lut);
       __lut2d_fine_keep_old(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, lut);
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __lut2d_fine_keep_old(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, lut);
         __Img_Busy(66);
         __lut2d_fine_keep_old(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, lut);
         __Img_Busy(99);
       }
       break;
  }
  free(lut);
  __Img_Busy(0);
  return ITRUE;
}  



// Rotate (image center) then translate and scale img to get des, des is the same size as img
// Positive ang is CW rotation,  Positive tx is translate right,  Positive ty is translate down 
IBOOL  image_rotate_trans_size(ImageDes *des, ImageDes img, double ang, double tx, double ty, double scale, IDWORD fills, IInterpType interpolation)
{
  long *lut;
  int dx, dy;
  
  if(!img.load) return IFALSE;
  __Img_Busy(1);
  __rot_transize(img.xsize, img.ysize, ang, tx, ty, scale, &dx, &dy);
  if(!AllocPicture(des, dx, dy, img.imagetype, img.alpha, img.numColors)) {__Img_Busy(0); return IFALSE;}
  lut = (long *)malloc(sizeof(long)*2*(dx+dy));
  if(lut == NULL) {
     FreePicture(des);
     __Img_Busy(0); 
     return IFALSE;
  } 
  __rtbuild_table(lut, des->xsize, des->ysize, img.xsize, img.ysize, ang, tx, ty, scale);

  des->load = ITRUE;
  des->transparent = img.transparent;
  des->trans = img.trans;
  des->background = img.background;
  des->back = img.back;
  des->gamma = img.gamma;
  if(img.imagetype == IColor256 || img.imagetype == IIndexedColor) {
	  interpolation = IInterp_fast;
	  memcpy(des->pal, img.pal, 768);
  }
  switch (interpolation) {
    case IInterp_normal:   // or bilinear interpolation
	   if(des->alpha) __lut2d(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, lut, DWORD_COLOR_ALPHA(fills));
       __lut2d(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, lut, DWORD_COLOR_RED(fills));
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __lut2d(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, lut, DWORD_COLOR_GREEN(fills));
         __Img_Busy(66);
         __lut2d(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, lut, DWORD_COLOR_BLUE(fills));
         __Img_Busy(99);
       }
       break;
    case IInterp_fast  :   // or near neighbor interpolation (palette image can only do this)
	   if(des->alpha) __lut2d_fast(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, lut, DWORD_COLOR_ALPHA(fills));
       __lut2d_fast(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, lut, DWORD_COLOR_RED(fills));
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __lut2d_fast(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, lut, DWORD_COLOR_GREEN(fills));
         __Img_Busy(66);
         __lut2d_fast(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, lut, DWORD_COLOR_BLUE(fills));
         __Img_Busy(99);
       }
       break;
    case IInterp_fine  :   // or bicubic interpolation
	   if(des->alpha) __lut2d_fine(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, lut, DWORD_COLOR_ALPHA(fills));
       __lut2d_fine(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, lut, DWORD_COLOR_RED(fills));
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __lut2d_fine(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, lut, DWORD_COLOR_GREEN(fills));
         __Img_Busy(66);
         __lut2d_fine(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, lut, DWORD_COLOR_BLUE(fills));
         __Img_Busy(99);
       }
       break;
  }
  free(lut);
  __Img_Busy(0);
  return ITRUE;
}  


