/**************************************************************************
*  Image process tool box                                                 *
*     by Yang Yudong                                                      *
*     yangyd@yahoo.com                                                    *
*                                                                         *
***************************************************************************
*  Copyright (C) 1992-1999, Yang Yudong, All rights reserved.             *
*  This file is part of Yang Yudong's image processing software package.  *
*  If you use this software, you agree to the following:                  *
*  This program package is purely experimental, and is licensed "as is".  *
*  Permission is granted to use, modify, and distribute this program      *
*  without charge for any purpose, provided this license/ disclaimer      *
*  notice appears in the copies.  No warranty or maintenance is given,    *
*  either expressed or implied.  In no event shall the author(s) be       *
*  liable to you or a third party for any special, incidental,            *
*  consequential, or other damages, arising out of the use or inability   *
*  to use the program for any purpose (or the loss of data), even if we   *
*  have been advised of such possibilities.  Any public reference or      *
*  advertisement of this source code should refer to it as Yang Yudong's  *
*  orignal.                                                               *
**************************************************************************/
//****************************************************************
//  Image process tool box
//     by Yang Yudong
//
// File : vdubfilter.cpp
// Description:  filter interface for VirtualDub
// Create Date:  2000. 10. 29
//
//****************************************************************

#include "vdfilter.h"

const char *szAppname = "PCVideo Image Processor V" _IMG_FILTER_VERSION_STR;

///////////////////////////////////////////////////////////////////////////

int imgprocInitProc(FilterActivation *fa, const FilterFunctions *ff);
void imgprocDeinitProc(FilterActivation *fa, const FilterFunctions *ff);
int imgprocRunProc(const FilterActivation *fa, const FilterFunctions *ff);
int imgprocStartProc(FilterActivation *fa, const FilterFunctions *ff);
int imgprocEndProc(FilterActivation *fa, const FilterFunctions *ff);
long imgprocParamProc(FilterActivation *fa, const FilterFunctions *ff);
int imgprocConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void imgprocStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void imgprocScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool imgprocFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);


///////////////////////////////////////////////////////////////////////////
ScriptFunctionDef imgproc_func_defs[]={
    { (ScriptFunctionPtr)imgprocScriptConfig, "Config", "0iii" },
    { NULL },
};

///////////////////////////////////////////////////////////////////////////


CScriptObject imgproc_obj={
    NULL, imgproc_func_defs
};

///////////////////////////////////////////////////////////////////////////

struct FilterDefinition filterDef_imgproc = {

    NULL, NULL, NULL,       // next, prev, module
    (char *)szAppname,             // name
	"Check the following site for future updates and source code:"	
	"\nhttp://pcvideo.tripod.com"
	"\nhttp://pcvideo.easthome.net"
	"\nhttp://personal.hb.cninfo.net/~pcvideo",                  // desc
    "Yudong Yang / yangyd@yahoo.com",               // maker
    NULL,                   // private_data
    sizeof(MyFilterData),   // inst_data_size

    imgprocInitProc,       // initProc
    imgprocDeinitProc,     // deinitProc
    imgprocRunProc,        // runProc
    imgprocParamProc,      // paramProc
    imgprocConfigProc,     // configProc
    imgprocStringProc,     // stringProc
    imgprocStartProc,      // startProc
    imgprocEndProc,        // endProc

    &imgproc_obj,          // script_obj
    imgprocFssProc,        // fssProc

};




///////////////////////////////////////////////////////////////////////////

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_imgproc;

FilterFunc filters[] = 
{
	{"Null Filter",			0,						"",							0},
	{"Adjust Saturation",	image_saturation,		"0 - 128(middle) - 256",	1, 128},
	{"Blur",				image_lowpass,			"",							0},
	{"Blur More",			image_blur,				"Radius",					1, 2},
	{"Expand Constast",		image_expandcontrast,	"Start and end levels",		2, 32, 224},
	{"Find Edge",			image_sobel,			"",							0},
	{"Gental Sharpen",		image_gentlesharpen,	"",							0},
	{"Histo Brighten",		image_histobrighten,	"",							0},
	{"Histo Linearize",		image_histolinearize,	"",							0},
	{"Majority Filter",		image_majority,			"Radius",					1, 2},
	{"Median Filter",		image_median,			"Radius",					1, 2},
	{"Mosiac Effect",		image_mosiac,			"Block size",				1, 8},
	{"Negtive Image",		image_negative,			"",							0},
	{"Posterize",			image_posterize,		"Quantization levels",		1, 8},
	{"Sharpen",				image_sharpen,			"",							0},
	{"Sketch Effect",		image_sketch,			"",							0},
	{0,						0,						0,							0}
};



///////////////////////////////////////////////////////////////////////////

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) 
{
    if (!(fd_imgproc = ff->addFilter(fm, &filterDef_imgproc, sizeof(FilterDefinition))))
        return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

    return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) 
{
    ff->removeFilter(fd_imgproc);
}

///////////////////////////////////////////////////////////////////////////

static bool g_MMXenabled;


int imgprocInitProc(FilterActivation *fa, const FilterFunctions *ff) 
{

    MyFilterData *mfd = (MyFilterData *)fa->filter_data;
    if(!mfd) return 1;
	mfd->hInst = fa->filter->module->hInstModule;
	mfd->hWin = NULL;
	mfd->img.alloc = IFALSE; 
	mfd->which = 0;
	mfd->parm1 = filters[mfd->which].defaultparm1;
	mfd->parm2 = filters[mfd->which].defaultparm2;
	return 0;
}

void imgprocDeinitProc(FilterActivation *fa, const FilterFunctions *ff)
{
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;
    if(!mfd) return;
	FreePicture(&mfd->img);
}


int imgprocStartProc(FilterActivation *fa, const FilterFunctions *ff) 
{
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	if(!mfd) return 1;

    g_MMXenabled = ff->isMMXEnabled(); 

	mfd->xsize = fa->src.w;
	mfd->ysize = fa->src.h;

	if(!AllocPicture(&mfd->img, mfd->xsize, mfd->ysize, ITrueColor, IFALSE, 256)) return 1;

    return 0;
}

int imgprocEndProc(FilterActivation *fa, const FilterFunctions *ff) 
{
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

    return 0;
}


int imgprocRunProc(const FilterActivation *fa, const FilterFunctions *ff) 
{
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	if(0 != mfd->which) // not NULL filter
	{
	    PixDim xsize, ysize;
	    Pixel32 *s_img, *d_img;
		IBOOL rtn;

	    ysize = fa->src.h;
	    xsize = fa->src.w;
		int linelength = xsize*sizeof(Pixel32)+fa->src.modulo;
	    s_img = (Pixel32 *)fa->src.data;
		d_img = (Pixel32 *)fa->dst.data;


		if(xsize != (PixDim)mfd->xsize || ysize != (PixDim)mfd->ysize) // the size changed, just reinit imgproc
		{
			if(!AllocPicture(&mfd->img, xsize, ysize, ITrueColor, IFALSE, 256)) return 1;
			mfd->xsize = xsize;
			mfd->ysize = ysize;
		}
		if(!img_packed_2_img_noalpha((IBYTE *)s_img+linelength*(ysize-1), xsize, ysize, -linelength, 2, 1, 0, sizeof(Pixel32), &mfd->img)) return 1;
		
		switch(filters[mfd->which].numparm)
		{
		case 0:
			rtn = ((proc_noparm)filters[mfd->which].func)(mfd->img);
			break;
		case 1:
			rtn = ((proc_1parm)filters[mfd->which].func)(mfd->img, mfd->parm1);
			break;
		case 2:
		default:
			rtn = ((proc_2parm)filters[mfd->which].func)(mfd->img, mfd->parm1, mfd->parm2);
			break;
		}
		if(rtn) 
		{
			if(mfd->clip)
			{
				int destline = fa->dst.w*sizeof(Pixel32)+fa->dst.modulo;
				img_r_g_b_2_packed_noalpha(fa->dst.w, fa->dst.h, 
					mfd->img.r + mfd->top * mfd->img.xsize, 
					mfd->img.g + mfd->top * mfd->img.xsize, 
					mfd->img.b + mfd->top * mfd->img.xsize, 
					mfd->xsize, 
					(IBYTE *)d_img+destline*(fa->dst.h-1), 
					-destline, 
					2, 1, 0, sizeof(Pixel32));
			}
			else
			{
				img_rgb_2_packed_noalpha(&mfd->img, (IBYTE *)d_img+linelength*(ysize-1), -linelength, 2, 1, 0, sizeof(Pixel32));
			}
		}
		return (int)!rtn;
	}
	else
	{
		int dstline = fa->dst.w*sizeof(Pixel32)+fa->dst.modulo;
		int srcline = fa->src.w*sizeof(Pixel32)+fa->src.modulo;
		int destx = fa->dst.w*sizeof(Pixel32);
		int desty = fa->dst.h;
	    char *src, *dst;

		src = (char *)fa->src.data;
		dst = (char *)fa->dst.data;

		if(mfd->clip)
		{
			src += mfd->left*sizeof(Pixel32) + mfd->bottom * srcline;
		}
		for(int i=0; i<desty; i++)
		{
			memmove(dst, src, destx);
			dst += dstline;
			src += srcline;
		}
		return 0;
	}
}


long imgprocParamProc(FilterActivation *fa, const FilterFunctions *ff) 
{
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	if(mfd->clip)
	{
		if(mfd->left + mfd->right >= fa->dst.w || mfd->top + mfd->bottom >= fa->dst.h) 
		{
			mfd->clip = false;
		}
		else
		{
			fa->dst.w -= mfd->left + mfd->right;
			fa->dst.h -= mfd->top + mfd->bottom;
		}
	}
 	fa->dst.pitch = (fa->dst.w*4 + 7) & -8;
    return 0; // FILTERPARAM_SWAP_BUFFERS;
}



int imgprocConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd) 
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	mfd->hWin = hwnd;
	return filter_config(mfd);
}

void imgprocStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) 
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	switch(filters[mfd->which].numparm)
	{
	case 0:
		sprintf(str, " [%d]: %s", mfd->which, filters[mfd->which].desc);
		break;
	case 1:
		sprintf(str, " [%d]: %s(%d)", mfd->which, filters[mfd->which].desc, mfd->parm1);
		break;
	case 2:
	default:
		sprintf(str, " [%d]: %s(%d, %d)", mfd->which, filters[mfd->which].desc, mfd->parm1, mfd->parm2);
		break;
	}
}

void imgprocScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) 
{
    FilterActivation *fa = (FilterActivation *)lpVoid;
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

    mfd->which   = argv[0].asInt();
    mfd->parm1  = argv[1].asInt();
    mfd->parm2  = argv[2].asInt();
}

bool imgprocFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) 
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
    _snprintf(buf, buflen, "Config(%d, %d, %d)", mfd->which, mfd->parm1, mfd->parm2);

    return true;
}


